/*******************************************************************************
 * Copyright (c) 2008 Actuate Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Actuate Corporation  - initial API and implementation
 *******************************************************************************/

package org.eclipse.birt.chart.reportitem;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.birt.chart.exception.ChartException;
import org.eclipse.birt.chart.log.ILogger;
import org.eclipse.birt.chart.log.Logger;
import org.eclipse.birt.chart.model.Chart;
import org.eclipse.birt.chart.model.ChartWithAxes;
import org.eclipse.birt.chart.model.attribute.IntersectionType;
import org.eclipse.birt.chart.model.attribute.Position;
import org.eclipse.birt.chart.model.attribute.TickStyle;
import org.eclipse.birt.chart.model.component.Axis;
import org.eclipse.birt.chart.reportitem.plugin.ChartReportItemPlugin;
import org.eclipse.birt.chart.util.SecurityUtil;
import org.eclipse.birt.core.data.ExpressionUtil;
import org.eclipse.birt.core.data.IColumnBinding;
import org.eclipse.birt.core.data.IDimLevel;
import org.eclipse.birt.core.exception.BirtException;
import org.eclipse.birt.data.aggregation.api.IBuildInAggregation;
import org.eclipse.birt.data.engine.api.aggregation.AggregationManager;
import org.eclipse.birt.data.engine.api.aggregation.IAggrFunction;
import org.eclipse.birt.data.engine.api.aggregation.IParameterDefn;
import org.eclipse.birt.data.engine.olap.api.query.ICubeElementFactory;
import org.eclipse.birt.report.item.crosstab.core.ICrosstabConstants;
import org.eclipse.birt.report.item.crosstab.core.IMeasureViewConstants;
import org.eclipse.birt.report.item.crosstab.core.de.AggregationCellHandle;
import org.eclipse.birt.report.item.crosstab.core.de.CrosstabCellHandle;
import org.eclipse.birt.report.item.crosstab.core.de.CrosstabReportItemHandle;
import org.eclipse.birt.report.item.crosstab.core.de.CrosstabViewHandle;
import org.eclipse.birt.report.item.crosstab.core.de.DimensionViewHandle;
import org.eclipse.birt.report.item.crosstab.core.de.LevelViewHandle;
import org.eclipse.birt.report.item.crosstab.core.de.MeasureViewHandle;
import org.eclipse.birt.report.model.api.AggregationArgumentHandle;
import org.eclipse.birt.report.model.api.ComputedColumnHandle;
import org.eclipse.birt.report.model.api.DataItemHandle;
import org.eclipse.birt.report.model.api.DesignElementHandle;
import org.eclipse.birt.report.model.api.ExtendedItemHandle;
import org.eclipse.birt.report.model.api.ReportItemHandle;
import org.eclipse.birt.report.model.api.StructureFactory;
import org.eclipse.birt.report.model.api.activity.SemanticException;
import org.eclipse.birt.report.model.api.elements.DesignChoiceConstants;
import org.eclipse.birt.report.model.api.elements.structures.ComputedColumn;
import org.eclipse.birt.report.model.api.metadata.DimensionValue;
import org.eclipse.birt.report.model.api.olap.CubeHandle;
import org.eclipse.birt.report.model.api.olap.DimensionHandle;
import org.eclipse.birt.report.model.api.olap.HierarchyHandle;
import org.eclipse.birt.report.model.api.olap.LevelHandle;
import org.eclipse.birt.report.model.api.olap.MeasureGroupHandle;
import org.eclipse.birt.report.model.api.olap.MeasureHandle;
import org.eclipse.birt.report.model.elements.interfaces.ICubeModel;

/**
 * Utility class for XTab integration
 */

public class ChartXTabUtil extends ChartReportItemUtil implements
		ChartReportItemConstants
{

	private static ICubeElementFactory cubeFactory = null;

	private static ILogger logger = Logger.getLogger( "org.eclipse.birt.chart.reportitem/trace" ); //$NON-NLS-1$

	protected final static DimensionValue DEFAULT_COLUMN_WIDTH = new DimensionValue( 80,
			DesignChoiceConstants.UNITS_PT );
	protected final static DimensionValue DEFAULT_ROW_HEIGHT = new DimensionValue( 30,
			DesignChoiceConstants.UNITS_PT );
	
	public synchronized static ICubeElementFactory getCubeElementFactory( )
			throws BirtException
	{
		if ( cubeFactory != null )
		{
			return cubeFactory;
		}

		try
		{
			Class<?> cls = Class.forName( ICubeElementFactory.CUBE_ELEMENT_FACTORY_CLASS_NAME );
			cubeFactory = (ICubeElementFactory) SecurityUtil.newClassInstance( cls );
		}
		catch ( Exception e )
		{
			throw new ChartException( ChartReportItemPlugin.ID,
					BirtException.ERROR,
					e );
		}
		return cubeFactory;
	}

	/**
	 * Returns the binding cube if the element or its container has cube binding
	 * or the reference to the cube
	 * 
	 * @param element
	 *            element handle
	 * @return the binding cube or null
	 * @since 2.3
	 */
	public static CubeHandle getBindingCube( DesignElementHandle element )
	{
		if ( element == null )
		{
			return null;
		}
		if ( element instanceof ReportItemHandle )
		{
			CubeHandle cube = ( (ReportItemHandle) element ).getCube( );
			if ( cube != null )
			{
				return cube;
			}
			else if ( ( (ReportItemHandle) element ).getDataBindingType( ) == ReportItemHandle.DATABINDING_TYPE_REPORT_ITEM_REF )
			{
				return getBindingCube( ( (ReportItemHandle) element ).getDataBindingReference( ) );
			}
			else if ( ( (ReportItemHandle) element ).getDataBindingType( ) == ReportItemHandle.DATABINDING_TYPE_DATA )
			{
				return null;
			}
		}
		if ( element.getContainer( ) != null )
		{
			return getBindingCube( element.getContainer( ) );
		}
		return null;
	}

	/**
	 * Gets all measure handles in the cube.
	 * 
	 * @param cube
	 *            cube handle
	 * @return all measure handles or empty list if no measure. The element in
	 *         list is <code>MeasureHandle</code>
	 * @since 2.3
	 */
	public static List<MeasureHandle> getAllMeasures( CubeHandle cube )
	{
		if ( cube.getContentCount( ICubeModel.MEASURE_GROUPS_PROP ) > 0 )
		{
			List<MeasureHandle> measures = new ArrayList<MeasureHandle>( );
			Iterator measureGroups = cube.getContents( ICubeModel.MEASURE_GROUPS_PROP )
					.iterator( );
			while ( measureGroups.hasNext( ) )
			{
				MeasureGroupHandle measureGroup = (MeasureGroupHandle) measureGroups.next( );
				measures.addAll( measureGroup.getContents( MeasureGroupHandle.MEASURES_PROP ) );
			}
			return measures;
		}
		return Collections.emptyList( );
	}

	/**
	 * Gets all level handles in the cube.
	 * 
	 * @param cube
	 *            cube handle
	 * @return all level handles or empty list if no level. The element in list
	 *         is <code>LevelHandle</code>
	 * @since 2.3
	 */
	public static List<LevelHandle> getAllLevels( CubeHandle cube )
	{
		if ( cube.getContentCount( ICubeModel.DIMENSIONS_PROP ) > 0 )
		{
			List<LevelHandle> levels = new ArrayList<LevelHandle>( );
			Iterator dimensions = cube.getContents( ICubeModel.DIMENSIONS_PROP )
					.iterator( );
			while ( dimensions.hasNext( ) )
			{
				DimensionHandle dimensionHandle = (DimensionHandle) dimensions.next( );
				HierarchyHandle hierarchy = (HierarchyHandle) ( dimensionHandle ).getContent( DimensionHandle.HIERARCHIES_PROP,
						0 );
				int count = hierarchy.getLevelCount( );
				for ( int i = 0; i < count; i++ )
				{
					levels.add( hierarchy.getLevel( i ) );
				}
			}
			return levels;
		}
		return Collections.emptyList( );
	}

	/**
	 * Returns dimension number of cube.
	 * 
	 * @param cube
	 * @return
	 * @since 2.3
	 */
	public static int getDimensionCount( CubeHandle cube )
	{
		if ( cube == null )
		{
			return 0;
		}

		return cube.getContentCount( ICubeModel.DIMENSIONS_PROP );
	}

	/**
	 * Gets all innermost level expressions from cross tab.
	 * 
	 * @param xtab
	 *            cross tab
	 * @return Levels list, each element is String
	 * @since 2.3
	 */
	public static List<String> getAllLevelsBindingExpression(
			CrosstabReportItemHandle xtab )
	{
		List<String> levels = new ArrayList<String>( );

		// Add column levels
		Object content = getFirstContent( getInnermostLevelCell( xtab,
				ICrosstabConstants.COLUMN_AXIS_TYPE ) );
		if ( content instanceof DataItemHandle )
		{
			DataItemHandle dataItemHandle = (DataItemHandle) content;
			levels.add( ExpressionUtil.createJSDataExpression( dataItemHandle.getResultSetColumn( ) ) );
		}

		// Add row levels
		content = getFirstContent( getInnermostLevelCell( xtab,
				ICrosstabConstants.ROW_AXIS_TYPE ) );
		if ( content instanceof DataItemHandle )
		{
			DataItemHandle dataItemHandle = (DataItemHandle) content;
			levels.add( ExpressionUtil.createJSDataExpression( dataItemHandle.getResultSetColumn( ) ) );
		}
		return levels;
	}

	public static List<String> getAllLevelsBindingExpression(
			Iterator columnBindings )
	{
		List<String> bindings = new ArrayList<String>( );
		while ( columnBindings.hasNext( ) )
		{
			ComputedColumnHandle cc = (ComputedColumnHandle) columnBindings.next( );
			if ( isDimensionExpresion( cc.getExpression( ) ) )
			{
				bindings.add( ExpressionUtil.createJSDataExpression( cc.getName( ) ) );
			}
		}
		return bindings;
	}

	public static List<String> getAllMeasuresBindingExpression(
			Iterator columnBindings )
	{
		List<String> bindings = new ArrayList<String>( );
		while ( columnBindings.hasNext( ) )
		{
			ComputedColumnHandle cc = (ComputedColumnHandle) columnBindings.next( );
			if ( isMeasureExpresion( getAggregationExpression( cc ) ) )
			{
				bindings.add( ExpressionUtil.createJSDataExpression( cc.getName( ) ) );
			}
			else if ( IBuildInAggregation.TOTAL_COUNT_FUNC.equals( cc.getAggregateFunction( ) )
					|| IBuildInAggregation.TOTAL_COUNTDISTINCT_FUNC.equals( cc.getAggregateFunction( ) ) )
			{
				// Add count aggregation to measure list
				bindings.add( ExpressionUtil.createJSDataExpression( cc.getName( ) ) );
			}
		}
		return bindings;
	}

	/**
	 * This method is used for back-forward compatible. For the new DTE api
	 * store the original ComputedColumnHandle expression as a parameter value
	 * now, we should retrieve expression value from ComputedColumnHandle's
	 * expression(old) or argument value.
	 * 
	 * @param bindingColumn
	 * @return
	 */
	private static String getAggregationExpression(
			ComputedColumnHandle bindingColumn )
	{
		if ( bindingColumn.getExpression( ) != null )
		{
			return bindingColumn.getExpression( );
		}
		String functionName = bindingColumn.getAggregateFunction( );
		try
		{
			IAggrFunction function = AggregationManager.getInstance( )
					.getAggregation( functionName );
			for ( IParameterDefn param : function.getParameterDefn( ) )
			{
				if ( param.isDataField( ) )
				{
					for ( Iterator iterator = bindingColumn.argumentsIterator( ); iterator.hasNext( ); )
					{
						AggregationArgumentHandle arg = (AggregationArgumentHandle) iterator.next( );
						if ( arg.getName( ).equals( param.getName( ) ) )
						{
							return arg.getValue( );
						}
					}
				}
			}
		}
		catch ( BirtException e )
		{
			logger.log( e );
		}
		return null;
	}

	public static Object getFirstContent( CrosstabCellHandle cell )
	{
		if ( cell != null )
		{
			List contents = cell.getContents( );
			if ( contents != null && contents.size( ) >= 1 )
			{
				return contents.get( 0 );
			}
		}
		return null;
	}

	/**
	 * Gets the level by index of all levels
	 * 
	 * @param xtab
	 * @param axisType
	 * @param levelIndex
	 *            index of all levels in xtab
	 * @return level
	 */
	public static LevelViewHandle getLevel( CrosstabReportItemHandle xtab,
			int axisType, int levelIndex )
	{
		if ( xtab == null )
		{
			return null;
		}
		CrosstabViewHandle xtabView = xtab.getCrosstabView( axisType );
		if ( xtabView == null )
		{
			return null;
		}
		int countAll = 0;
		int dimensionCount = xtabView.getDimensionCount( );
		for ( int i = 0; i < dimensionCount; i++ )
		{
			DimensionViewHandle dim = xtabView.getDimension( i );
			if ( dim != null )
			{
				if ( levelIndex >= countAll
						&& levelIndex < countAll + dim.getLevelCount( ) )
				{
					return dim.getLevel( levelIndex - countAll );
				}
				countAll += dim.getLevelCount( );
			}
		}
		return null;
	}

	public static CrosstabCellHandle getInnermostLevelCell(
			CrosstabReportItemHandle xtab, int axisType )
	{
		int levelCount = getLevelCount( xtab, axisType );
		LevelViewHandle levelView = getLevel( xtab, axisType, levelCount - 1 );
		if ( levelView != null )
		{
			return levelView.getCell( );
		}
		return null;
	}

	/**
	 * Gets the count of all levels in the xtab
	 * 
	 * @param xtab
	 * @param axisType
	 * @return level count of all levels of all dimensions in xtab
	 */
	public static int getLevelCount( CrosstabReportItemHandle xtab, int axisType )
	{
		if ( xtab == null )
		{
			return 0;
		}
		CrosstabViewHandle xtabView = xtab.getCrosstabView( axisType );
		if ( xtabView == null )
		{
			return 0;
		}
		int countAll = 0;
		int dimensionCount = xtabView.getDimensionCount( );
		for ( int i = 0; i < dimensionCount; i++ )
		{
			DimensionViewHandle dim = xtabView.getDimension( i );
			if ( dim != null )
			{
				countAll += dim.getLevelCount( );
			}
		}
		return countAll;
	}

	/**
	 * Gets the cell in cross tab which contains the chart
	 * 
	 * @param chartHandle
	 *            the handle with chart
	 * @return the cell which contains the chart or null
	 * @throws BirtException
	 * @since 2.3
	 */
	public static AggregationCellHandle getXtabContainerCell(
			DesignElementHandle chartHandle ) throws BirtException
	{
		DesignElementHandle container = chartHandle.getContainer( );
		if ( container instanceof ExtendedItemHandle )
		{
			ExtendedItemHandle xtabHandle = (ExtendedItemHandle) container;
			String exName = xtabHandle.getExtensionName( );
			if ( ICrosstabConstants.AGGREGATION_CELL_EXTENSION_NAME.equals( exName ) )
			{
				return (AggregationCellHandle) xtabHandle.getReportItem( );
			}
		}
		return null;
	}

	/**
	 * Creates the dimension expression according to level
	 * 
	 * @param level
	 *            level handle
	 * @return the dimension expression or null
	 * @since 2.3
	 */
	public static String createDimensionExpression( LevelHandle level )
	{
		if ( level == null )
		{
			return null;
		}
		return ExpressionUtil.createJSDimensionExpression( level.getContainer( )
				.getContainer( )
				.getName( ), level.getName( ) );
	}

	/**
	 * Creates the measure expression according to measure
	 * 
	 * @param measure
	 *            measure handle
	 * @return the measure expression or null
	 * @since 2.3
	 */
	public static String createMeasureExpression( MeasureHandle measure )
	{
		if ( measure == null )
		{
			return null;
		}
		return ExpressionUtil.createJSMeasureExpression( measure.getName( ) );
	}

	public static String createLevelBindingName( LevelHandle level )
	{
		if ( level == null )
		{
			return null;
		}
		return level.getContainer( ).getContainer( ).getName( )
				+ EXPRESSION_SPLITTOR
				+ level.getName( );
	}

	public static String createMeasureBindingName( MeasureHandle measure )
	{
		if ( measure == null )
		{
			return null;
		}
		return measure.getContainer( ).getName( )
				+ EXPRESSION_SPLITTOR
				+ measure.getName( );
	}

	public static ComputedColumnHandle findBinding( ReportItemHandle handle,
			String expression )
	{
		if ( expression != null )
		{
			for ( Iterator bindings = getAllColumnBindingsIterator( handle ); bindings.hasNext( ); )
			{
				ComputedColumnHandle cc = (ComputedColumnHandle) bindings.next( );
				if ( expression.equals( cc.getExpression( ) ) )
				{
					return cc;
				}
			}
		}
		return null;
	}

	/**
	 * Checks current chart is in cross tab's measure cell.
	 * 
	 * @param chartHandle
	 *            the handle holding chart
	 * @return true means within cross tab, false means not
	 * @since 2.3
	 */
	public static boolean isInXTabMeasureCell( DesignElementHandle chartHandle )
	{
		DesignElementHandle container = chartHandle.getContainer( );
		if ( container instanceof ExtendedItemHandle )
		{
			String exName = ( (ExtendedItemHandle) container ).getExtensionName( );
			if ( ICrosstabConstants.AGGREGATION_CELL_EXTENSION_NAME.equals( exName ) )
			{
				return true;
			}
		}
		return false;
	}

	public static boolean isPlotChart( DesignElementHandle chartHandle )
	{
		if ( ChartReportItemUtil.isChartHandle( chartHandle ) )
		{
			return TYPE_PLOT_CHART.equals( chartHandle.getProperty( PROPERTY_CHART_TYPE ) );
		}
		return false;
	}

	public static boolean isAxisChart( DesignElementHandle chartHandle )
	{
		if ( ChartReportItemUtil.isChartHandle( chartHandle ) )
		{
			return TYPE_AXIS_CHART.equals( chartHandle.getProperty( PROPERTY_CHART_TYPE ) );
		}
		return false;
	}

	/**
	 * Updates runtime model to render chart plot only.
	 * 
	 * @param cm
	 *            chart model
	 * @param bRtL
	 *            indicates if in right-to-left context
	 * @return the modified chart model
	 * @since 2.3
	 */
	public static Chart updateModelToRenderPlot( Chart cm, boolean bRtL )
	{
		if ( cm instanceof ChartWithAxes )
		{
			ChartWithAxes chart = (ChartWithAxes) cm;
			boolean bTransposed = chart.isTransposed( );

			chart.getLegend( ).setVisible( false );
			chart.getTitle( ).setVisible( false );
			chart.getBlock( ).getInsets( ).set( 0, 0, 0, 0 );
			chart.setReverseCategory( bTransposed || bRtL );
			// To set visible back in case of invisible in axis chart
			chart.getPlot( ).getOutline( ).setVisible( false );
			chart.getPlot( ).getClientArea( ).setVisible( true );
			chart.getPlot( ).getInsets( ).set( 0, 0, 0, 0 );
			chart.getPlot( ).getClientArea( ).getInsets( ).set( 0, 0, 0, 0 );
			chart.getPlot( ).setVerticalSpacing( 0 );
			chart.getPlot( ).setHorizontalSpacing( 0 );

			Axis xAxis = chart.getAxes( ).get( 0 );
			Axis yAxis = xAxis.getAssociatedAxes( ).get( 0 );

			xAxis.getTitle( ).setVisible( false );
			xAxis.getLabel( ).setVisible( false );
			xAxis.getLineAttributes( ).setVisible( false );
			xAxis.getMajorGrid( ).getTickAttributes( ).setVisible( false );
			xAxis.getMinorGrid( ).getTickAttributes( ).setVisible( false );
			xAxis.setCategoryAxis( true );

			yAxis.getTitle( ).setVisible( false );
			yAxis.getLabel( ).setVisible( false );
			yAxis.getLineAttributes( ).setVisible( false );
			yAxis.getMajorGrid( ).getTickAttributes( ).setVisible( false );
			yAxis.getMinorGrid( ).getTickAttributes( ).setVisible( false );
			yAxis.setLabelWithinAxes( true );
		}
		return cm;
	}

	/**
	 * Updates runtime model to render chart axis only.
	 * 
	 * @param cm
	 *            chart model
	 * @param bRtL
	 *            indicates if in right-to-left context
	 * @return the modified chart model
	 * @since 2.3
	 */
	public static Chart updateModelToRenderAxis( Chart cm, boolean bRtL )
	{
		if ( cm instanceof ChartWithAxes )
		{
			ChartWithAxes chart = (ChartWithAxes) cm;
			chart.getLegend( ).setVisible( false );
			chart.getTitle( ).setVisible( false );
			chart.getPlot( ).getOutline( ).setVisible( false );
			chart.getPlot( ).getClientArea( ).setVisible( false );
			chart.getBlock( ).getInsets( ).set( 0, 0, 0, 0 );
			chart.getPlot( ).getInsets( ).set( 0, 0, 0, 0 );
			chart.getPlot( ).getClientArea( ).getInsets( ).set( 0, 0, 0, 0 );
			chart.getPlot( ).setVerticalSpacing( 0 );
			chart.getPlot( ).setHorizontalSpacing( 0 );

			boolean bTransposed = chart.isTransposed( );
			Axis xAxis = chart.getAxes( ).get( 0 );
			Axis yAxis = xAxis.getAssociatedAxes( ).get( 0 );

			xAxis.getTitle( ).setVisible( false );
			xAxis.getLabel( ).setVisible( false );
			xAxis.getLineAttributes( ).setVisible( false );
			xAxis.getMajorGrid( ).getTickAttributes( ).setVisible( false );
			xAxis.getMajorGrid( ).getLineAttributes( ).setVisible( false );
			xAxis.getMinorGrid( ).getTickAttributes( ).setVisible( false );
			xAxis.getMinorGrid( ).getLineAttributes( ).setVisible( false );

			yAxis.getLabel( ).setVisible( true );
			yAxis.getMajorGrid( ).getTickAttributes( ).setVisible( true );
			yAxis.getTitle( ).setVisible( false );
			yAxis.getLineAttributes( ).setVisible( false );
			yAxis.getMajorGrid( ).getLineAttributes( ).setVisible( false );
			yAxis.getMinorGrid( ).getLineAttributes( ).setVisible( false );
			yAxis.getMajorGrid( )
					.setTickStyle( bTransposed || bRtL ? TickStyle.LEFT_LITERAL
							: TickStyle.RIGHT_LITERAL );
			yAxis.setLabelPosition( bTransposed || bRtL ? Position.LEFT_LITERAL
					: Position.RIGHT_LITERAL );
			yAxis.setLabelWithinAxes( true );
			if ( bTransposed )
			{
				// Show axis in the top in vertical direction
				yAxis.getOrigin( ).setType( IntersectionType.MAX_LITERAL );
			}
		}
		else
		{
			cm = null;
		}
		return cm;
	}

	/**
	 * Check if specified expression is a dimension expression.
	 * 
	 * @param expression
	 * @since 2.3
	 */
	public static boolean isDimensionExpresion( String expression )
	{
		if ( expression != null
				&& expression.matches( "\\Qdimension[\"\\E.*\\Q\"][\"\\E.*\\Q\"]\\E" ) ) //$NON-NLS-1$
		{
			return true;
		}
		return false;
	}

	/**
	 * This method is used to get the level name that reference by a level
	 * reference expression of following format:
	 * dimension["dimensionName"]["levelName"].
	 * 
	 * String[0] dimensionName; String[1] levelName;
	 * 
	 * @param expr
	 * @return String[]
	 * @since 2.3
	 */
	public static String[] getLevelNameFromDimensionExpression( String expr )
	{
		if ( isDimensionExpresion( expr ) )
		{
			try
			{
				Set<IDimLevel> levels = ExpressionUtil.getReferencedDimLevel( expr );
				if ( !levels.isEmpty( ) )
				{
					IDimLevel level = levels.iterator( ).next( );
					return new String[]{
							level.getDimensionName( ), level.getLevelName( )
					};
				}
			}
			catch ( BirtException e )
			{
				logger.log( e );
			}
		}
		return null;
	}

	/**
	 * Check if specified expression is a measure expression.
	 * 
	 * @param expression
	 * @since 2.3
	 */
	public static boolean isMeasureExpresion( String expression )
	{
		if ( expression != null
				&& expression.matches( "\\Qmeasure[\"\\E.*\\Q\"]\\E" ) ) //$NON-NLS-1$
		{
			return true;
		}
		return false;
	}

	/**
	 * This method is to get the measure name that referenced by a measure
	 * reference expression.
	 * 
	 * @param expr
	 * @return measure name
	 * @since 2.3
	 */
	public static String getMeasureName( String expr )
	{
		if ( isMeasureExpresion( expr ) )
		{
			try
			{
				return ExpressionUtil.getReferencedMeasure( expr );
			}
			catch ( BirtException e )
			{
				logger.log( e );
			}
		}
		return null;
	}

	/**
	 * Return the binding name of data["binding"]
	 * 
	 * @param expr
	 *            expression
	 * @param hasOperation
	 *            indicates if operation can be allowed in expression
	 * @throws BirtException 
	 */
	public static String getBindingName( String expr, boolean hasOperation )
	{
		if ( isBinding( expr, hasOperation ) )
		{
			try
			{
				List<IColumnBinding> bindings = ExpressionUtil.extractColumnExpressions( expr,
						ExpressionUtil.DATA_INDICATOR );
				if ( !bindings.isEmpty( ) )
				{
					return bindings.get( 0 ).getResultSetColumnName( );
				}
			}
			catch ( BirtException e )
			{
				logger.log( e );
			}
		}
		return null;
//		if ( hasOperation )
//		{
//			return expr.replaceFirst( ".*\\Qdata[\"\\E", "" ) //$NON-NLS-1$ //$NON-NLS-2$
//					.replaceFirst( "\\Q\"]\\E.*", "" ); //$NON-NLS-1$ //$NON-NLS-2$
//		}
//		return expr.replaceFirst( "\\Qdata[\"\\E", "" ) //$NON-NLS-1$ //$NON-NLS-2$
//				.replaceFirst( "\\Q\"]\\E", "" ); //$NON-NLS-1$ //$NON-NLS-2$
	}

	/**
	 * Gets the binding name list in the expression like data["year"]+"
	 * Q"+data["quarter"]
	 * 
	 * @param expr
	 * @return binding name list or empty list
	 */
	public static List<String> getBindingNameList( String expr )
	{
		List<String> names = new ArrayList<String>( );
		try
		{
			List<IColumnBinding> bindings = ExpressionUtil.extractColumnExpressions( expr,
					ExpressionUtil.DATA_INDICATOR );
			for ( IColumnBinding binding : bindings )
			{
				names.add( binding.getResultSetColumnName( ) );
			}
		}
		catch ( BirtException e )
		{
			logger.log( e );
		}

		return names;
	}

	/**
	 * Checks if the expression references a binding name
	 * 
	 * @param expr
	 *            expression
	 * @param hasOperation
	 *            indicates if operation can be allowed in expression
	 */
	public static boolean isBinding( String expr, boolean hasOperation )
	{
		if ( expr == null )
		{
			return false;
		}
		String regExp = hasOperation ? ".*\\Qdata[\"\\E.*\\Q\"]\\E.*" //$NON-NLS-1$
				: "\\Qdata[\"\\E.*\\Q\"]\\E"; //$NON-NLS-1$
		return expr.matches( regExp );
	}

	/**
	 * Generates the name of binding which references to xtab's measure.
	 * 
	 * @param cell
	 *            measure cell or total cell
	 */
	public static String generateComputedColumnName( AggregationCellHandle cell )
	{
		MeasureViewHandle measureView = (MeasureViewHandle) cell.getContainer( );
		LevelHandle rowLevelHandle = cell.getAggregationOnRow( );
		LevelHandle colLevelHandle = cell.getAggregationOnColumn( );
		String aggregationOnRow = rowLevelHandle == null ? null
				: rowLevelHandle.getFullName( );
		String aggregationOnColumn = colLevelHandle == null ? null
				: colLevelHandle.getFullName( );

		String name = ""; //$NON-NLS-1$
		String temp = measureView.getCubeMeasureName( );
		if ( temp != null && temp.length( ) > 0 )
			name = name + temp;

		if ( aggregationOnRow != null && aggregationOnRow.length( ) > 0 )
		{
			if ( name.length( ) > 0 )
			{
				name = name + "_" + aggregationOnRow; //$NON-NLS-1$
			}
			else
			{
				name = name + aggregationOnRow;
			}
		}
		if ( aggregationOnColumn != null && aggregationOnColumn.length( ) > 0 )
		{
			if ( name.length( ) > 0 )
			{
				name = name + "_" + aggregationOnColumn; //$NON-NLS-1$
			}
			else
			{
				name = name + aggregationOnColumn;
			}
		}
		if ( name.length( ) <= 0 )
		{
			name = "measure"; //$NON-NLS-1$
		}

		ComputedColumn column = StructureFactory.newComputedColumn( cell.getCrosstabHandle( ),
				name );
		String dataType = measureView.getDataType( );
		column.setDataType( dataType );
		column.setExpression( ExpressionUtil.createJSMeasureExpression( measureView.getCubeMeasureName( ) ) );
		column.setAggregateFunction( getDefaultMeasureAggregationFunction( measureView ) );
		if ( aggregationOnRow != null )
		{
			column.addAggregateOn( aggregationOnRow );
		}
		if ( aggregationOnColumn != null )
		{
			column.addAggregateOn( aggregationOnColumn );
		}

		// add the computed column to crosstab
		try
		{
			ComputedColumnHandle columnHandle = ( (ReportItemHandle) cell.getCrosstabHandle( ) ).addColumnBinding( column,
					false );
			return columnHandle.getName( );
		}
		catch ( SemanticException e )
		{
			logger.log( e );
		}

		return name;
	}

	/**
	 * Returns the default aggregation function for specific measure view
	 */
	static String getDefaultMeasureAggregationFunction( MeasureViewHandle mv )
	{
		if ( mv != null && mv.getCubeMeasure( ) != null )
		{
			String func = mv.getCubeMeasure( ).getFunction( );

			if ( func != null )
			{
				return getRollUpAggregationFunction( func );
			}
		}

		return DesignChoiceConstants.MEASURE_FUNCTION_SUM;
	}

	/**
	 * TODO this method should provide by DTE?
	 */
	static String getRollUpAggregationFunction( String functionName )
	{
		if ( DesignChoiceConstants.AGGREGATION_FUNCTION_AVERAGE.equals( functionName )
				|| DesignChoiceConstants.AGGREGATION_FUNCTION_COUNT.equals( functionName )
				|| DesignChoiceConstants.AGGREGATION_FUNCTION_COUNTDISTINCT.equals( functionName ) )
		{
			return DesignChoiceConstants.AGGREGATION_FUNCTION_SUM;
		}
		else
		{
			return functionName;
		}
	}

	/**
	 * Returns bindings names whose dimension expressions equal with specified
	 * expression.
	 * 
	 * @param dimExpr
	 * @param values
	 * @return binding name list
	 */
	public static List<String> getRelatedBindingNames( String dimExpr,
			Collection values )
	{
		List<String> bindingNames = new ArrayList<String>( 1 );
		for ( Iterator iter = values.iterator( ); iter.hasNext( ); )
		{
			ComputedColumnHandle cch = (ComputedColumnHandle) iter.next( );
			if ( dimExpr.equals( cch.getExpression( ) ) )
			{
				bindingNames.add( cch.getName( ) );
			}
		}
		return bindingNames;
	}

	/**
	 * Finds the reference chart from plot chart or axis chart.
	 * 
	 * @param chartHandle
	 * @return reference chart handle or null
	 */
	public static ExtendedItemHandle findReferenceChart(
			ExtendedItemHandle chartHandle )
	{
		if ( isAxisChart( chartHandle ) )
		{
			return (ExtendedItemHandle) chartHandle.getElementProperty( PROPERTY_HOST_CHART );
		}
		else if ( isPlotChart( chartHandle ) )
		{
			for ( Iterator iterator = chartHandle.clientsIterator( ); iterator.hasNext( ); )
			{
				DesignElementHandle client = (DesignElementHandle) iterator.next( );
				if ( isAxisChart( client ) )
				{
					return (ExtendedItemHandle) client;
				}
			}
		}
		return null;
	}

	public static boolean isDetailCell( AggregationCellHandle cell )
	{
		return IMeasureViewConstants.DETAIL_PROP.equals( cell.getModelHandle( )
				.getContainerPropertyHandle( )
				.getPropertyDefn( )
				.getName( ) );
	}

	public static boolean isAggregationCell( AggregationCellHandle cell )
	{
		return IMeasureViewConstants.AGGREGATIONS_PROP.equals( cell.getModelHandle( )
				.getContainerPropertyHandle( )
				.getPropertyDefn( )
				.getName( ) );
	}

	public static int getXTabAxisType( boolean bTransposed )
	{
		return bTransposed ? ICrosstabConstants.ROW_AXIS_TYPE
				: ICrosstabConstants.COLUMN_AXIS_TYPE;
	}
}
