/*******************************************************************************
 * Copyright (c) 2004, 2005, 2006, 2007, 2008, 2009 Actuate Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Actuate Corporation  - initial API and implementation
 *******************************************************************************/

package org.eclipse.birt.report.model.adapter.oda.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import org.eclipse.birt.report.model.api.DynamicFilterParameterHandle;
import org.eclipse.birt.report.model.api.FilterConditionHandle;
import org.eclipse.birt.report.model.api.OdaDataSetHandle;
import org.eclipse.birt.report.model.api.ParameterHandle;
import org.eclipse.birt.report.model.api.StructureFactory;
import org.eclipse.birt.report.model.api.activity.SemanticException;
import org.eclipse.birt.report.model.api.elements.structures.FilterCondition;
import org.eclipse.birt.report.model.api.util.StringUtil;
import org.eclipse.birt.report.model.elements.interfaces.IDataSetModel;
import org.eclipse.datatools.connectivity.oda.design.CompositeFilterExpression;
import org.eclipse.datatools.connectivity.oda.design.CustomFilterExpression;
import org.eclipse.datatools.connectivity.oda.design.DataSetDesign;
import org.eclipse.datatools.connectivity.oda.design.DesignFactory;
import org.eclipse.datatools.connectivity.oda.design.DynamicFilterExpression;
import org.eclipse.datatools.connectivity.oda.design.ExpressionArguments;
import org.eclipse.datatools.connectivity.oda.design.ExpressionParameterDefinition;
import org.eclipse.datatools.connectivity.oda.design.ExpressionVariable;
import org.eclipse.datatools.connectivity.oda.design.FilterExpression;
import org.eclipse.datatools.connectivity.oda.design.ParameterDefinition;
import org.eclipse.datatools.connectivity.oda.design.ResultSetCriteria;
import org.eclipse.datatools.connectivity.oda.design.ResultSetDefinition;

/**
 * The utility class that converts between ROM filter conditions and ODA filter
 * expression
 * 
 * @see FilterConditionHandle
 * @see FilterExpression
 */
public class FilterAdapter
{

	/**
	 * The data set handle.
	 */

	private final OdaDataSetHandle setHandle;

	/**
	 * The data set design.
	 */

	private final DataSetDesign setDesign;

	/**
	 * Parameter convert utility
	 */
	private final AbstractReportParameterAdapter paramAdapter = new AbstractReportParameterAdapter( );

	/**
	 * Constant to seperate date set name and column name.
	 */
	private final String SEPERATOR = ":";

	/**
	 * Prefix constant for custom expression.
	 */
	private final String CUSTOM_PREFIX = "#";

	/**
	 * Prefix constant for dynamic expression.
	 */
	private final String DYNAMIC_PREFIX = "!";

	/**
	 * The constructor.
	 * 
	 * @param setHandle
	 *            the data set handle
	 * @param setDesign
	 *            the data set design
	 * 
	 */
	public FilterAdapter( OdaDataSetHandle setHandle, DataSetDesign setDesign )
	{
		this.setHandle = setHandle;
		this.setDesign = setDesign;
	}

	/**
	 * Updates rom filter condition by ODA filter expression
	 */

	public void updateROMFilterCondition( ) throws SemanticException
	{
		FilterExpression filterExpression = null;

		ResultSetDefinition resultSet = setDesign.getPrimaryResultSet( );
		if ( resultSet != null )
		{
			ResultSetCriteria criteria = resultSet.getCriteria( );
			if ( criteria != null )
			{
				filterExpression = criteria.getFilterSpecification( );
			}
		}
		Map<String, FilterExpression> filterExprMap = buildFilterExpressionMap( filterExpression );

		// clears up old filter conditions and finds parameters to refresh
		cleanUpROMFilterCondition( filterExprMap );

		// update exists filter conditions
		updateExistingROMFilterConditions( filterExprMap );
		// sets new filter conditions
		createROMFilterConditions( filterExprMap );

		filterExprMap.clear( );
		filterExprMap = null;
	}

	/**
	 * Builds the filter expression map to convert
	 * 
	 * @param filterExpr
	 *            the filter expression
	 * @return the map containing filter expression to convert
	 */
	private Map<String, FilterExpression> buildFilterExpressionMap(
			FilterExpression filterExpr )
	{
		HashMap<String, FilterExpression> filterExpressions = new LinkedHashMap<String, FilterExpression>( );
		if ( filterExpr != null )
		{
			if ( filterExpr instanceof CompositeFilterExpression )
			{
				CompositeFilterExpression compositeFilterExp = (CompositeFilterExpression) filterExpr;
				for ( FilterExpression child : compositeFilterExp.getChildren( ) )
				{
					filterExpressions.putAll( buildFilterExpressionMap( child ) );
				}
			}
			else
			{
				String key = getMapKey( filterExpr );
				if ( key != null )
				{
					if ( filterExpr instanceof CustomFilterExpression )
					{
						filterExpressions.put( key, filterExpr );
					}
					else if ( filterExpr instanceof DynamicFilterExpression )
					{
						DynamicFilterExpression dynamicFilterExp = (DynamicFilterExpression) filterExpr;
						ExpressionArguments arguments = dynamicFilterExp.getContextArguments( );
						if ( arguments != null
								&& arguments.getExpressionParameterDefinitions( ) != null
								&& !arguments.getExpressionParameterDefinitions( )
										.isEmpty( ) )
						{
							filterExpressions.put( key, dynamicFilterExp );
						}
					}
				}
			}
		}
		return filterExpressions;
	}

	/**
	 * Returns the map key for the given dynamic filter expression
	 * 
	 * @param filterExpr
	 *            the filter expression
	 * @return the key for the given filter expression
	 */
	private String getMapKey( FilterExpression filterExpr )
	{
		String key = null;
		String columnExpr = getColumnExpr( filterExpr );
		if ( !StringUtil.isBlank( columnExpr ) )
		{
			if ( filterExpr instanceof CustomFilterExpression )
			{
				key = CUSTOM_PREFIX + filterExpr.toString( );
			}
			else if ( filterExpr instanceof DynamicFilterExpression )
			{
				key = DYNAMIC_PREFIX
						+ setHandle.getName( )
						+ SEPERATOR
						+ columnExpr;
			}
			else
			{
				assert false;
			}
		}
		return key;
	}

	/**
	 * Returns the column expression defined in the given filter expression for
	 * filter condition.
	 * 
	 * @param filterExpr
	 *            the filter expression
	 * @return the column expression defined in the given filter expression
	 */
	private String getColumnExpr( FilterExpression filterExpr )
	{
		ExpressionVariable variable = null;
		if ( filterExpr instanceof CustomFilterExpression )
		{
			variable = ( (CustomFilterExpression) filterExpr ).getContextVariable( );
		}
		else if ( filterExpr instanceof DynamicFilterExpression )
		{
			variable = ( (DynamicFilterExpression) filterExpr ).getContextVariable( );
		}
		if ( variable != null )
		{
			return variable.getIdentifier( );
		}
		return null;
	}

	/**
	 * Returns the map key for the given filter condition
	 * 
	 * @param filterConditionHandle
	 *            the handle of the filter condition
	 * @return the key for the given filter handle,or null if the filter
	 *         condition is not dynamic.
	 */
	private String getMapKey( FilterConditionHandle filterConditionHandle )
	{
		String key = null;
		if ( !StringUtil.isBlank( filterConditionHandle.getDynamicFilterParameter( ) ) )
		{
			ParameterHandle parameterHandle = setHandle.getModuleHandle( )
					.findParameter( filterConditionHandle.getDynamicFilterParameter( ) );
			if ( parameterHandle instanceof DynamicFilterParameterHandle )
			{
				DynamicFilterParameterHandle dynamicFilterParamHandle = (DynamicFilterParameterHandle) parameterHandle;
				key = DYNAMIC_PREFIX
						+ dynamicFilterParamHandle.getDataSetName( )
						+ SEPERATOR
						+ dynamicFilterParamHandle.getColumn( );
			}
		}
		return key;
	}

	/**
	 * Updates existing filter conditions with the given filter expressions
	 * 
	 * @param filterExpMap
	 *            the map containing the given filter expressions
	 * @throws SemanticException
	 */
	private void updateExistingROMFilterConditions(
			Map<String, FilterExpression> filterExpMap )
			throws SemanticException
	{
		for ( Iterator iter = setHandle.filtersIterator( ); iter.hasNext( ); )
		{
			FilterConditionHandle filterConditionHandle = (FilterConditionHandle) iter.next( );
			String key = getMapKey( filterConditionHandle );
			if ( key != null && filterExpMap.containsKey( key ) )
			{
				DynamicFilterParameterHandle dynamicFilterParamHandle = (DynamicFilterParameterHandle) setHandle.getModuleHandle( )
						.findParameter( filterConditionHandle.getDynamicFilterParameter( ) );
				updateDynamicFilterCondition( filterConditionHandle,
						(DynamicFilterExpression) filterExpMap.get( key ),
						dynamicFilterParamHandle );
				// Removes the filter from the map after updated
				filterExpMap.remove( key );
			}
			else
			{// not expected
				assert false;
			}
		}
	}

	/**
	 * Creates new filter conditions by the given filter expressions
	 * 
	 * @param filterExpMap
	 *            the map containing filter expressions
	 * @throws SemanticException
	 */
	private void createROMFilterConditions(
			Map<String, FilterExpression> filterExpMap )
			throws SemanticException
	{
		for ( FilterExpression filterExpr : filterExpMap.values( ) )
		{
			FilterCondition filterCondition = StructureFactory.createFilterCond( );
			filterCondition.setExpr( getColumnExpr( filterExpr ) );
			FilterConditionHandle filterConditionHandle = (FilterConditionHandle) setHandle.getPropertyHandle( IDataSetModel.FILTER_PROP )
					.addItem( filterCondition );
			if ( filterExpr instanceof CustomFilterExpression )
			{
				CustomFilterExpression customFilterExp = (CustomFilterExpression) filterExpr;
				updateCustomFilterCondition( filterConditionHandle,
						customFilterExp );
			}
			else if ( filterExpr instanceof DynamicFilterExpression )
			{
				DynamicFilterExpression dynamicFilterExp = (DynamicFilterExpression) filterExpr;
				// creates new dynamic filter parameter
				DynamicFilterParameterHandle dynamicFilterParamHandle = setHandle.getModuleHandle( )
						.getElementFactory( )
						.newDynamicFilterParameter( null );
				dynamicFilterParamHandle.setDataSetName( setHandle.getName( ) );
				dynamicFilterParamHandle.setColumn( dynamicFilterExp.getContextVariable( )
						.getIdentifier( ) );
				setHandle.getModuleHandle( )
						.getParameters( )
						.add( dynamicFilterParamHandle );
				// sets the reference
				filterConditionHandle.setDynamicFilterParameter( dynamicFilterParamHandle.getName( ) );
				// updates the dynamic filter parameter
				updateDynamicFilterCondition( filterConditionHandle,
						dynamicFilterExp,
						dynamicFilterParamHandle );
			}
		}
	}

	/**
	 * Updates the filter condition by the given custom filter expression
	 * 
	 * @param filterConditionHandle
	 *            the handle of the filter condition to update
	 * @param customFilterExpr
	 *            the custom filter expression
	 * @throws SemanticException
	 */
	private void updateCustomFilterCondition(
			FilterConditionHandle filterConditionHandle,
			CustomFilterExpression customFilterExpr ) throws SemanticException
	{
		filterConditionHandle.setExtensionName( customFilterExpr.getDeclaringExtensionId( ) );
		filterConditionHandle.setExtensionExprId( customFilterExpr.getId( ) );
		filterConditionHandle.setPushDown( true );
		filterConditionHandle.setOptional( customFilterExpr.isOptional( ) );
	}

	/**
	 * Updates the filter condition by the given dynamic filter expression
	 * 
	 * @param filterConditionHandle
	 *            the handle of the filter condition to update
	 * @param dynamicFilterExpr
	 *            the dynamic filter expression
	 * @throws SemanticException
	 */
	private void updateDynamicFilterCondition(
			FilterConditionHandle filterConditionHandle,
			DynamicFilterExpression dynamicFilterExpr,
			DynamicFilterParameterHandle dynamicFilterParamHandle )
			throws SemanticException
	{
		ExpressionVariable variable = dynamicFilterExpr.getContextVariable( );
		ExpressionArguments arguments = dynamicFilterExpr.getContextArguments( );
		// Only convert the first parameter
		ExpressionParameterDefinition paramDefn = arguments.getExpressionParameterDefinitions( )
				.get( 0 );
		if ( paramDefn != null )
		{
			filterConditionHandle.setOptional( dynamicFilterExpr.isOptional( ) );
			filterConditionHandle.setExpr( variable.getIdentifier( ) );
			updateDynamicFilterParameter( dynamicFilterParamHandle, paramDefn );
		}
	}

	/**
	 * Updates the dynamic filter parameter by the given expression parameter.
	 * 
	 * @param dynamicFilterParamHandle
	 *            the handle of the dynamic filter parameter to update
	 * @param expParamDefn
	 *            the definition of the given expression parameter
	 * @throws SemanticException
	 */

	private void updateDynamicFilterParameter(
			DynamicFilterParameterHandle dynamicFilterParamHandle,
			ExpressionParameterDefinition expParamDefn )
			throws SemanticException
	{
		ParameterDefinition paramDefn = expParamDefn.getDynamicInputParameter( );
		if ( paramDefn == null )
		{
			return;
		}
		paramAdapter.updateAbstractScalarParameter( dynamicFilterParamHandle,
				paramDefn,
				null,
				setHandle );
	}

	/**
	 * Clears up all unnecessary dynamic filter parameter and filter conditions
	 * 
	 * @param filterExpMap
	 *            the map contains filter expressions
	 * @throws SemanticException
	 * 
	 */
	private void cleanUpROMFilterCondition(
			Map<String, FilterExpression> filterExpMap )
			throws SemanticException
	{
		ArrayList<FilterCondition> dropList = new ArrayList<FilterCondition>( );
		for ( Iterator iter = setHandle.filtersIterator( ); iter.hasNext( ); )
		{
			boolean needDrop = false;
			FilterConditionHandle filterHandle = (FilterConditionHandle) iter.next( );
			String dynamicParameterName = filterHandle.getDynamicFilterParameter( );
			String key = getMapKey( filterHandle );
			// Check if contains such filter.
			if ( key != null && !filterExpMap.containsKey( key ) )
			{
				// Remove the filter condition which is not contained.
				if ( !StringUtil.isBlank( dynamicParameterName ) )
				{
					ParameterHandle parameterHandle = setHandle.getModuleHandle( )
							.findParameter( dynamicParameterName );
					parameterHandle.drop( );
				}
				dropList.add( (FilterCondition) filterHandle.getStructure( ) );
			}
		}
		for ( FilterCondition fc : dropList )
		{
			setHandle.removeFilter( fc );
		}

	}

	/**
	 * Updates oda filter expression by ROM filter condition
	 */
	public void updateOdaFilterExpression( )
	{
		ResultSetDefinition resultSet = setDesign.getPrimaryResultSet( );
		if ( resultSet == null )
		{
			return;
		}
		ResultSetCriteria criteria = resultSet.getCriteria( );
		if ( criteria == null )
		{
			return;
		}
		int count = 0;
		FilterExpression filterExpr = null;
		for ( Iterator iter = setHandle.filtersIterator( ); iter.hasNext( ); )
		{
			FilterConditionHandle filterHandle = (FilterConditionHandle) iter.next( );
			FilterExpression filter = createOdaFilterExpression( filterHandle );
			if ( filter == null )
			{
				continue;
			}
			count++;
			switch ( count )
			{
				case 1 :
					filterExpr = filter;
					break;
				case 2 :
					CompositeFilterExpression compositeFilterExp = DesignFactory.eINSTANCE.createCompositeFilterExpression( );
					compositeFilterExp.add( filterExpr );
					filterExpr = compositeFilterExp;
				default :
					( (CompositeFilterExpression) filterExpr ).add( filter );
			}
		}
		criteria.setFilterSpecification( filterExpr );
	}

	/**
	 * Creates the oda filter expression by the given filter condition.
	 * 
	 * @param filterHandle
	 *            the handle of the given filter condition
	 * @return the filter expression created
	 */
	private FilterExpression createOdaFilterExpression(
			FilterConditionHandle filterHandle )
	{
		FilterExpression filterExpr = null;
		ExpressionVariable variable = DesignFactory.eINSTANCE.createExpressionVariable( );
		variable.setIdentifier( filterHandle.getExpr( ) );
		if ( !StringUtil.isBlank( filterHandle.getExtensionName( ) ) )
		{
			CustomFilterExpression customFilterExpr = DesignFactory.eINSTANCE.createCustomFilterExpression( );
			customFilterExpr.setContextVariable( variable );
			customFilterExpr.setDeclaringExtensionId( filterHandle.getExtensionName( ) );
			customFilterExpr.setId( filterHandle.getExtensionExprId( ) );
			customFilterExpr.setIsOptional( filterHandle.isOptional( ) );
			filterExpr = customFilterExpr;
		}
		else if ( !StringUtil.isBlank( filterHandle.getDynamicFilterParameter( ) ) )
		{
			ParameterHandle paramHandle = setHandle.getModuleHandle( )
					.findParameter( filterHandle.getDynamicFilterParameter( ) );
			if ( paramHandle instanceof DynamicFilterParameterHandle )
			{
				DynamicFilterParameterHandle dynamicParamHandle = (DynamicFilterParameterHandle) paramHandle;
				DynamicFilterExpression dynamicFilterExpr = DesignFactory.eINSTANCE.createDynamicFilterExpression( );
				dynamicFilterExpr.setIsOptional( filterHandle.isOptional( ) );
				dynamicFilterExpr.setContextVariable( variable );

				ExpressionArguments arguments = DesignFactory.eINSTANCE.createExpressionArguments( );
				ParameterDefinition paramDefn = DesignFactory.eINSTANCE.createParameterDefinition( );

				paramAdapter.updateParameterDefinitionFromReportParam( paramDefn,
						dynamicParamHandle,
						setDesign );

				arguments.addDynamicParameter( paramDefn );
				dynamicFilterExpr.setContextArguments( arguments );
				
				filterExpr = dynamicFilterExpr;
			}
		}
		return filterExpr;
	}

}
