/*******************************************************************************
 * Copyright (c) 2009 Actuate Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Actuate Corporation  - initial API and implementation
 *******************************************************************************/

package org.eclipse.birt.core.archive.compound;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import org.eclipse.birt.core.archive.compound.v3.Ext2Entry;
import org.eclipse.birt.core.archive.compound.v3.Ext2File;
import org.eclipse.birt.core.archive.compound.v3.Ext2FileSystem;

public class ArchiveFileV3 implements IArchiveFile
{

	public static final String PROPERTY_SYSTEM_ID = "archive.system-id";
	public static final String PROPERTY_DEPEND_ID = "archive.depened-id";

	protected Ext2FileSystem fs;
	protected HashSet<ArchiveEntryV3> openedEntries = new HashSet<ArchiveEntryV3>( );

	public ArchiveFileV3( String fileName, String mode ) throws IOException

	{
		this( fileName, null, mode );
	}

	public ArchiveFileV3( String fileName, RandomAccessFile rf, String mode )
			throws IOException

	{
		fs = new Ext2FileSystem( fileName, rf, mode );
		if ( ArchiveFile.enableSystemCache )
		{
			fs.setCacheManager( ArchiveFile.systemCacheManager );
		}
		if ( ArchiveFile.enableFileCache && fs.isRemoveOnExit( ) )
		{
			fs.setCacheSize( ArchiveFile.FILE_CACHE_SIZE * 4096 );
		}
	}

	synchronized public void close( ) throws IOException
	{
		if ( !openedEntries.isEmpty( ) )
		{
			ArrayList<ArchiveEntryV3> entries = new ArrayList<ArchiveEntryV3>(
					openedEntries );
			for ( ArchiveEntryV3 entry : entries )
			{
				entry.close( );
			}
			openedEntries.clear( );
		}
		if ( fs != null )
		{
			fs.close( );
			fs = null;
		}
	}

	public void setSystemId( String id )
	{
		fs.setProperty( PROPERTY_SYSTEM_ID, id );
	}

	public void setDependId( String id )
	{
		fs.setProperty( PROPERTY_DEPEND_ID, id );
	}

	synchronized public ArchiveEntry createEntry( String name ) throws IOException
	{
		Ext2File file = fs.createFile( name );
		return new ArchiveEntryV3( this, file );
	}

	public boolean exists( String name )
	{
		return fs.existFile( name );
	}

	synchronized public void flush( ) throws IOException
	{
		// first flush all the ext2 files
		for ( ArchiveEntryV3 entry : openedEntries )
		{
			entry.flush( );
		}
		fs.flush( );
		// then refresh all the opened files
		for ( ArchiveEntryV3 entry : openedEntries )
		{
			entry.refresh( );
		}
	}

	public String getDependId( )
	{
		return fs.getProperty( PROPERTY_DEPEND_ID );
	}

	synchronized public ArchiveEntry openEntry( String name ) throws IOException
	{
		if ( fs.existFile( name ) )
		{
			Ext2File file = fs.openFile( name );
			return new ArchiveEntryV3( this, file );
		}
		throw new FileNotFoundException( name );
	}

	public String getName( )
	{
		return fs.getFileName( );
	}

	public String getSystemId( )
	{
		return fs.getProperty( PROPERTY_SYSTEM_ID );
	}

	public long getUsedCache( )
	{
		return (long) fs.getUsedCacheSize( ) * 4096;
	}

	public List listEntries( String namePattern )
	{
		ArrayList<String> files = new ArrayList<String>( );
		for ( String file : fs.listFiles( ) )
		{
			if ( file.startsWith( namePattern ) )
			{
				files.add( file );
			}
		}
		return files;
	}

	public synchronized Object lockEntry( String name ) throws IOException
	{
		if ( !fs.existFile( name ) )
		{
			if ( !fs.isReadOnly( ) )
			{
				Ext2File file = fs.createFile( name );
				file.close( );
			}
		}
		Ext2Entry entry = fs.getEntry( name );
		if ( entry != null )
		{
			return entry;
		}
		throw new FileNotFoundException( name );
	}

	synchronized public void refresh( ) throws IOException
	{
		fs.refresh( );
		// refresh all the opened files
		for ( ArchiveEntryV3 entry : openedEntries )
		{
			entry.refresh( );
		}
	}

	public boolean removeEntry( String name ) throws IOException
	{
		fs.removeFile( name );
		return true;
	}

	public void save( ) throws IOException
	{
		fs.setRemoveOnExit( false );
		fs.flush( );
	}

	public void setCacheSize( long cacheSize )
	{
		long cacheBlock = cacheSize / 4096;
		if ( cacheBlock > Integer.MAX_VALUE )
		{
			fs.setCacheSize( Integer.MAX_VALUE );
		}
		else
		{
			fs.setCacheSize( (int) cacheBlock );
		}
	}

	synchronized public void unlockEntry( Object locker ) throws IOException
	{
		assert ( locker instanceof Ext2Entry );
	}

	protected void openEntry( ArchiveEntryV3 entry )
	{
		openedEntries.add( entry );
	}

	protected void closeEntry( ArchiveEntryV3 entry )
	{
		openedEntries.remove( entry );
	}

	private static class ArchiveEntryV3 extends ArchiveEntry
	{

		ArchiveFileV3 archive;
		Ext2File file;

		ArchiveEntryV3( ArchiveFileV3 archive, Ext2File file )
		{
			this.archive = archive;
			this.file = file;
			this.archive.openEntry( this );
		}

		public String getName( )
		{
			return file.getName( );
		}

		protected long _getLength( ) throws IOException
		{
			return file.length( );
		}

		public void close( ) throws IOException
		{
			archive.closeEntry( this );
			file.close( );
		}

		@Override
		protected void _flush( ) throws IOException
		{
		}

		@Override
		public int read( long pos, byte[] b, int off, int len )
				throws IOException
		{
			file.seek( pos );
			return file.read( b, off, len );
		}

		@Override
		protected void _refresh( ) throws IOException
		{
		}

		@Override
		protected void _setLength( long length ) throws IOException
		{
			file.setLength( length );
		}

		@Override
		public void write( long pos, byte[] b, int off, int len )
				throws IOException
		{
			file.seek( pos );
			file.write( b, off, len );
		}
	}
}
