/*****************************************************************************
 * Copyright (c) 2006-2007, Cloudsmith Inc.
 * The code, documentation and other materials contained herein have been
 * licensed under the Eclipse Public License - v 1.0 by the copyright holder
 * listed above, as the Initial Contributor under such license. The text of
 * such license is available at www.eclipse.org.
 *****************************************************************************/
package org.eclipse.buckminster.core.metadata;

import java.io.File;

import org.eclipse.buckminster.core.CorePlugin;
import org.eclipse.buckminster.core.cspec.model.CSpec;
import org.eclipse.buckminster.core.metadata.model.Materialization;
import org.eclipse.buckminster.core.metadata.model.Resolution;
import org.eclipse.buckminster.core.metadata.model.WorkspaceBinding;
import org.eclipse.buckminster.core.parser.IParserFactory;
import org.eclipse.buckminster.core.rmap.model.Provider;
import org.eclipse.buckminster.opml.model.OPML;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;

/**
 * @author Thomas Hallgren
 */
public class StorageManager
{
	private static StorageManager s_defaultManager;

	public static synchronized StorageManager getDefault() throws CoreException
	{
		if(s_defaultManager == null)
		{
			s_defaultManager = new StorageManager(CorePlugin.getDefault().getStateLocation().toFile());
			s_defaultManager.initialize();
		}
		return s_defaultManager;
	}

	private final ISaxableStorage<CSpec> m_cspecs;

	private final ISaxableStorage<WorkspaceBinding> m_wsBindings;

	private final ISaxableStorage<Resolution> m_resolutions;

	private final ISaxableStorage<Provider> m_providers;

	private final ISaxableStorage<Materialization> m_materializations;

	private final ISaxableStorage<OPML> m_opmls;

	public StorageManager(File baseLocation) throws CoreException
	{
		CorePlugin plugin = CorePlugin.getDefault();
		IParserFactory pf = plugin.getParserFactory();

		// NOTE: The order in which these entries are created and cleared
		// in case of changes is important. It is in depencency order.
		//
		m_providers = new MemoryStorage<Provider>(Provider.class);

		m_cspecs = new MemoryStorage<CSpec>(CSpec.class);

		m_resolutions = new MemoryStorage<Resolution>(Resolution.class);

		m_materializations = new FileStorage<Materialization>(new File(baseLocation, Materialization.TAG),
				pf.getMaterializationParser(), Materialization.class, Materialization.SEQUENCE_NUMBER);

		m_opmls = new FileStorage<OPML>(new File(baseLocation, OPML.TAG), pf.getOPMLParser(false), OPML.class,
				OPML.SEQUENCE_NUMBER);

		m_wsBindings = new FileStorage<WorkspaceBinding>(new File(baseLocation, WorkspaceBinding.TAG),
				pf.getWorkspaceBindingParser(false), WorkspaceBinding.class, WorkspaceBinding.SEQUENCE_NUMBER);
	}

	public ISaxableStorage<CSpec> getCSpecs() throws CoreException
	{
		return m_cspecs;
	}

	public ISaxableStorage<Materialization> getMaterializations() throws CoreException
	{
		return m_materializations;
	}

	public ISaxableStorage<OPML> getOPMLs() throws CoreException
	{
		return m_opmls;
	}

	public ISaxableStorage<Provider> getProviders() throws CoreException
	{
		return m_providers;
	}

	public ISaxableStorage<Resolution> getResolutions() throws CoreException
	{
		return m_resolutions;
	}

	public ISaxableStorage<WorkspaceBinding> getWorkspaceBindings() throws CoreException
	{
		return m_wsBindings;
	}

	private void initialize() throws CoreException
	{
		if(m_materializations.sequenceChanged() || m_resolutions.sequenceChanged() || m_cspecs.sequenceChanged()
				|| m_providers.sequenceChanged() || m_opmls.sequenceChanged() || m_wsBindings.sequenceChanged())
		{
			// Don't use another thread here. It will deadlock
			//
			WorkspaceInfo.forceRefreshOnAll(new NullProgressMonitor());
		}
	}
}
