/*******************************************************************************
 * Copyright (c) 2012, 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.pde.api.tools.internal;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.pde.api.tools.internal.problems.ApiProblemFilter;
import org.eclipse.pde.api.tools.internal.provisional.problems.IApiProblem;
import org.eclipse.pde.api.tools.internal.provisional.problems.IApiProblemFilter;

/**
 * This filter store is only used to filter problem using existing filters. It
 * doesn't add or remove any filters.
 */
public class AntFilterStore extends FilterStore {

	String fComponentId = null;
	String fFiltersRoot = null;
	public static ArrayList<IApiProblem> filteredAPIProblems = new ArrayList<IApiProblem>();
	/**
	 * Constructor
	 * 
	 * @param debug
	 * @param filtersRoot
	 * @param componentID
	 */
	public AntFilterStore(String filtersRoot, String componentID) {
		fComponentId = componentID;
		fFiltersRoot = filtersRoot;
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.eclipse.pde.api.tools.internal.provisional.IApiFilterStore#isFiltered
	 * (org.eclipse.pde.api.tools.internal.provisional.problems.IApiProblem)
	 */
	@Override
	public boolean isFiltered(IApiProblem problem) {
		boolean isFiltered = super.isFiltered(problem);
		if(isFiltered){
			filteredAPIProblems.add(problem);
		}
		return isFiltered;
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.eclipse.pde.api.tools.internal.FilterStore#initializeApiFilters()
	 */
	@Override
	protected synchronized void initializeApiFilters() {
		if (fFilterMap != null) {
			return;
		}
		fFilterMap = new HashMap<String, Set<IApiProblemFilter>>(5);
		InputStream contents = null;
		try {
			File filterFileParent = new File(fFiltersRoot, fComponentId);
			if (!filterFileParent.exists()) {
				return;
			}
			contents = new BufferedInputStream(new FileInputStream(new File(filterFileParent, IApiCoreConstants.API_FILTERS_XML_NAME)));
			readFilterFile(contents);
		} catch (IOException ioe) {
		} finally {
			if (contents != null) {
				try {
					contents.close();
				} catch (IOException e) {
					// ignore
				}
			}
		}
	}

	/**
	 * Internal use method that allows auto-persisting of the filter file to be
	 * turned on or off
	 * 
	 * @param problems the problems to add the the store
	 * @param persist if the filters should be auto-persisted after they are
	 *            added
	 */
	@Override
	protected void internalAddFilters(IApiProblem[] problems, String[] comments) {
		if (problems == null || problems.length == 0) {
			return;
		}
		// This filter store doesn't handle resources so all filters are added
		// to GLOBAL
		Set<IApiProblemFilter> globalFilters = fFilterMap.get(GLOBAL);
		if (globalFilters == null) {
			globalFilters = new HashSet<IApiProblemFilter>();
			fFilterMap.put(GLOBAL, globalFilters);
		}

		for (int i = 0; i < problems.length; i++) {
			IApiProblem problem = problems[i];
			String comment = comments != null ? comments[i] : null;
			IApiProblemFilter filter = new ApiProblemFilter(fComponentId, problem, comment);
			globalFilters.add(filter);
		}
	}

}