/*******************************************************************************
* Copyright (c) 2019 IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v2.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v20.html
*
* Contributors:
*     IBM Corporation - initial API and implementation
*******************************************************************************/
import { HttpPostOutputQueue } from "./HttpPostOutputQueue";
import { PostQueueChunk } from "./PostQueueChunk";
export declare enum ChunkStatus {
    AVAILABLE_TO_SEND = 0,
    WAITING_FOR_ACK = 1,
    COMPLETE = 2
}
/**
 * The 'chunk group' maintains the list of chunks for a specific timestamp that
 * we are currently trying to send to the server.
 *
 * Each chunk in the chunk group is in one of these states:
 * - AVAILABLE_TO_SEND: Chunks in this state are available to be sent by the next available worker.
 * - WAITING_FOR_ACK: Chunks in this state are in the process of being sent by one of the workers.
 * - COMPLETE: Chunks in this state have been sent and acknowledged by the server.
 *
 * The primary goal of chunk groups is to ensure that chunks will never be sent
 * to the server out of ascending-timestamp order: eg we will never send the
 * server a chunk of 'timestamp 20', then a chunk of 'timestamp 19'. The
 * 'timestamp 20' chunks will wait for all of the 'timestamp 19' chunks to be
 * sent.
 */
export declare class PostQueueChunkGroup {
    /** List of chunks; this map is immutable once created (as are the chunks themselves) */
    private readonly _chunkMap;
    private readonly _chunkStatus;
    private readonly _parent;
    private readonly _timestamp;
    private readonly _projectId;
    /**
     * After X hours (eg 24), give up on trying to send this chunk group to the
     * server. At this point the data is too stale to be useful.
     */
    private readonly _expireTime;
    constructor(timestamp: number, projectId: string, expireTime: number, base64Compressed: string[], parent: HttpPostOutputQueue);
    /** A group is complete if every chunk is ChunkStatus.COMPLETE */
    isGroupComplete(): boolean;
    /** Called by a worker thread to report a successful send. */
    informPacketSent(chunk: PostQueueChunk): void;
    /**
     * Called by a worker to report a failed send; we make the chunk
     * available to send by another worker.
     */
    informPacketFailedToSend(chunk: PostQueueChunk): void;
    /** Returns the next chunk to be sent, or empty if none are currently available. */
    acquireNextChunkAvailableToSend(): PostQueueChunk;
    readonly timestamp: number;
    readonly projectId: string;
    readonly expireTime: number;
}
