/*******************************************************************************
* Copyright (c) 2019, 2020 IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v2.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v20.html
*
* Contributors:
*     IBM Corporation - initial API and implementation
*******************************************************************************/
import { ChangedFileEntry } from "./ChangedFileEntry";
import { FileWatcher } from "./FileWatcher";
/**
 * When file/directory change events occur in quick succession (within
 * milliseconds of each other), it tends to imply that they are related. For
 * example, if you were refactoring Java code, that refactoring operation might
 * touch a bunch of source files at once. This means, in order to avoid
 * performing extra builds, we should batch together changes that occur within
 * close temporal proximity.
 *
 * However, we don't want to wait TOO long for new events, otherwise this
 * introduces latency between when the user makes a change, and when their build
 * actually starts.
 *
 * This class implements an algorithm that groups together changes that occur
 * within TIME_TO_WAIT_FOR_NO_NEW_EVENTS_IN_MSECS milliseconds of each other.
 *
 * The algorithm is: After at least one event is received, wait for there to be
 * be no more events in the stream of events (within eg 1000 msecs) before
 * sending them to the server. If an event is seen within 1000 msecs, the
 * timer is reset and a new 1000 msec timer begins. Batch together events seen
 * since within a given timeframe, and send them as a single request.
 *
 * This class receives file change events from the watch service, and forwards
 * batched groups of events to the HTTP POST output queue.
 *
 */
export declare class FileChangeEventBatchUtil {
    private static readonly TIME_TO_WAIT_FOR_NO_NEW_EVENTS_IN_MSECS;
    private _files;
    private _timer;
    private _disposed;
    private readonly _parent;
    private readonly _projectId;
    constructor(projectId: string, parent: FileWatcher);
    /**
     * When files have changed, add them to the list and reset the timer task ahead
     * X millseconds.
     */
    addChangedFiles(changedFileEntries: ChangedFileEntry[]): void;
    dispose(): void;
    private doTimerTask;
    private generateChangeListSummaryForDebug;
    /** For any given path: If there are multiple entries of the same type in a row, then remove all but the first. */
    private removeDuplicateEventsOfType;
}
