"use strict";
/*******************************************************************************
* Copyright (c) 2019, 2020 IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v2.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v20.html
*
* Contributors:
*     IBM Corporation - initial API and implementation
*******************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Simple singleton logger with 4 log levels.
 *
 * Log levels:
 * - DEBUG: Fine-grained, noisy, excessively detailed, and mostly irrelevant messages.
 * - INFO: Coarse-grained messages related to the high-level inner workings of the code.
 * - ERROR: Errors which are bad, but not entirely unexpected, such as errors I/O errors when
 *          running on a flaky network connection.
 * - SEVERE: Unexpected errors that strongly suggest a client/server implementation bug or a
 *           serious client/server runtime issue.
 */
var LogLevel;
(function (LogLevel) {
    LogLevel[LogLevel["DEBUG"] = 1] = "DEBUG";
    LogLevel[LogLevel["INFO"] = 2] = "INFO";
    LogLevel[LogLevel["ERROR"] = 3] = "ERROR";
    LogLevel[LogLevel["SEVERE"] = 4] = "SEVERE";
})(LogLevel = exports.LogLevel || (exports.LogLevel = {}));
class LogSettings {
    constructor() {
        this._outputLogsToScreen = true;
        this._logLevel = LogLevel.INFO;
        this._startTimeInMsecs = new Date().getTime();
    }
    get logLevel() {
        return this._logLevel;
    }
    set logLevel(e) {
        this._logLevel = e;
    }
    static getInstance() {
        return LogSettings.instance;
    }
    generatePrefix() {
        const elapsed = new Date().getTime() - this._startTimeInMsecs;
        const seconds = Math.floor(elapsed / 1000);
        const msecs = 1000 + (elapsed % 1000);
        const msecsString = ("" + msecs).substring(1);
        let properDate = "";
        {
            const date = new Date();
            const milliseconds = 1000 + date.getMilliseconds();
            // request a weekday along with a long date
            const options = {
                day: "numeric",
                formatMatcher: "basic",
                hour: "numeric",
                hour12: false,
                minute: "2-digit",
                month: "long",
                second: "2-digit",
            };
            properDate = "[" + new Intl.DateTimeFormat("en-US", options).format(date)
                + "." + milliseconds.toString().substring(1, 4) + "]";
        }
        return properDate + " [" + seconds + "." + msecsString + "] ";
    }
    setOutputLogsToScreen(outputToScreen) {
        this._outputLogsToScreen = outputToScreen;
    }
    setFileLogger(fileLogger) {
        this._fileLogger = fileLogger;
    }
    internalGetFileLogger() {
        return this._fileLogger;
    }
    get outputLogsToScreen() {
        return this._outputLogsToScreen;
    }
}
LogSettings.instance = new LogSettings();
exports.LogSettings = LogSettings;
function setLogLevel(l) {
    LogSettings.getInstance().logLevel = l;
}
exports.setLogLevel = setLogLevel;
function debug(str) {
    if (LogSettings.getInstance().logLevel > LogLevel.DEBUG) {
        return;
    }
    const prefix = LogSettings.getInstance().generatePrefix();
    const msg = prefix + str;
    printOut(msg);
}
exports.debug = debug;
function info(str) {
    if (LogSettings.getInstance().logLevel > LogLevel.INFO) {
        return;
    }
    const prefix = LogSettings.getInstance().generatePrefix();
    const msg = prefix + str;
    printOut(msg);
}
exports.info = info;
function error(str, err) {
    if (LogSettings.getInstance().logLevel > LogLevel.ERROR) {
        return;
    }
    const prefix = LogSettings.getInstance().generatePrefix();
    const suffix = err ? ": " : "";
    const msg = prefix + "! ERROR !  " + str + suffix;
    printErr(msg);
    if (err) {
        printErr(err);
    }
}
exports.error = error;
function severe(str, err) {
    const prefix = LogSettings.getInstance().generatePrefix();
    const suffix = err ? ": " : "";
    const msg = prefix + "!!! SEVERE !!!  " + str + suffix;
    printErr(msg);
    if (err) {
        printErr(err);
    }
}
exports.severe = severe;
function printOut(entry) {
    const outputLogsToScreen = LogSettings.getInstance().outputLogsToScreen;
    const fileLogger = LogSettings.getInstance().internalGetFileLogger();
    if (fileLogger) {
        fileLogger.log(entry);
    }
    if (outputLogsToScreen) {
        console.log(entry);
    }
}
function printErr(entry) {
    const fileLogger = LogSettings.getInstance().internalGetFileLogger();
    const outputLogsToScreen = LogSettings.getInstance().outputLogsToScreen;
    if (entry instanceof Error) {
        const entryError = entry;
        const msg = entryError.name + " " + entryError.message + " " + entryError.stack;
        if (outputLogsToScreen) {
            console.error(msg);
        }
        if (fileLogger) {
            fileLogger.log(msg);
        }
    }
    else {
        // Non-error
        if (outputLogsToScreen) {
            console.error(entry);
        }
        if (fileLogger) {
            fileLogger.log(entry);
        }
    }
}
function logLevelToString(l) {
    if (l === LogLevel.DEBUG) {
        return "DEBUG";
    }
    else if (l === LogLevel.ERROR) {
        return "ERROR";
    }
    else if (l === LogLevel.INFO) {
        return "INFO";
    }
    else {
        return "SEVERE";
    }
}
exports.logLevelToString = logLevelToString;
