"use strict";
/*******************************************************************************
* Copyright (c) 2019, 2020 IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v2.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v20.html
*
* Contributors:
*     IBM Corporation - initial API and implementation
*******************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const WatchedPath_1 = require("./WatchedPath");
const log = require("./Logger");
/**
 * This class uses the directory/file monitoring functionality of the 3rd party
 * Chokidar library for file monitoring (but most of the interface with Chokidar
 * is in WatchedPath)
 *
 * See IPlatformWatchService for general information on watch services.
 */
class WatchService {
    constructor() {
        this._disposed = false;
        this._watchedProjects = new Map();
    }
    setParent(parent) {
        this._parent = parent;
    }
    addPath(fileToMonitor, ptw) {
        if (this._disposed) {
            return;
        }
        const key = ptw.projectId;
        log.info("Path '" + fileToMonitor + "' added to WatchService for " + key);
        const wp = this._watchedProjects.get(key);
        if (wp) {
            wp.dispose();
        }
        this._watchedProjects.set(key, new WatchedPath_1.WatchedPath(fileToMonitor, this, ptw));
    }
    removePath(fileToMonitor, oldProjectToWatch) {
        if (this._disposed) {
            return;
        }
        const key = oldProjectToWatch.projectId;
        const wp = this._watchedProjects.get(key);
        if (wp) {
            log.info("Path '" + fileToMonitor + "' removed from WatchService for project " + key);
            wp.dispose();
            this._watchedProjects.delete(key);
        }
        else {
            log.error("Path '" + fileToMonitor + "' attempted to be removed for project "
                + key + " , but could not be found in WatchService.");
        }
    }
    handleEvent(event) {
        if (this._disposed) {
            return;
        }
        try {
            log.info("Received event " + event.toString());
            const timeReceived = Math.round(new Date().getTime());
            if (this._parent) {
                this._parent.receiveNewWatchEventEntry(event, timeReceived);
            }
            else {
                log.severe("Watch server parent was null thus event could not be passed.");
            }
        }
        catch (e) {
            log.severe("handleEvent caught an uncaught exception.", e);
        }
    }
    dispose() {
        if (this._disposed) {
            return;
        }
        log.info("dispose() called on WatchService");
        this._disposed = true;
        for (const [, value] of this._watchedProjects) {
            value.dispose();
        }
    }
    get parent() {
        return this._parent;
    }
    generateDebugState() {
        if (this._disposed) {
            return "[disposed]";
        }
        let result = "";
        for (const [key, value] of this._watchedProjects) {
            result += "- " + key + " | " + value.pathRoot + "\n";
        }
        return result;
    }
}
exports.WatchService = WatchService;
