"use strict";
/*******************************************************************************
* Copyright (c) 2019, 2020 IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v2.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v20.html
*
* Contributors:
*     IBM Corporation - initial API and implementation
*******************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const log = require("./Logger");
/**
 * AuthTokenWrapper is the conduit through the internal filewatcher codebase
 * requests secure authentication tokens from the IDE. In cases where the
 * authTokenProvider is null (eg is secure auth is not required), the methods of
 * this class are no-ops.
 *
 * This class was created as part of issue codewind/1309.
 */
class AuthTokenWrapper {
    constructor(authTokenProvider) {
        this._recentInvalidKeysQueue = new Array();
        this._authTokenProvider = authTokenProvider;
        this._invalidKeysSet = new Set();
    }
    getLatestToken() {
        if (!this._authTokenProvider) {
            return undefined;
        }
        const token = this._authTokenProvider.getLatestAuthToken();
        if (!token) {
            return undefined;
        }
        log.info("IDE returned a new security token to filewatcher: " + this.digest(token));
        return token;
    }
    /** Inform the IDE when a token is rejected. */
    informBadToken(token) {
        if (!this._authTokenProvider || !token || !token.accessToken) {
            return;
        }
        // We've already reported this key as invalid, so just return it
        if (this._invalidKeysSet.has(token.accessToken)) {
            log.info("Filewatcher informed us of a bad token, but we've already reported it to the IDE: "
                + this.digest(token));
            return;
        }
        // We have a new token that we have not previously reported as invalid.
        this._invalidKeysSet.add(token.accessToken);
        this._recentInvalidKeysQueue.push(token); // append to end
        while (this._recentInvalidKeysQueue.length > AuthTokenWrapper.KEEP_LAST_X_STALE_KEYS) {
            const keyToRemove = this._recentInvalidKeysQueue.shift(); // remove from front
            if (!keyToRemove) {
                break;
            }
            this._invalidKeysSet.delete(keyToRemove.accessToken);
        }
        this._authTokenProvider.informReceivedInvalidAuthToken(token);
    }
    /**
     * Return a representation of the token that is at most 32 characters long, so
     * as not to overwhelm the log file.
     */
    digest(token) {
        if (!token) {
            return undefined;
        }
        const key = token.accessToken;
        if (!key) {
            return undefined;
        }
        return key.substring(0, Math.min(key.length, 32));
    }
}
AuthTokenWrapper.KEEP_LAST_X_STALE_KEYS = 10;
exports.AuthTokenWrapper = AuthTokenWrapper;
