"use strict";
/*******************************************************************************
* Copyright (c) 2019 IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v2.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v20.html
*
* Contributors:
*     IBM Corporation - initial API and implementation
*******************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const chokidar = require("chokidar");
const WatchEventEntry_1 = require("./WatchEventEntry");
const watchEvents = require("./WatchEventEntry");
const log = require("./Logger");
const PathFilter_1 = require("./PathFilter");
const fs_1 = require("fs");
const pathLib = require("path");
const PathUtils_1 = require("./PathUtils");
class WatchedPath {
    constructor(pathRoot, parent, projectToWatch) {
        this._disposed = false;
        this._parent = parent;
        this._pathRoot = pathRoot;
        this._normalizedPath = PathUtils_1.normalizePath(pathRoot);
        this._pathFilter = new PathFilter_1.PathFilter(projectToWatch);
        this._projectToWatch = projectToWatch;
        this._watchIsReady = false;
        this.waitForPathToExist(0, new Date());
    }
    waitForPathToExist(attempts, startTime) {
        let success = false;
        try {
            if (fs_1.existsSync(this._pathRoot) && fs_1.statSync(this._pathRoot).isDirectory()) {
                success = true;
                this.init();
            }
        }
        catch (e) { /* ignore*/ }
        if (!success) {
            if (attempts % 5 === 0) {
                log.info("Waiting for path to exist: " + this._pathRoot);
            }
            // TODO: Replace this with a monotonically increasing time value
            if (new Date().getTime() - startTime.getTime() > 60 * 5 * 1000) {
                // After 5 minutes, stop waiting for the watch and report the error
                log.error("Watch failed on " + this._pathRoot);
                this._parent.parent.sendWatchResponseAsync(false, this._projectToWatch);
            }
            else {
                // Wait up to 5 minutes for the directory to appear.
                setTimeout(() => {
                    this.waitForPathToExist(attempts + 1, startTime);
                }, 1000);
            }
        }
    }
    init() {
        const normalizedPathConst = this._normalizedPath;
        const pathFilterConst = this._pathFilter;
        /** Decide what to ignore using our custom function; true if should ignore, false otherwise. */
        const ignoreFunction = (a, b) => {
            if (!a) {
                return false;
            }
            a = PathUtils_1.convertAbsolutePathWithUnixSeparatorsToProjectRelativePath(PathUtils_1.normalizePath(a), normalizedPathConst);
            if (!a) {
                return false;
            }
            return pathFilterConst.isFilteredOutByFilename(a) || pathFilterConst.isFilteredOutByPath(a);
        };
        // Use polling on Windows, due to high CPU and memory usage of non-polling when using chokidar.
        // Use polling on Mac due to unreliable directory deletion detection.
        const usePollingVal = true; // os.platform().indexOf("win32") !== -1 || os.platform().indexOf("darwin") !== -1;
        this._watcher = chokidar.watch(this._pathRoot, {
            binaryInterval: 2000,
            disableGlobbing: true,
            ignoreInitial: true,
            ignored: ignoreFunction,
            interval: 250,
            persistent: true,
            usePolling: usePollingVal,
        });
        this._watcher
            .on("add", (path) => this.handleFileChange(path, "CREATE"))
            .on("change", (path) => this.handleFileChange(path, "MODIFY"))
            .on("unlink", (path) => this.handleFileChange(path, "DELETE"));
        this._watcher
            .on("addDir", (path) => this.handleDirChange(path, "CREATE"))
            .on("unlinkDir", (path) => this.handleDirChange(path, "DELETE"))
            .on("error", (error) => log.severe(`Watcher error: ${error}`, error))
            .on("raw", (event, path, details) => {
            log.debug("Watcher raw event info:" + event + "|" + path);
            if (path === this._pathRoot && !fs_1.existsSync(path)) {
                this.handleDirChange(path, "DELETE");
            }
        })
            .on("ready", () => {
            if (this._disposed) {
                return;
            }
            this._watchIsReady = true;
            log.info("Initial scan of '" + this._pathRoot + "' complete. Ready for changes");
            this._parent.parent.sendWatchResponseAsync(true, this._projectToWatch);
        });
        // this._watcher.on("change", (path, stats) => {
        //     if (stats) { console.log(`File ${path} changed size to ${stats.size}`); }
        // });
    }
    dispose() {
        if (this._disposed) {
            return;
        }
        this._disposed = true;
        log.info("dispose() called on WatchedPath for " + this._pathRoot);
        this._watchIsReady = false;
        this.closeWatcherAsync();
    }
    async closeWatcherAsync() {
        if (this._watcher) {
            this._watcher.close();
        }
    }
    handleFileChange(path, type) {
        if (this._disposed) {
            return;
        }
        // File changes are not valid until the watch is ready.
        if (!this._watchIsReady) {
            log.debug("Ignoring  file " + path);
            return;
        }
        const eventType = watchEvents.getEventTypeFromString(type);
        const entry = new WatchEventEntry_1.WatchEventEntry(eventType, path, false);
        this._parent.handleEvent(entry);
    }
    handleDirChange(path, type) {
        if (this._disposed) {
            return;
        }
        // File changes are not valid until the watch is ready.
        if (!this._watchIsReady) {
            log.debug("Ignoring dir " + path);
            return;
        }
        const eventType = watchEvents.getEventTypeFromString(type);
        const entry = new WatchEventEntry_1.WatchEventEntry(eventType, path, true);
        this._parent.handleEvent(entry);
        // this.scanCreatedDirectory(path);
        // If it is the root directory that is going away, then dispose of the watcher.
        if (path === this._pathRoot && type === "DELETE") {
            log.debug("Watcher observed deletion of the root path "
                + this._pathRoot + ", so disposing in 30 seconds from now.");
            if (this._watcher) {
                this._watcher.unwatch(this._pathRoot);
            }
            // Wait 30 seconds to ensure any other related events have been processed.
            setTimeout(() => {
                log.debug("Watcher previously observed deletion of the root path "
                    + this._pathRoot + ", now calling dispose().");
                this.dispose();
            }, 30 * 1000);
        }
    }
    /**
     * Recursively scan a directory and create a list of any files/folders found inside. My thinking
     * in creating this was it would reduce incidents of events being missed by Chokidar, but the code
     * had no obvious effect.
     * @param path Path to scan
     */
    scanCreatedDirectory(path) {
        const directoriesFound = [];
        const filesFound = [];
        const newWatchEvents = new Array();
        this.scanCreatedDirectoryInner(path, directoriesFound, filesFound);
        for (const dir of directoriesFound) {
            newWatchEvents.push(new WatchEventEntry_1.WatchEventEntry(watchEvents.EventType.CREATE, dir, true));
        }
        for (const file of filesFound) {
            newWatchEvents.push(new WatchEventEntry_1.WatchEventEntry(watchEvents.EventType.CREATE, file, false));
        }
        for (const we of newWatchEvents) {
            this._parent.handleEvent(we);
        }
    }
    scanCreatedDirectoryInner(path, directoriesFound, filesFound) {
        fs_1.readdir(path, (err, files) => {
            if (err) {
                log.error("Error received when attempting to read directory: " + path, err);
                return;
            }
            if (!files) {
                return;
            }
            files.forEach((fileInner) => {
                if (!fileInner) {
                    return;
                }
                const fullPath = path + pathLib.sep + fileInner;
                if (fs_1.statSync(fullPath).isDirectory()) {
                    if (fullPath === path) {
                        return;
                    }
                    directoriesFound.push(fullPath);
                    this.scanCreatedDirectoryInner(fullPath, directoriesFound, filesFound);
                }
                else {
                    filesFound.push(fullPath);
                }
            });
        });
    }
}
exports.WatchedPath = WatchedPath;
