"use strict";
/*******************************************************************************
* Copyright (c) 2019 IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v2.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v20.html
*
* Contributors:
*     IBM Corporation - initial API and implementation
*******************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const request = require("request-promise-native");
const log = require("./Logger");
const ExponentialBackoffUtil_1 = require("./ExponentialBackoffUtil");
const ProjectToWatch_1 = require("./ProjectToWatch");
class HttpGetStatusThread {
    constructor(baseUrl, parent) {
        this._timer = null;
        this._disposed = false;
        this._baseUrl = baseUrl;
        this._parent = parent;
        this.resetTimer();
        this.queueStatusUpdate();
    }
    async queueStatusUpdate() {
        if (this._disposed) {
            return;
        }
        if (this._inInnerLoop) {
            return;
        }
        this.innerLoop();
    }
    dispose() {
        if (this._disposed) {
            return;
        }
        log.info("dispose() called on HttpGetStatusThread");
        this._disposed = true;
    }
    async doHttpGet() {
        const options = {
            json: true,
            resolveWithFullResponse: true,
            timeout: 20000,
        };
        try {
            const url = this._baseUrl + "/api/v1/projects/watchlist";
            log.info("Initiating GET request to " + url);
            const httpResult = await request.get(url, options);
            if (httpResult.statusCode && httpResult.statusCode === 200 && httpResult.body) {
                // Strip EOL characters to ensure it fits on one log line.
                let bodyVal = JSON.stringify(httpResult.body);
                bodyVal = bodyVal.replace("\n", "");
                bodyVal = bodyVal.replace("\r", "");
                log.info("GET response received: " + bodyVal);
                const w = httpResult.body;
                if (w == null || w === undefined) {
                    log.error("Expected value not found for GET watchlist endpoint");
                    return null;
                }
                const result = new Array();
                for (const e of w.projects) {
                    // Santity check the json parsing
                    if (!e.projectID || !e.pathToMonitor) {
                        log.error("JSON parsing of GET watchlist endpoint failed with missing values");
                        return null;
                    }
                    result.push(new ProjectToWatch_1.ProjectToWatch(e, false));
                }
                return result;
            }
            else {
                return null;
            }
        }
        catch (err) {
            log.error("GET request failed. [" + (err.message) + "]");
            return null;
        }
    }
    async innerLoop() {
        try {
            this._inInnerLoop = true;
            log.debug("HttpGetStatus - beginning GET request loop.");
            let success = false;
            const delay = ExponentialBackoffUtil_1.ExponentialBackoffUtil.getDefaultBackoffUtil(4000);
            let result = null;
            while (!success && !this._disposed) {
                result = await this.doHttpGet();
                success = result != null;
                if (!success) {
                    log.error("HTTP get request failed");
                    await delay.sleepAsync();
                    delay.failIncrease();
                }
            }
            if (result && result.length > 0) {
                this._parent.updateFileWatchStateFromGetRequest(result);
            }
        }
        finally {
            log.info("HttpGetStatus - GET request loop complete.");
            this._inInnerLoop = false;
            this.resetTimer();
        }
    }
    resetTimer() {
        if (this._timer != null) {
            clearTimeout(this._timer);
        }
        // Refresh every 120 seconds.
        this._timer = setTimeout(() => {
            this.queueStatusUpdate();
        }, 120 * 1000);
    }
}
HttpGetStatusThread.REFRESH_EVERY_X_SECONDS = 60;
exports.HttpGetStatusThread = HttpGetStatusThread;
