/*
 * Copyright (c) 2007-2008 Compuware Corporation and others.
 * 
 * All rights reserved. This program and the accompanying 
 * materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at:
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors:
 *     Compuware Corporation - initial API and implementation
 */
package org.eclipse.corona.internal.test;

import java.util.ArrayList;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.PriorityQueue;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.corona.test.ITestEvent;
import org.eclipse.corona.test.ITestManager;
import org.eclipse.corona.test.ITestRunner;
import org.eclipse.corona.test.ITestSuite;
import org.eclipse.corona.test.ITestSummary;
import org.eclipse.corona.test.InvalidTestSuite;
import org.eclipse.corona.test.ITestResult.TestSuiteExecutionStatus;
import org.osgi.framework.BundleContext;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.event.Event;
import org.osgi.service.event.EventAdmin;


/**
 * Corona Test Framework Manager
 * 
 */
public class TestManager implements ITestManager {
	private Logger logger = Logger.getLogger(TestManager.class.getName());
	
	private Hashtable<String, ITestRunner> listTestRunners = new Hashtable<String, ITestRunner>();

	private EventAdmin eventAdmin;
	
	/**
	 * Construct and initialize the Corona Test Framework environment.
	 * 
	 */
	public TestManager() {
	}
	
	/**
	 * Activate the service component.
	 *
	 * This method is called from OSGi to start the service.
	 * 
	 * @param ctxComponent
	 */
	protected void activate(ComponentContext ctxComponent) {
		ctxComponent.getBundleContext();
		
		initTestRunners();

		/*
		 * get the OSGi EventAdmin service
		 */
		eventAdmin = (EventAdmin) ctxComponent.locateService("EventAdmin");
	}

	/**
	 * Activate the service component.
	 * 
	 * This method is called from OSGi to stop the service.
	 * 
	 * @param ctxComponent
	 */
	protected void deactivate(ComponentContext ctxComponent) {
	}

	/**
	 * Initialize the list of TestRunners that will be used to run test suites.
	 */
	private void initTestRunners() {
		/*
		 * get all registered test runners via Eclipse extension points
		 */
		IExtensionPoint extPoint= Platform.getExtensionRegistry().getExtensionPoint(ITestRunner.EXT_POINT);
		if ( extPoint != null ) {
			IConfigurationElement[] extensions = extPoint.getConfigurationElements();
			for( IConfigurationElement extension : extensions ) {
				/*
				 * build a Corona Test Suite from its ext point definition
				 */
				ITestRunner testRunner;
				try {
					testRunner = (ITestRunner) extension.createExecutableExtension(ITestRunner.PROPERTY_CLASS);
				} catch (CoreException e) {
					logger.warn( "Unable to create ITestRunner extension", e);
					continue;
				}

				// set name of the test runner
				String name = extension.getAttribute(ITestRunner.PROPERTY_NAME);
				testRunner.setName(name);

				// set type of the test runner
				String type = extension.getAttribute(ITestRunner.PROPERTY_TYPE);
				testRunner.setType(type);

				this.listTestRunners.put(type, testRunner);
			}
		}
	}
	/**
	 * Run all of the registered ITestSuite's
	 * 
	 * Note: check both OSGi service references and Eclipse
	 * extension point registry for test suites.
	 */
	public ITestSummary[] runAllTestSuites() throws Exception {
		ArrayList<ITestSummary> listTestSummary = new ArrayList<ITestSummary>();
		
		/*
		 * run the test cases for each test suite
		 */
		ITestSuite[] testSuites = this.testSuites();
		for (int i = 0; i < testSuites.length; i++) {
			ITestSummary testSummary = runTestSuite( testSuites[i] );
			
			/*
			 * aggregate all test results
			 */
			listTestSummary.add(testSummary);
		}
			
		return (ITestSummary[]) listTestSummary.toArray(new ITestSummary[listTestSummary.size()]);
	}

	/**
	 * Run the ITestSuite w/ the appropriate ITestRunner
	 * 
	 * @return test summary results, or null if no test runner match
	 */
	public ITestSummary runTestSuite(ITestSuite testSuite) {
		
		/*
		 * lookup which test runner to use
		 */
		ITestRunner testRunner = (ITestRunner) this.listTestRunners.get(testSuite.getRunnerType());
		if ( testRunner == null ) {
			logger.warn( "Could not find ITestRunner of type: "+testSuite.getRunnerType());
						
			return null;
		}
		
		/*
		 * Run the ITestSuite w/ the appropriate ITestRunner
		 */		
		postCTFTestSuiteEvent( ITestEvent.TEST_START, testSuite, TestSuiteExecutionStatus.EXECUTING.label());		
		logger.info( "Running CTF test suite: "+testSuite.getName());
		ITestSummary testSummary = testRunner.runTestSuite(testSuite);
		if( testSummary != null ){
			postCTFTestSuiteEvent( ITestEvent.TEST_STOP, testSuite, testSummary.getTestResult().getCTFTestSuiteExecutionStatus().label());
		}
		
		return testSummary;
	}
		
	private void postCTFTestSuiteEvent(String testState, ITestSuite testSuite,
			String testSuiteExecutionStatus) {
		/*
		 * init event dictionary
		 */
		Dictionary<String, Object> dictionary = new Hashtable<String, Object>();
		dictionary.put(ITestEvent.TIMESTAMP, System.currentTimeMillis());
		dictionary.put(ITestEvent.EVENT, testState);
		dictionary.put(ITestEvent.TEST_SUITE, testSuite);
				
		dictionary.put(ITestEvent.TEST_STATUS, testSuiteExecutionStatus);
		
		/*
		 * post the test event
		 */
		Event testEvent = new Event(ITestEvent.TOPIC_ROOT, dictionary);
		this.eventAdmin.postEvent(testEvent);
	}

	public ITestSuite[] testSuites() {
		PriorityQueue<ITestSuite> priorityQueue = new PriorityQueue<ITestSuite>();
		
		/*
		 * get all registered test suites via OSGi declarative services
		 */
		BundleContext ctxBundle = Activator.getDefault().getBundleContext();		
		ServiceReference srvRefs[];
		try {
			srvRefs = ctxBundle.getServiceReferences(ITestSuite.class.getName(), null);
			if ( srvRefs != null ) {
				for (ServiceReference srvRef : srvRefs) {
					ITestSuite testSuite = (ITestSuite)ctxBundle.getService(srvRef);
					
					priorityQueue.add(testSuite);
				}
			}
		} catch (InvalidSyntaxException e) {
			logger.error( "Unable to lookup OSGi srvref of ITestSuite", e);
		}

		/*
		 * get all registered test suites via Eclipse extension points
		 */
		IExtensionPoint extPoint= Platform.getExtensionRegistry().getExtensionPoint(ITestSuite.EXT_POINT);
		if ( extPoint != null ) {
			IConfigurationElement[] extensions = extPoint.getConfigurationElements();
			for( IConfigurationElement extension : extensions ) {
				/*
				 * build a Corona Test Suite from its ext point definition
				 */
				ITestSuite testSuite;
				try {
					testSuite = (ITestSuite) extension.createExecutableExtension(ITestSuite.PROPERTY_CLASS);
				} catch (Throwable t) {
					logger.error( "Unable to create ext point ITestSuite", t);
					testSuite = new InvalidTestSuite(t);
				}
				
				// set test suite properties
				testSuite.setName(extension.getAttribute(ITestSuite.PROPERTY_NAME));
				
				// set the contributor (bundle or fragment) of the test suite
				testSuite.setContributorLabel(extension.getDeclaringExtension().getContributor().toString());

				// set the test suite run level
				String level = extension.getAttribute(ITestSuite.PROPERTY_LEVEL);
				if ( level == null ) {
					level = ITestSuite.DEFAULT_LEVEL;
				}
				testSuite.setLevel(level);
				
				// set the test runner
				String typeRunner = extension.getAttribute(ITestSuite.PROPERTY_RUNNER);
				if ( typeRunner == null ) {
					typeRunner = ITestSuite.DEFAULT_RUNNER;
				}
				testSuite.setRunnerType(typeRunner);
				
				// add test suite to collection
				priorityQueue.add(testSuite);
			}
		}
		
		return priorityQueue.toArray( new ITestSuite[priorityQueue.size()]);
	}	
}
