/*
 * Copyright (c) 2007-2008 Compuware Corporation and others.
 * 
 * All rights reserved. This program and the accompanying 
 * materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at:
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors:
 *     Compuware Corporation - initial API and implementation
 */
package org.eclipse.corona.diagnostic;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Dictionary;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Properties;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.log4j.Logger;
import org.eclipse.corona.internal.diagnostic.Activator;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

/**
 * AbstractDiagnosticData provides the base class for
 * diagnostic data handling. 
 * 
 * @author doflynn
 */
public abstract class AbstractDiagnosticData implements
		IDiagnosticData {

	public static final String ATTR_COUNT = "count";

	private transient Logger logger = Logger.getLogger(AbstractDiagnosticData.class.getName());
	
	private static final long serialVersionUID = 1L;
	protected String name;
	protected String version;
	protected Hashtable<String, Object> properties;
	protected Document docDiag;
	protected Element rootDiag;
	
	protected AbstractDiagnosticData() {
		init();
	}
	
	public String toString() {
		String str = "UnknownDiagnosticData";
		
		if ( name != null ) {
			str = name;
		}
		
		if ( version != null ) {
			str += " ("+version+")";
		}
		
		return str;
	}
	private void init() {
		this.properties = new Hashtable<String, Object>();
	}

	/**
	 * Initialize OSGi service component
	 * 
	 * @param dictionary
	 * @return 
	 */
	@SuppressWarnings("unchecked")
	protected void initComponent(Dictionary dictionary) {
		Object obj;
		
		/*
		 * get the diagnostic component's name
		 */
		obj = dictionary.get(IDiagnosticData.COMPONENT_NAME);
		if ( obj instanceof String[] ) {
			name = ((String[])obj)[0];
		}

		/*
		 * get the diagnostic component's version
		 */
		obj = dictionary.get(IDiagnosticData.COMPONENT_VERSION);
		if ( obj instanceof String[] ) {
			version = ((String[])obj)[0];
		}
	}
	
	public String getName() {
		return this.name;
	}
	
	public void setName(String name) {
		this.name = name;
	}

	public String getVersion() {
		return this.version;
	}

	public void setVersion(String version) {
		this.version = version;		
	}

	public Hashtable<String, Object> getProperties() {
		return this.properties;
	}
	
	public void capture() {
		init();
		
		addProperty("timestamp", new Long(System.currentTimeMillis()) );
	}

	public void addProperty( String name, Object value) {
		this.properties.put(name, value);
	}
	
	/**
	 * 
	 */
	public Element createElementPropertyList() {
		
		Element rootPropertyList = docDiag.createElement("PropertyList");
		
		/*
		 * collect all properties
		 */
		Enumeration<String> propNames = this.properties.keys();
		while (propNames.hasMoreElements()) {
			String name = propNames.nextElement();
			Object value = properties.get(name);
			
			Element propertyElement = createElementProperty(docDiag, name, value);
			
			rootPropertyList.appendChild(propertyElement);
		}
		
		return rootPropertyList;
	}

	/**
	 * @see IDiagnosticData#save(OutputStream)
	 */
	public void save(OutputStream outStream) throws IOException {
		
		try {
			TransformerFactory transformerFactory = TransformerFactory.newInstance();
	        Transformer transformer = transformerFactory.newTransformer();
	        DOMSource source = new DOMSource(this.docDiag);
	        
	        StreamResult result =  new StreamResult(outStream);
	        transformer.transform(source, result);
		}
		catch(Exception x) {
			logger.warn("Unable to save diagnostic data", x);
		}
	}
	
	/**
	 * @throws ParserConfigurationException 
	 * @see IDiagnosticData#load(InputStream inStream)
	 */
	public void load(InputStream inStream) throws DiagnosticException {
		
		throw new DiagnosticException("Not Yet Implemented!");
	}

	/**
	 * Close the object i/o stream used to load/save the 
	 * diagnostic data.
	 * 
	 * @throws IOException
	 */
	protected void cleanUp() throws IOException {
		this.docDiag = null;
	}

	protected void initXmlDocument() throws ParserConfigurationException {
	    DocumentBuilderFactory factory =  DocumentBuilderFactory.newInstance();
	    DocumentBuilder builder = factory.newDocumentBuilder();
	    docDiag = builder.newDocument();
		rootDiag = docDiag.createElement(this.getName());
		rootDiag.setAttribute("version", this.getVersion() );
	    docDiag.appendChild(rootDiag);
	}

	protected Element createElement(String name, String value) {
		Element element = docDiag.createElement(name);
		Text text = docDiag.createTextNode(value.toString());
		element.appendChild(text);
		
		return element;
	}

	protected Element createElementProperty(Document doc, String name,
			Object value) {
		Element propertyElement = doc.createElement("Property");
		propertyElement.setAttribute("name", name);
		
		/*
		 * set type property's type (default=string)
		 */
		String type="unknown";
		if ( value instanceof java.lang.String) {
			type="string";
		}
		else if ( value instanceof java.lang.Integer ) {
			type = "int";
		}
		else if ( value instanceof java.lang.Long) {
			type = "long";
		}

		/*
		 * special handling of 'objectClass'.  property type is used by
		 * OSGi service references.  a one-to-many relationship can exist.
		 */
		if ( name.equals("objectClass") ) {
			String[] srvClasses = ((String[])value);
			for (int i = 0; i < srvClasses.length; i++) {
				return createElementProperty(doc, "service.class", srvClasses[i]);
			}
			
		}
		propertyElement.setAttribute("type", type);

		Text text = doc.createTextNode(value.toString());
		propertyElement.appendChild(text);

		return propertyElement;
	}
}
