/*
 * Copyright (c) 2007-2008 Compuware Corporation and others.
 * 
 * All rights reserved. This program and the accompanying 
 * materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at:
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors:
 *     Compuware Corporation - initial API and implementation
 */
package org.eclipse.corona;

import org.osgi.service.application.ScheduledApplication;


/**
 * The Corona ApplicationManager provides the lifecycle management
 * for Eclipse applications.
 * <p>
 * Eclipse applications are defined via <code>org.eclipse.core.runtime.applications</code>
 * extension point.
 * </p>
 * 
 * @author doflynn
 */
public interface IApplicationManager {

	/**
	 * Start an Eclipse Application
	 * <p>
	 * The application to start is defined by the <code>app</code> parameter.  The 
	 * syntax for this is:
	 * <p>
	 * &ltAppID&gt[,arg0][,arg1][,arg2]...
	 * <br>
	 * Where <i>AppID</i> is the application's ID, followed by its optional arguments.
	 * </br>
	 * </p>
	 * </p>
	 * 
	 * @param app is the application launch syntax
	 * @throws CoronaException TODO
	 */
	public void startApplication( String app ) throws CoronaException;
	
	/**
	 * Stop an Eclipse application.
	 * 
	 * @param appId of the application to stop
	 * @throws CoronaException TODO
	 */
	public void stopApplication(String appId) throws CoronaException;
	
	/**
	 * Stop all applications that were started by the ApplicationManager
	 * 
	 * @throws CoronaException
	 */
	public void stopAllApplications() throws CoronaException;
	
	/**
	 * Count the number of running applications.
	 * 
	 * @return number of running applications
	 */
	public int countRunningApplications();

	/** 
	 * Count the number of scheduled applications.
	 * 
	 * @return number of scheduled applications
	 */
	public int countScheduledApplications();
	
	/**
	 * Test if an application is running.
	 * 
	 * @param appId
	 * @return
	 */
	public boolean isApplicationRunning(String appId);
	
	/**
	 * List the set of OSGi service applications
	 * 
	 * @return
	 */
	public String[] listApplications();
	
	/**
	 * List the set of scheduled OSGi service applications
	 * 
	 * @return
	 */
	public ScheduledApplication[] listScheduledApplications();
	
	/**
	 * Shutdown Corona's runtime environment
	 * 
	 */
	public void shutdown();
	
	/**
	 * Schedule an OSGi service application
	 * 
	 * @param appId is the ID of the application to schedule
	 * @param filter is the OSGi filter used to determine when to execute
	 * @param recurring indicates if the app scheduling is recurring
	 * @throws CoronaException 
	 */
	public void scheduleApplication(String appId, String filter, boolean recurring) throws CoronaException;
}
