/*
 * Copyright (c) 2007-2008 Compuware Corporation and others.
 * 
 * All rights reserved. This program and the accompanying 
 * materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at:
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors:
 *     Compuware Corporation - initial API and implementation
 */
package org.eclipse.corona.internal;

import java.util.ArrayList;
import java.util.Dictionary;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.StringTokenizer;

import org.apache.log4j.Logger;
import org.eclipse.corona.CoronaException;
import org.eclipse.corona.IApplicationManager;
import org.eclipse.equinox.app.IApplicationContext;
import org.osgi.framework.BundleContext;
import org.osgi.framework.Bundle;
import org.osgi.framework.Filter;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.service.application.ApplicationDescriptor;
import org.osgi.service.application.ApplicationHandle;
import org.osgi.service.application.ScheduledApplication;
import org.osgi.service.component.ComponentContext;
import org.osgi.util.tracker.ServiceTracker;

/**
 * The Corona ApplicationManager provides the lifecycle management
 * for Eclipse applications.
 * <p>
 * Eclipse applications are defined via <code>org.eclipse.core.runtime.applications</code>
 * extension point.
 * </p>
 * <p>
 * The ApplicationManager is an OSGi component service.
 * </p>
 *  
 * @author doflynn
 */
public class ApplicationManager implements IApplicationManager {
	private static final String EVENT_TOPIC_APPLICATION_TIMER = "org/osgi/application/timer";

	private Logger logger = Logger.getLogger(ApplicationManager.class.getName());
	
	private BundleContext ctxBundle;
	private ServiceTracker appDescriptors;
	private ServiceTracker appHandles;

	private Filter filterRunningApp;
	private final static String FILTER_RUNNING_APP = "(!(application.state=STOPPING))"; //$NON-NLS-1$
	
	private Hashtable<String, ApplicationHandle> listStartedApps;

	/**
	 * Construct the ApplicationManager component service
	 */
	public ApplicationManager() {
		
		listStartedApps = new Hashtable<String, ApplicationHandle>();
	}
	
	/**
	 * Activate the service component.
	 *
	 * This method is called from OSGi to start the service.
	 * 
	 * @param ctxComponent
	 * @throws CoronaException 
	 */
	protected void activate(ComponentContext ctxComponent) throws CoronaException {
		logger.debug( "activate ApplicationManager");

		this.ctxBundle = ctxComponent.getBundleContext();
		
		/*
		 * create filter used to lookup running applications
		 */
		try {
			filterRunningApp = ctxBundle.createFilter(FILTER_RUNNING_APP);
		} catch (InvalidSyntaxException e) {
			String message = "Invalid filter syntax: "+FILTER_RUNNING_APP;
			logger.error( message);
			CoronaException xCorona = new CoronaException(message);
			xCorona.initCause(e);
			throw xCorona;
		}
		
		this.appDescriptors = new ServiceTracker(ctxBundle, ApplicationDescriptor.class.getName(), null);
		this.appDescriptors.open();

		this.appHandles = new ServiceTracker(ctxBundle, ApplicationHandle.class.getName(), null);
		this.appHandles.open();

	}
	
	/**
	 * Activate the service component.
	 * 
	 * This method is called from OSGi to stop the service.
	 * 
	 * @param ctxComponent
	 */
	protected void deactivate(ComponentContext ctxComponent) {		
		logger.debug( "deactivate ApplicationManager");

		this.appHandles.close();
		this.appHandles = null;
		
		this.appDescriptors.close();
		this.appDescriptors = null;
		
		this.ctxBundle = null;
	}
	
	/**
	 * @see org.eclipse.corona.IApplicationManager#startApplication(java.lang.String)
	 */
	public void startApplication(String app) throws CoronaException {
		logger.debug( "starting application: "+app);

		/*
		 * parse the application id & arguments
		 */
		StringTokenizer argTokenizer = new StringTokenizer(app, ",");
		String appId = argTokenizer.nextToken();
		ArrayList<String> argList = new ArrayList<String>();
		while( argTokenizer.hasMoreTokens() ) {
			argList.add(argTokenizer.nextToken());
		}
		String[] args = (argList.size()==0 ? null : argList.toArray(new String[argList.size()]));

		/*
		 * initialize application's launch arguments
		 */
		HashMap<String, String[]> appArgs = new HashMap<String, String[]>(1);
		if (args != null) {
			appArgs.put(IApplicationContext.APPLICATION_ARGS, args);
		}

		/*
		 * start the application
		 */
		startApplication(appId, appArgs);
	}

	/**
	 * @see org.eclipse.corona.IApplicationManager#startApplication(java.lang.String, java.util.HashMap)
	 */
	protected void startApplication(String appId, HashMap<String, String[]> appArgs) throws CoronaException {
		logger.debug( "starting application: "+appId);
		
		/*
		 * get the OSGi component service reference to the application
		 */
		ServiceReference appService = getServiceReference(appDescriptors.getServiceReferences(), appId, ApplicationDescriptor.APPLICATION_PID, false);
		if ( appService == null ) {
			String msg = "Unable to find application service component: "+appId;
			logger.warn( msg);
			throw new CoronaException(msg);
		}
		ApplicationDescriptor appDesc = ((ApplicationDescriptor) ctxBundle.getService(appService));

		/*
		 * launch the application
		 */
		try {
			ApplicationHandle handle = appDesc.launch(appArgs);
			listStartedApps.put(appId, handle);
		} catch (Throwable t) {
			String msg = "Failed to start application: "+appId;
			logger.warn( msg, t);
			CoronaException xCorona = new CoronaException(msg);
			xCorona.initCause(t);
			throw xCorona;
		}
		finally {
			ctxBundle.ungetService(appService);
		}

	}

	/**
	 * @see org.eclipse.corona.IApplicationManager#stopApplication(java.lang.String)
	 */
	public void stopApplication(String appId) throws CoronaException {
		logger.debug( "Stopping aplication: "+appId);

		/*
		 * get the ApplicationHandle of the application we started
		 */
		ApplicationHandle appStartedHandle = this.listStartedApps.get(appId);
		if ( appStartedHandle == null ) {
			String msg = "Cannot stop application that ApplicationManager did not start: "+appId;
			logger.warn( msg );
			throw new CoronaException(msg);
		}
		this.listStartedApps.remove(appId);
		
		/*
		 * Get the ApplicationHandle for the application this is running
		 */
		ApplicationHandle appRunningHandle = getRunningApplication(appId);
		if ( appRunningHandle == null ) {
			logger.info( "Unable to stop non-running application: "+appId);
		}
		else {
			if ( appStartedHandle.equals(appRunningHandle) ) {
				appRunningHandle.destroy();
			}
			else {
				logger.warn( "Unable to stop application - different instance: "+appId);
			}
		}
	}

	/**
	 * @see org.eclipse.corona.IApplicationManager#stopAllApplications()
	 */
	public void stopAllApplications() throws CoronaException {
		logger.debug( "stopping all applications");
		
		Enumeration<String> enumAppId = listStartedApps.keys();
		while (enumAppId.hasMoreElements()) {
			String appId = (String) enumAppId.nextElement();
			stopApplication(appId);			
		}
	}

	/**
	 * @see org.eclipse.corona.IApplicationManager#countRunningApplications()
	 */
	public int countRunningApplications() {
		int count = 0;

		Enumeration<String> enumAppId = listStartedApps.keys();
		while (enumAppId.hasMoreElements()) {
			String appId = (String) enumAppId.nextElement();
			
			/*
			 * get the ApplicationHandle of the application we started
			 */
			ApplicationHandle appStartedHandle = this.listStartedApps.get(appId);

			/*
			 * Get the ApplicationHandle for the application this is running
			 */
			ApplicationHandle appRunningHandle = getRunningApplication(appId);

			/*
			 * check if application we started terminated without us knowing
			 */
			if ( (appRunningHandle == null) || !appStartedHandle.equals(appRunningHandle) ) {
				this.listStartedApps.remove(appId);
				continue;
			}
			
			count++;
		}
		
		return count;
	}

	/**
	 * @see org.eclipse.corona.IApplicationManager#countScheduledApplications()
	 */
	public int countScheduledApplications() {
		logger.warn("Not Yet Implemented: countScheduledApplications()");
		return 0;
	}

	/**
	 * @see org.eclipse.corona.IApplicationManager#isApplicationRunning(java.lang.String)
	 */
	public boolean isApplicationRunning(String appId) {
		boolean result = false;
		
		/*
		 * get the ApplicationHandle of the application we started
		 */
		ApplicationHandle appStartedHandle = this.listStartedApps.get(appId);		
		if ( appStartedHandle != null ) {
			
			/*
			 * Get the ApplicationHandle for the application this is running
			 */
			ApplicationHandle appRunningHandle = getRunningApplication(appId);

			/*
			 * check if application we started terminated without us knowing
			 */
			if ( (appRunningHandle != null) || appStartedHandle.equals(appRunningHandle) ) {
				result = true;
			}
			else {
				this.listStartedApps.remove(appId);
			}
		}

		return result;
	}

	/**
	 * Get the ApplicationHandle for an application from the list of applications
	 * that are registered with Eclipse.
	 * 
	 * @param appId of the application to find
	 * @return Application's handle, or <code>null</code>
	 */
	private ApplicationHandle getRunningApplication(String appId){
		ApplicationHandle appHandle = null;
		
		/*
		 * get the ServiceReference to the application
		 */
		ServiceReference appReference = getServiceReference(appHandles.getServiceReferences(), appId, ApplicationHandle.APPLICATION_PID, false);
		if ( appReference == null ){
			appReference = getServiceReference(appHandles.getServiceReferences(), appId, ApplicationHandle.APPLICATION_DESCRIPTOR, false);
		}
		
		if ( appReference == null ) {
			logger.info( "unable to find running application: "+appId);
		}
		else {
			if (filterRunningApp.match( getProperties(appReference) )) {
				try {
					appHandle = (ApplicationHandle) ctxBundle.getService(appReference);
				} finally {
					ctxBundle.ungetService(appReference);
				}
			}
		}

		return appHandle;
	}
	
	/**
	 * Get a specific ServiceRefence from an array of ServiceReferce's
	 * 
	 * Note: this code was based upon...
	 * see org.eclipse.equinox.internal.app.AppCommands
	 */
	private ServiceReference getServiceReference(ServiceReference[] srvRefs, String targetId, String idKey, boolean perfectMatch) {
		if (srvRefs == null || targetId == null) {
			return null;
		}

		ServiceReference result = null;
		boolean ambigous = false;
		for (int i = 0; i < srvRefs.length; i++) {
			String id = (String) srvRefs[i].getProperty(idKey);
			if (targetId.equals(id)) {
				return srvRefs[i]; // always return a perfect match
			}
			if (perfectMatch) {
				continue;
			}
			if (id.indexOf(targetId) >= 0) {
				if (result != null)
					ambigous = true;
				result = srvRefs[i];
			}
		}
		
		return ambigous ? null : result;
	}

	/**
	 * The the properties of a specific ServiceReference
	 * 
	 * @param ref
	 * @return
	 */
	public Dictionary<String, Object> getProperties(ServiceReference ref) {
		String[] keys = ref.getPropertyKeys();
		Hashtable<String, Object> props = new Hashtable<String, Object>(keys.length);
		for (int i = 0; i < keys.length; i++) {
			props.put(keys[i], ref.getProperty(keys[i]));
		}
		return props;
	}

	/**
	 * @see IApplicationManager#listApplications()
	 */
	public String[] listApplications() {
		
		String[] apps = null;
		
		ServiceReference[] srvRefs = this.appDescriptors.getServiceReferences();
		if ( (srvRefs!= null) &&(srvRefs.length>0) ) {
			apps = new String[srvRefs.length];
			for (int i = 0; i < srvRefs.length; i++) {
				apps[i] = (String) srvRefs[i].getProperty(ApplicationDescriptor.APPLICATION_PID);
			}
		}

		return apps;
	}

	/**
	 * @see IApplicationManager#listScheduledApplications()
	 */
	public ScheduledApplication[] listScheduledApplications() {
		ScheduledApplication[] apps = null;
		ServiceReference[] srvRefs = null;
		
		try {
			srvRefs = this.ctxBundle.getAllServiceReferences(org.osgi.service.application.ScheduledApplication.class.getName(), null);
		} catch (InvalidSyntaxException e) {
			logger.warn(e);
		}

		if ( (srvRefs!= null) &&(srvRefs.length>0) ) {
			apps = new ScheduledApplication[srvRefs.length];
			for (int i = 0; i < srvRefs.length; i++) {
				apps[i] = (ScheduledApplication) (ctxBundle.getService(srvRefs[i]));
				
				ctxBundle.ungetService(srvRefs[i]);
			}
		}

		return apps;
	}

	/**
	 * Get the OSGi SystemBundle and shutdown the environment
	 * 
	 * @see IApplicationManager#shutdown()
	 */
	public void shutdown() {
		logger.warn("shutting down Corona runtimer environment");

		try {
			Bundle sysBundle = this.ctxBundle.getBundle(0);
			sysBundle.stop();
		} catch (Throwable t) {
			logger.error(t);
		}
		
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.corona.IApplicationManager#scheduleApplication(java.lang.String, java.lang.String, boolean)
	 */
	public void scheduleApplication(String appId, String filter,
			boolean recurring) throws CoronaException {
		/*
		 * get the OSGi component service reference to the application
		 */
		ServiceReference appService = getServiceReference(appDescriptors.getServiceReferences(), appId, ApplicationDescriptor.APPLICATION_PID, false);
		if ( appService == null ) {
			String msg = "Unable to find application service component: "+appId;
			logger.warn( msg);
			throw new CoronaException(msg);
		}
		ApplicationDescriptor appDesc = ((ApplicationDescriptor) ctxBundle.getService(appService));
		
		/*
		 * skip the application is already scheduled
		 */
		ScheduledApplication[] schedApps = listScheduledApplications();
		for (int i = 0; i < schedApps.length; i++) {
			String schedAppId = schedApps[i].getApplicationDescriptor().getApplicationId();
			if ( schedAppId.equals(appId) ) {
				if ( schedApps[i].getEventFilter().equals(filter) ) {
					return;
				}
			}
		}
		
		/*
		 * schedule the application
		 */
		try {
			appDesc.schedule(null, null, EVENT_TOPIC_APPLICATION_TIMER, filter, recurring);
		} catch (Throwable t) {
			String msg = "Failed to schedule application: "+appId;
			logger.warn( msg, t);
			CoronaException xCorona = new CoronaException(msg);
			xCorona.initCause(t);
			throw xCorona;
		}
		finally {
			ctxBundle.ungetService(appService);
		}
	}	
	
	
}
