/*
 * Copyright (c) 2007-2008 Compuware Corporation and others.
 * 
 * All rights reserved. This program and the accompanying 
 * materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at:
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors:
 *     Compuware Corporation - initial API and implementation
 */
package org.eclipse.corona.internal.test;

import java.util.Date;
import java.util.Hashtable;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.corona.test.ITestEvent;
import org.eclipse.corona.test.ITestManager;
import org.eclipse.corona.test.ITestSuite;
import org.eclipse.corona.test.ITestSummary;
import org.eclipse.osgi.framework.console.CommandInterpreter;
import org.eclipse.osgi.framework.console.CommandProvider;
import org.osgi.framework.BundleContext;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.event.Event;
import org.osgi.service.event.EventConstants;
import org.osgi.service.event.EventHandler;

/**
 * TestCommands - OSGi console commands to interact with Corona Test Framework.
 * 
 * @author doflynn
 */
public class TestCommands implements CommandProvider, EventHandler {

	private ServiceRegistration srvRegEventHandler;
		
	/*
	 * help for test commands
	 */
	private static String help = "---Corona Test Framework---\n"
			+ "\t testList - list all of test suites\n"
			+ "\t testAll - run ALL test suites\n"
			+ "\t testSuite <name> - run test suite 'name'\n";

	private BundleContext ctxBundle;

	private CommandInterpreter cmdInterpreter;

	/**
	 * Default CTOR used by Declarative Services
	 */
	public TestCommands() {

	}

	/**
	 * Activate the service component
	 * 
	 * This method is called from OSGi to start the service.
	 * 
	 * @param ctxComponent
	 */
	protected void activate(ComponentContext ctxComponent) {
		this.ctxBundle = ctxComponent.getBundleContext();
		
		 String[] topics = new String[] {EventConstants.EVENT_TOPIC, ITestEvent.TOPIC_ROOT+"/*"};
		 Hashtable<String, String[]> ht = new Hashtable<String, String[]>();
		 ht.put(EventConstants.EVENT_TOPIC, topics);
		 srvRegEventHandler = ctxBundle.registerService(EventHandler.class.getName(), this, ht);
 
	}

	/**
	 * Activate the service component.
	 * 
	 * This method is called from OSGi to stop the service.
	 * 
	 * @param ctxComponent
	 */
	protected void deactivate(ComponentContext ctxComponent) {
		this.srvRegEventHandler.unregister();
		this.srvRegEventHandler = null;
		
		this.ctxBundle = null;
	}

	/**
	 * @see CommandProvider#getHelp()
	 */
	public String getHelp() {
		return help;
	}

	/**
	 * OSGi Command - testAll
	 * 
	 * @param ci
	 */
	public void _testAll(CommandInterpreter ci) {
		ci.println("Running ALL active CTF test suites...");

		ITestSummary[] testSummaries = null;

		try {
			/*
			 * get the TestManager service
			 */
			ServiceReference srvRef = this.ctxBundle
					.getServiceReference(ITestManager.class.getName());

			if (srvRef != null) {
				ITestManager testManager = (ITestManager) this.ctxBundle
						.getService(srvRef);

				/*
				 * set the class's ci so TestEvent handler can
				 * echo testing progress
				 */
				this.cmdInterpreter = ci;
					
				ITestSuite[] testSuites = testManager.testSuites();
				for (int i = 0; i < testSuites.length; i++) {
					ITestSummary testSummary = testManager.runTestSuite(testSuites[i]);
					if ( testSummary != null ) {
						printTestSummary(ci, testSummary);							
					} else {				
						ci.println("Warning: not test summary results to report");	
						break;
					} 
				}
				
				/*
				 * get TestEvent chance to process
				 */
				Thread.sleep(2*1000); // give chance 
				this.cmdInterpreter = null;	// reset ci
				
				this.ctxBundle.ungetService(srvRef);
			}
			else {
				ci.println("Warning: unable to locate TestManager service");
			}
		} catch (Exception e) {
			ci.printStackTrace(e);
		}		
	}

	/**
	 * Run a specific test suite
	 * 
	 * @param ci
	 */
	public void _testSuite(CommandInterpreter ci) {
		ci.println("Running test suite...");
		String name = ci.nextArgument();
		if (name == null) {
			ci.println("\tERROR: test suite name not defined");
			return;
		}
		
		ITestSummary testSummary = null;

		try {
			/*
			 * get the TestManager service
			 */
			ServiceReference srvRef = this.ctxBundle
					.getServiceReference(ITestManager.class.getName());

			if (srvRef != null) {
				ITestManager testManager = (ITestManager) this.ctxBundle
						.getService(srvRef);

				/*
				 * set the class's ci so TestEvent handler can
				 * echo testing progress
				 */
				this.cmdInterpreter = ci;

				/*
				 * find the test suite to run
				 */
				ITestSuite[] testSuites = testManager.testSuites();
				for (int i = 0; i < testSuites.length; i++) {
					if ( testSuites[i].getName().equals(name) ) {

						/*
						 * found test suite - now run it
						 */
						testSummary = testManager.runTestSuite(testSuites[i]);

						break;
					}
				}

				/*
				 * get TestEvent chance to process
				 */
				Thread.sleep(2*1000); // give chance 
				this.cmdInterpreter = null;	// reset ci
				
				this.ctxBundle.ungetService(srvRef);
			}
			else {
				ci.println("Warning: unable to locate TestManager service");
			}
		} catch (Exception e) {
			ci.printStackTrace(e);
		}
		
		/*
		 * display test result summary
		 */
		if (testSummary != null) {
			printTestSummary(ci, testSummary);
		}
		else {
			ci.println("Warning: not test summary results to report");
		}
	}

	/**
	 * Print the test summary results
	 */
	private void printTestSummary(CommandInterpreter ci, ITestSummary testSum) {
		ci.println(testSum.toString());
	}

	/**
	 * OSGi Command - testList
	 * 
	 * @param ci
	 * @throws InvalidSyntaxException
	 * @throws CoreException
	 */
	public void _testList(CommandInterpreter ci) throws InvalidSyntaxException,
			CoreException {
		ci.println("Listing all test suites...");
		ITestSuite[] testSuites = null;
		
		try {
			/*
			 * get the TestManager service
			 */
			ServiceReference srvRef = this.ctxBundle
					.getServiceReference(ITestManager.class.getName());

			if (srvRef != null) {
				ITestManager testManager = (ITestManager) this.ctxBundle
						.getService(srvRef);

				testSuites = testManager.testSuites();
				this.ctxBundle.ungetService(srvRef);
			}
		} catch (Exception e) {
			ci.printStackTrace(e);
		}
		
		if ( testSuites != null ) {
			for (int i = 0; i < testSuites.length; i++) {
				ci.println("\tITestSuite: "+testSuites[i].getName());
				ci.println("\t\t  type: "+testSuites[i].getRunnerType());
				ci.println("\t\t level: "+testSuites[i].getLevel());
			}
		}
	}

	public synchronized void handleEvent(Event event) {

		/*
		 * if we have a CommandInterpreter, echo ITestEvent to console 
		 */
		if ( this.cmdInterpreter != null ) {
			String state = (String)event.getProperty(ITestEvent.EVENT);
			ITestSuite testSuite = (ITestSuite)event.getProperty(ITestEvent.TEST_SUITE);

			String testCase = (String)event.getProperty(ITestEvent.TEST_CASE);
			if ( testCase == null ) {
				testCase = "";
			}

			String status = (String)event.getProperty(ITestEvent.TEST_STATUS);
			if ( status == null ) {
				status = "unknown";
			}
			
			Date date = new Date();
		    
			this.cmdInterpreter.println("...testing progress:");			
			this.cmdInterpreter.println("\t      Time: "+ new Date() );
			this.cmdInterpreter.println("\t TestSuite: "+testSuite.getName() );
			this.cmdInterpreter.println("\t  TestCase: "+testCase );
			this.cmdInterpreter.println("\t     State: "+state );
			this.cmdInterpreter.println("\t    Status: "+status );			
		}
	}

}
