/*
 * Copyright (c) 2007-2008 Compuware Corporation and others.
 * 
 * All rights reserved. This program and the accompanying 
 * materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at:
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors:
 *     Compuware Corporation - initial API and implementation
 */
package org.eclipse.corona.test;

import java.util.ArrayList;
import java.util.Dictionary;
import java.util.Enumeration;
import java.util.Properties;

import org.osgi.service.component.ComponentContext;

/**
 * AbstractTestSuite OSGi service component for Corona Kernel.
 * <p>
 * This class is an OSGi service component.  It provides
 * the implementation of the ITestSuite service interface used
 * by the Corona Test Framework.
 * </p>
 * 
 */
public abstract class AbstractTestSuite implements ITestSuite {

	private ArrayList<Object> listTestCases;
	protected Properties properties;
	protected boolean valid = true;
	protected Throwable xception;
	protected String contributorLabel;
	
	protected AbstractTestSuite() {
		listTestCases = new ArrayList<Object>();
		this.properties = new Properties();
	}

	/**
	 * Activate the OSGi service component.
	 * 
	 * This method is invoked by OSGi when this service
	 * component is first activated.
	 * 
	 * @param context
	 */
	protected void activate(ComponentContext context) {
		Dictionary<String, String> dict = context.getProperties();
		Enumeration<String> enumProperties = context.getProperties().elements();
		while (enumProperties.hasMoreElements()) {
			String name = (String) enumProperties.nextElement();
			String value = (String) dict.get(name);

			this.properties.setProperty(name, value );
		}
	}

	/**
	 * Deactivate the OSGi service component.
	 * 
	 * This method is invoked by OSGi when this service
	 * component is no longer needed.
	 * 
	 * @param context
	 */
	protected void deactivate(ComponentContext context) {
	}

	public boolean isValid() {
		return this.valid;
	}
	
	/**
	 * Add a test case to the test suite
	 * 
	 * @param testCase
	 */
	protected void addTestCase(Class testCase){
		this.listTestCases.add(testCase);
	}

	/**
	 * Add a test case to the test suite
	 * 
	 * @param testCase
	 */
	protected void addTestCase(Object testCase){
		this.listTestCases.add(testCase);
	}
	
	public Object[] listTestCases() {
		
		return (Object[]) this.listTestCases.toArray(new Object[this.listTestCases.size()]);
	}

	public String getProperty(String name, String defaultValue) {
		String value = (String) this.properties.get(name);
		if ( value == null ) {
			value = defaultValue;
		}
		return value;		
	}
	
	public void setName(String value) {
		this.properties.setProperty(PROPERTY_NAME, value );
	}
	
	public String getName() {
		String name = getProperty(PROPERTY_NAME, DEFAULT_NAME);
		
		return name;
	}

	public void setLevel(int value) {
		this.setLevel(Integer.toString(value));
	}
	public void setLevel(String value) {
		if ( value == null ) {
			value = DEFAULT_LEVEL;
		}
		this.properties.setProperty(PROPERTY_LEVEL, value );
	}
	
	public int getLevel() {
		String level = getProperty(PROPERTY_LEVEL, DEFAULT_LEVEL);
		return Integer.parseInt(level);
	}

	public int compareTo(ITestSuite testSuite) {
		int compare = 0;
		
		if ( this.getLevel() < testSuite.getLevel() ) {
			compare = -1;
		}
		else if ( this.getLevel() > testSuite.getLevel() ) {
			compare = 1;
		}
		
		return compare;
	}

	public String getRunnerType() {
		String type = getProperty(PROPERTY_RUNNER, DEFAULT_RUNNER);
		
		return type;
	}

	public void setRunnerType(String type) {
		this.properties.setProperty(PROPERTY_RUNNER, type);
	}

	public Throwable getException() {
		return this.xception;
	}
	
	public void setContributorLabel(String contributorLabel){
		this.contributorLabel = contributorLabel;
	}
	
	public String getContributorLabel(){
		return this.contributorLabel;
	}

}
