/*
 * Copyright (c) 2007-2008 Compuware Corporation and others.
 * 
 * All rights reserved. This program and the accompanying 
 * materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at:
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors:
 *     Compuware Corporation - initial API and implementation
 */
package org.eclipse.corona.test;

import java.util.List;

import org.apache.tools.ant.taskdefs.optional.junit.JUnitTestRunner;

/**
 * ITestResult - results from a specific set of tests
 *
 */
public interface ITestResult {
	
	// Status unknown	
	public static final int UNKNOWN = -1;
	
	// No problems with this test case.     
    public static final int SUCCESS = JUnitTestRunner.SUCCESS;
    
    // Some tests failed.    
    public static final int FAILURES = JUnitTestRunner.FAILURES;

    // An error occurred during execution of test cases.     
    public static final int ERRORS = JUnitTestRunner.ERRORS;
   
    // Test cases in progress.     
    public static final int EXECUTING = 4;
	
	// Test suite execution status -1: UNKNOWN, 0: SUCCESS, 1: FAILURES, 2: ERRORS, 3: EXECUTING	
	public enum TestSuiteExecutionStatus {
		UNKNOWN(ITestResult.UNKNOWN, "unknown"), 
		SUCCESS(ITestResult.SUCCESS, "success"), 
		FAILURES(ITestResult.FAILURES, "failures"), 
		ERRORS(ITestResult.ERRORS, "errors"), 
		EXECUTING(ITestResult.EXECUTING, "executing");

		private final int value;
		private final String label;
		
		TestSuiteExecutionStatus(int value, String label) {
			this.value = value;	
			this.label = label;
		}

		public int value() {
			return value;
		}	
		
		public String label() {
			return label;
		}
		
		public static TestSuiteExecutionStatus getTestSuiteExecutionStatus(int testSuiteExecutionStatusCode){
			switch (testSuiteExecutionStatusCode) {
			case ITestResult.UNKNOWN:
				return TestSuiteExecutionStatus.UNKNOWN;
			case ITestResult.SUCCESS:
				return TestSuiteExecutionStatus.SUCCESS;
			case ITestResult.FAILURES:
				return TestSuiteExecutionStatus.FAILURES;
			case ITestResult.ERRORS:
				return TestSuiteExecutionStatus.ERRORS;
			case ITestResult.EXECUTING:
				return TestSuiteExecutionStatus.EXECUTING;
			default:
				return TestSuiteExecutionStatus.UNKNOWN;
			}
		}
	}	
	
	
   	
	/**
	 * Get the count of tests that were run.
	 * 
	 * @return
	 */
	public long getRunCount();
	
	/**
	 * Get the count of failed tests
	 * 
	 * @return
	 */
	public long getFailureCount();
	
	/**
	 * Get the elapsed time it took to run all tests
	 * 
	 * @return
	 */
	public long getRunTime();
	
	/**
	 * Get the count of tests that were ignored
	 * 
	 * @return
	 */
	public long getIgnoreCount();
	
	/**
	 * Get the count of tests causing error
	 * 
	 * @return
	 */
	public long getErrorCount();
	
	
	
	/**
	 * int indicator to determine CTF Test Suites execution status.  
	 * 
	 * @return A global status from all combined active CTF Test Suites result. 
	 */
	public TestSuiteExecutionStatus getCTFTestSuiteExecutionStatus();
	
	/**
	 * Sets the indicator that determine CTF Test Suites execution status.  
	 * 
	 * @param A global status for all combined active CTF Test Suites result.  
	 */
	public void setCTFTestSuiteExecutionStatus(TestSuiteExecutionStatus ctfTestSuiteExecutionStatus);
		
	
	/**
	 * Get the test suite associated with these results
	 * 
	 * @return test suite
	 */
	public ITestSuite getTestSuite();
	
	/**
	 * Adds a Test Suite execution status to the list of statuses.  
	 * 
	 * @param A status for for the Test Suite execution.  
	 */
	public void addTestSuiteExecutionStatus(TestSuiteExecutionStatus testSuiteExecutionStatus);
	
	/**
	 * Returns the list of execution statuses.  
	 * 
	 * @return The list of execution statuses.  
	 */
	public List<TestSuiteExecutionStatus> getTestSuiteExecutionStatuses();
	
	

}
