/*
 * Copyright (c) 2007-2008 Compuware Corporation and others.
 * 
 * All rights reserved. This program and the accompanying 
 * materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at:
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors:
 *     Compuware Corporation - initial API and implementation
 */
package org.eclipse.corona.test;

import java.util.Date;

import org.eclipse.corona.test.ITestResult;
import org.eclipse.corona.test.ITestSuite;
import org.eclipse.corona.test.ITestSummary;

/**
 * TestSummary
 * </p>
 * A <i>TestSummary</i> is associated to a <i>TestSuite</i>.  It provides
 * a summary of all of the <i>TestResults</i>.
 * 
 * @author doflynn
 */
public class TestSummary implements ITestSummary {

	private long tsRunStart;
	private long tsRunStop;
	
	private String[] output;
	private ITestSuite testSuite;

	private ITestResult testResult;
	
	public TestSummary() {
		this.start();
	}
	
	public void initialize(ITestSuite testSuite, ITestResult testResult, String[] output ) {
		this.testSuite = testSuite;
		this.output = output;
		
		this.testResult = testResult;
	}
	
	public void start() {
		this.tsRunStart = System.currentTimeMillis();
	}
	
	public void stop() {
		this.tsRunStop = System.currentTimeMillis();
	}
	
	public long getRunStartTime() {
		return this.tsRunStart;
	}

	public long getRunStopTime() {
		return this.tsRunStop;
	}

	public String[] getOutput() {
		return this.output;
	}

	public String getName() {
		return this.testSuite.getName();
	}
	
	public int getLevel() {
		return this.testSuite.getLevel();
	}

	public ITestSuite getTestSuite() {
		return this.testSuite;
	}

	public ITestResult getTestResult() {
		return this.testResult;
	}

	@Override
	public String toString() {
		StringBuffer strbuf = new StringBuffer();
		
		strbuf.append("\n\tCTF Test Suite: "+ this.getName() +"\n");
		strbuf.append("\n");

		/*
		 * display test times
		 */
		strbuf.append("\tTime\n");
		long startTime = this.getRunStartTime();
		long stopTime = this.getRunStopTime();
		double elapsed = ((double) stopTime - (double) startTime) / 1000.0;
		strbuf.append("\t\t   Start: " + new Date(startTime)+"\n");
		strbuf.append("\t\t    Stop: " + new Date(stopTime)+"\n");
		strbuf.append("\t\t Elapsed: " + elapsed + " (seconds)\n");

		ITestResult result = this.getTestResult();

		/*
		 * display test counts
		 */
		strbuf.append("\tCounts\n");
		strbuf.append("\t\t Tests count : " + result.getRunCount()+"\n");
		strbuf.append("\t\t Success     : " + (result.getRunCount() - result.getErrorCount() - result.getFailureCount()) +"\n");
		strbuf.append("\t\t Failures    : " + result.getFailureCount()+"\n");
		strbuf.append("\t\t Errors      : " + result.getErrorCount()+"\n");

		strbuf.append("\tResult: " + result.getCTFTestSuiteExecutionStatus().label() + "\n");

		//TODO: Try to get failures using a Listner Wrapper on the Test Runner (or another approach).
		/*
		 * display any test failures
		 */
		/*ITestFailure[] testFailures = result.listFailures();
		for (int i = 0; i < testFailures.length; i++) {
			strbuf.append("\tTest Failure\n");

			ITestFailure testFailure = testFailures[i];

			strbuf.append("\t\t Name: " + testFailure.getTestDescription() +"\n");

			strbuf.append("\t\t Message: " + testFailure.getMessage()+"\n");

			/*
			 * print only the first 3 lines of trace...
			 *//*
			StringBuffer strbufException = new StringBuffer();
			String trace = testFailure.getTrace();
			StringTokenizer strtok = new StringTokenizer(trace, "\n");
			for (int j = 0; (j < 3) && (strtok.hasMoreTokens()); j++) {
				String text = strtok.nextToken();
				strbufException.append(text);
				strbufException.append("\t\t\t");
			}
			strbuf.append("\t\t Exception: "+strbufException.toString()+"\n");
		}*/

		strbuf.append(getTestMergedOutput(this.getOutput()));
	
		return strbuf.toString();
	}
	
	/*
	 * display test merged output (from sysout and syserr)
	 */
	public static String getTestMergedOutput(String[] output){
		StringBuffer strbuf = new StringBuffer();
		
		if (output != null) {
			strbuf.append("\tTest System Output...\n");
			for (int i = 0; i < output.length; i++) {
				strbuf.append("\t" + output[i]+"\n");
			}
		}
		
		return strbuf.toString();
	}
}
