/*
 * Copyright (c) 2007-2008 Compuware Corporation and others.
 * 
 * All rights reserved. This program and the accompanying 
 * materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at:
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors:
 *     Compuware Corporation - initial API and implementation
 */
package org.eclipse.corona.test;

import java.io.PrintStream;

import org.apache.log4j.Logger;
import org.eclipse.corona.internal.test.Activator;
import org.eclipse.corona.test.ITestRunner;
import org.eclipse.corona.test.ITestSuite;
import org.eclipse.corona.test.ITestSummary;
import org.eclipse.corona.test.TestMergedOutput;
import org.eclipse.corona.test.TestOutput;

/**
 * Corona Test Framework abstract test runner
 *
 */
public abstract class AbstractTestRunner implements ITestRunner {

	private Logger logger = Logger.getLogger(AbstractTestRunner.class.getName());
	
	private String name;
	private String type;

	public AbstractTestRunner() {
	}

	public String getName() {
		return this.name;
	}

	public String getType() {
		return this.type;
	}

	/**
	 * Run a specific test suite.
	 * 
	 * @return test results summary
	 */
	public ITestSummary runTestSuite(ITestSuite testSuite) {
		logger.info( "entering runTestSuite()" );
		
		/*
		 * save System out/err state
		 */
		PrintStream sysOut = System.out;
		PrintStream sysErr = System.err;
		
		ITestSummary testSummary = new TestSummary();
		TestMergedOutput mergedOutput = new TestMergedOutput();
		try {
			
			/*
			 * capture any system output created by the test cases
			 */
			TestOutput testOut = new TestOutput("sysout", mergedOutput);
			System.setOut( new PrintStream(testOut) );
			TestOutput testErr = new TestOutput("syserr", mergedOutput);
			System.setErr( new PrintStream(testErr) );

			/*
			 * run the test cases
			 */
			ITestResult testResult = null;
			if ( testSuite.isValid() ) {
				testResult = runTestCases(testSuite);
			}
			else {
				testResult = TestResultsBuilder.createTestResult(testSuite);
			}

			/*
			 * create test summary
			 */
			testSummary.initialize(testSuite, testResult, mergedOutput.getOutput() );
			testSummary.stop();
			
		} catch (Exception e) {
			String mergedOutputStr = TestSummary.getTestMergedOutput(mergedOutput.getOutput());
			/*
			 * restore System out/err
			 */
			System.setOut(sysOut);
			System.setErr(sysErr);
			
			logger.error( "Error running JUnit for ITestSuite: "+testSuite.getName(),e);
			logger.error( mergedOutputStr);			
			
			testSummary = null;
			
		} finally {
			/*
			 * restore System out/err
			 */
			System.setOut(sysOut);
			System.setErr(sysErr);
		}

		logger.info( "exiting runTestSuite()" );
		return testSummary;
	}

	/**
	 * Run the test cases for the specific test runner
	 * @param testSuite TODO
	 */
	protected abstract ITestResult runTestCases(ITestSuite testSuite) throws Exception;

	public void setName(String name) {
		this.name = name;
	}

	public void setType(String type) {
		this.type = type;
	}
}
