/*
 * Copyright (c) 2007-2008 Compuware Corporation and others.
 * 
 * All rights reserved. This program and the accompanying 
 * materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at:
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors:
 *     Compuware Corporation - initial API and implementation
 */
package org.eclipse.corona.test;

import java.util.Hashtable;

import org.apache.log4j.Logger;
import org.eclipse.equinox.app.IApplicationContext;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.event.Event;
import org.osgi.service.event.EventConstants;
import org.osgi.service.event.EventHandler;
import org.osgi.util.tracker.ServiceTracker;

import org.eclipse.corona.AbstractServiceApplication;
import org.eclipse.corona.IApplicationManager;
import org.eclipse.corona.internal.test.Activator;

/**
 * Corona Test Framework OSGI application.
 * 
 */
public class Application extends AbstractServiceApplication implements EventHandler {
	private Logger logger = Logger.getLogger(Application.class);
	
	private transient IApplicationContext ctxApplication;
	private transient BundleContext ctxBundle;
	private transient boolean stopRequest = false;
	private boolean consoleDisplay = false;

	/**
	 * Exit codes
	 */
	public static final Integer TEST_OK = EXIT_OK;
	public static final Integer TEST_ERR_TEST_MANAGER = new Integer(-200);
	public static final Integer TEST_ERR_TEST_SUMMARY = new Integer(-201);
	public static final Integer TEST_ERR_DOC_BUILDER = new Integer(-202);
	public static final Integer TEST_ERR_SAVE_RESULTS = new Integer(-203);
	
	/**
	 * Create the Corona Test Framework application.
	 * 
	 */
	public Application() {
	}
	
	/**
	 * Start the OSGi application
	 * 
	 */
	public Object start(IApplicationContext context) throws Exception {
		
		logger.info( "starting test application");
				
		this.ctxApplication = context;
		this.ctxBundle = Activator.getDefault().getBundleContext();

		super.start(ctxBundle, context, Application.class.getName());

		/*
		 * register the TestEvent handler
		 */
		String[] topics = new String[] {EventConstants.EVENT_TOPIC, ITestEvent.TOPIC_ROOT+"/*"};
		Hashtable<String, String[]> ht = new Hashtable<String, String[]>();
		ht.put(EventConstants.EVENT_TOPIC, topics);
		ServiceRegistration srvRegEventHandler = ctxBundle.registerService(EventHandler.class.getName(), this, ht);

		ctxApplication.applicationRunning();
		
		Integer rc = new Integer(-51);
		try {
			rc = testAll();
		}
		catch(Throwable t) {
			logger.warn( "Unable to performa all test suites", t);
		}

		/*
		 * get chance for TestEvent handling
		 */
		Thread.sleep(5*1000);
		
		/*
		 * unregister TestEvent handler
		 */
		srvRegEventHandler.unregister();

		shutdown();
		
		logger.debug( "exiting test application");
		return rc;
	}

	/**
	 * Stop the OSGi application
	 */
	public void stop() {
		logger.debug( "stopping test application");
		this.stopRequest = true;
	}

	/**
	 * Shutdown the Corona test environment
	 * 
	 * see https://bugs.eclipse.org/bugs/show_bug.cgi?id=279025
	 */
	private void shutdown() {
		String testShutdown = System.getProperty("corona.test.shutdown");
		try {			
			if ( testShutdown != null ) {
				Boolean flag = new Boolean(testShutdown);
				if ( flag.booleanValue() ) {
					
					/*
					 * get Corona's Service Application Manager
					 */
					ServiceReference srvRef = ctxBundle.getServiceReference(IApplicationManager.class.getName());
					IApplicationManager appManager = (IApplicationManager)ctxBundle.getService(srvRef);
					appManager.shutdown();
					
					/*
					 * at this point, the system should be shutting down
					 * ... but let's do our clean-up anyway
					 */
					ctxBundle.ungetService(srvRef);
				}
			}
		}
		catch(Throwable t) {
			logger.error("error trying to shutdown test environment", t);
		}
		
	}

	/**
	 * OSGi Command - testAll
	 * 
	 * @param ci
	 */
	public Integer testAll() {
		logger.info( "Running ALL active CTF test suites...");
		
		ITestManager testManager = getTestManager();
		if ( testManager == null ) {
			logger.warn( "Warning: unable to locate TestManager");
			return TEST_ERR_TEST_MANAGER;
		}

		try {
			/*
			 * get the TestManager service
			 */
		
			ITestSuite[] testSuites = testManager.testSuites();
			for (int i = 0; i < testSuites.length; i++) {
				ITestSummary testSummary = testManager.runTestSuite(testSuites[i]);
				if ( testSummary != null ) {
					displayTestSummary(testSummary);
				} else {				
					logger.warn( "Warning: no TestSummary results");
					return TEST_ERR_TEST_SUMMARY;
				} 
				
				if ( this.stopRequest ) {
					logger.warn( "Execution of test suites has been interrupted");
					break;
				}
			}
		} catch (Exception e) {
			logger.warn( "ERROR: "+e.getMessage(), e);
			return TEST_ERR_TEST_MANAGER;
		}
		
		return TEST_OK;
	}

	/**
	 * Get the Corona Test Framework TestManager service 
	 * 
	 * @return ITestManager service object, or <i>null</i> if one could
	 * not be found.
	 */
	private ITestManager getTestManager() {
		/*
		 * open ServiceTracker to find ITestManager
		 */
		ServiceTracker srvTracker = new ServiceTracker( this.ctxBundle, ITestManager.class.getName(), null );
		srvTracker.open();
		
		/*
		 * if service not available, wait a little while for it to be registered
		 */
		ITestManager testManager = null;
		try {
			testManager = (ITestManager) srvTracker.waitForService( 1000*15 );
		} catch (Exception e) {
			logger.warn( "Unable to get ITestManager service", e);
		}
		
		srvTracker.close();
		
		return testManager;
	}

	/**
	 * Display test result summary
	 * 
	 * @param testSummary
	 */
	private void displayTestSummary(ITestSummary testSummary) {
		logger.info( testSummary.toString());
	}

	public void handleEvent(Event event) {
		StringBuffer strbuf = new StringBuffer();
		
		strbuf.append("Test Progress Event: ");
	
		String name;
		
		name = (String)event.getProperty(ITestEvent.EVENT);
		strbuf.append("state="+name+" ");

		name = (String)event.getProperty(ITestEvent.TEST_STATUS);
		name = (name == null ? "unknown" : name);
		strbuf.append("status="+name+" ");
		
		ITestSuite testSuite = (ITestSuite)event.getProperty(ITestEvent.TEST_SUITE);
		name = testSuite.getName();
		strbuf.append("suite="+name+" ");
		
		String testCase = (String)event.getProperty(ITestEvent.TEST_CASE);
		name = (testCase == null ? "" : testCase );
		strbuf.append("case="+name+" ");
	
		logger.info( strbuf.toString());
	}
}
