/*
 * Copyright (c) 2007-2008 Compuware Corporation and others.
 * 
 * All rights reserved. This program and the accompanying 
 * materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at:
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors:
 *     Compuware Corporation - initial API and implementation
 */
package org.eclipse.corona.internal.diagnostic;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

import org.osgi.framework.BundleContext;
import org.osgi.framework.Filter;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.service.component.ComponentContext;
import org.w3c.dom.Document;
import org.apache.log4j.Logger;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.corona.diagnostic.DiagnosticException;
import org.eclipse.corona.diagnostic.IDiagnosticData;
import org.eclipse.corona.diagnostic.IDiagnosticManager;

public class DiagnosticManager implements IDiagnosticManager {

	private static final String ATTR_VERSION = "version";

	private static final String ATTR_NAME = "name";

	private Logger logger = Logger.getLogger(DiagnosticManager.class.getName());
	
	private BundleContext ctxBundle;
	private Document docDiag;
	private StringBuffer manifestEntries;

	/**
	 * Default CTOR used by Declarative Services
	 */
	public DiagnosticManager() {
		manifestEntries = new StringBuffer();
	}

	/**
	 * Activate the service component
	 * 
	 * @param ctxComponent
	 */
	protected void activate(ComponentContext ctxComponent) {
		this.ctxBundle = ctxComponent.getBundleContext();
	}

	public String[] listDiagnosticData() {
		ArrayList<String> listNames = new ArrayList<String>();

		/*
		 * get all ext point registered diagnostic data components
		 */
		IExtensionPoint extPoint = Platform.getExtensionRegistry()
				.getExtensionPoint(IDiagnosticData.EXT_POINT_DIAGNOSTIC_DATA);
		if (extPoint != null) {

			IConfigurationElement[] configElements = extPoint
					.getConfigurationElements();
			for (IConfigurationElement cfgElement : configElements) {
				/*
				 * get bundle's start level
				 */
				String name = cfgElement.getAttribute(ATTR_NAME);
				String version = cfgElement.getAttribute(ATTR_VERSION);

				listNames.add(name+" ("+version+")");
			}

		}

		/*
		 * get all OSGi registered diag data service components
		 */
		ServiceReference[] diagSrvReferences;
		try {
			diagSrvReferences = this.ctxBundle.getServiceReferences(
					IDiagnosticData.class.getName(), null);
			if (diagSrvReferences != null) {
				for (int i = 0; i < diagSrvReferences.length; i++) {
					IDiagnosticData diagData = (IDiagnosticData) this.ctxBundle
							.getService(diagSrvReferences[i]);
					listNames.add(diagData.getName());
					this.ctxBundle.ungetService(diagSrvReferences[i]);
				}
			}

		} catch (InvalidSyntaxException e) {
			// TODO Auto-generated catch block
		}

		return listNames.toArray(new String[listNames.size()]);
	}

	public File saveData() throws DiagnosticException {
		return saveData(IDiagnosticManager.DEFAULT_FILTER);
	}

	/**
	 * @see IDiagnosticManager.dumpData(String)
	 */
	public File saveData(String filter) throws DiagnosticException {
		/*
		 * initialize diagnostic dump file
		 */
		String dumpName = "corona_" + Long.toString(System.currentTimeMillis())
				+ ".dmp";
		File dumpFile = this.ctxBundle.getDataFile(dumpName);
		FileOutputStream fos;
		try {
			fos = new FileOutputStream(dumpFile);
		} catch (FileNotFoundException e1) {
			DiagnosticException xDiag = new DiagnosticException(
					DiagnosticException.EXCEPTION_DUMP_FILE);
			xDiag.initCause(e1);
			throw xDiag;
		}
		ZipOutputStream zos = new ZipOutputStream(fos);

		/*
		 * create filter to select diagnostic data components to save
		 */
		Filter diagFilter = null;
		try {
			diagFilter = this.ctxBundle.createFilter(filter);
		} catch (InvalidSyntaxException e1) {
			DiagnosticException xDiag = new DiagnosticException(
					IDiagnosticManager.ERR_INVALID_FILTER);
			throw xDiag;
		}
		
		saveDataExtensions(diagFilter, zos);

		saveDataServices(filter, zos);

		/*
		 * create META-DATA/MANIFEST.MF
		 */
		createManifest(zos);

		/*
		 * close Corona's diagnostic dump file
		 */
		try {
			zos.close();
			fos.close();
		} catch (IOException e) {
			DiagnosticException xDiag = new DiagnosticException(
					DiagnosticException.EXCEPTION_DUMP_FILE);
			xDiag.initCause(e);
			throw xDiag;
		}

		return dumpFile;

	}

	/**
	 * Dump all of the diagnostic data components defined via Ext Point
	 * 
	 * @param name
	 * @param zos
	 */
	private void saveDataExtensions(Filter diagFilter, ZipOutputStream zos) {
		/*
		 * get all ext point registered diagnostic data components
		 */
		IExtensionPoint extPoint = Platform.getExtensionRegistry()
				.getExtensionPoint(IDiagnosticData.EXT_POINT_DIAGNOSTIC_DATA);
		if (extPoint != null) {

			IConfigurationElement[] configElements = extPoint
					.getConfigurationElements();
			for (IConfigurationElement cfgElement : configElements) {

				String diagName = cfgElement.getAttribute(ATTR_NAME);
				String diagVersion = cfgElement.getAttribute(ATTR_VERSION);

				Hashtable<String, String> cfgDictionary = new Hashtable<String, String>();
				cfgDictionary.put(ATTR_NAME, diagName);
				cfgDictionary.put(ATTR_VERSION, diagVersion);

				/*
				 * check if match to dump diag data component name
				 */
				boolean dumpFlag = false;
				if ( diagFilter.match(cfgDictionary) ) {
					dumpFlag = true;
				}

				/*
				 * dump only request diag data components
				 */
				if (dumpFlag) {
					IDiagnosticData diagData;
					try {
						diagData = (IDiagnosticData) cfgElement
								.createExecutableExtension("class");

						diagData.setName(diagName);
						diagData.setVersion(diagVersion);

						dumpDiagnosticData(diagData, zos);

						addManifestEntry(diagName + ".version=" + diagVersion);

					} catch (CoreException e) {
						logger.error( 
								"Unable to create diag data extension", e);
					} catch (DiagnosticException e) {
						logger.error(  
								"Unable to capture diag data: ", e);
					}
				}
			}
		}
	}

	/**
	 * Dump all of the diagnostic data components defined via OSGi service
	 * 
	 * @param name
	 * @param zos
	 * @throws DiagnosticException
	 */
	private void saveDataServices(String filter, ZipOutputStream zos)
			throws DiagnosticException {
		/*
		 * using a service component filter, get the list of diagnostic data
		 * components
		 */
		ServiceReference[] diagSrvReferences;
		try {
			diagSrvReferences = this.ctxBundle.getServiceReferences(
					IDiagnosticData.class.getName(), filter);
		} catch (InvalidSyntaxException e) {
			DiagnosticException xDiag = new DiagnosticException(
					DiagnosticException.EXCEPTION_GET_COMPONENT_SERVICES);
			xDiag.initCause(e);
			throw xDiag;
		}

		/*
		 * process each diagnostic data component
		 */
		for (int i = 0; (diagSrvReferences != null)
				&& (i < diagSrvReferences.length); i++) {
			ServiceReference srvReference = diagSrvReferences[i];

			IDiagnosticData diagData = (IDiagnosticData) this.ctxBundle
					.getService(srvReference);

			dumpDiagnosticData(diagData, zos);

			/*
			 * add all srv component properties to manifest
			 */
			String[] keys = srvReference.getPropertyKeys();
			for (int j = 0; j < keys.length; j++) {
				String key = keys[j];

				/*
				 * skip transient properties
				 */
				if (key.equals("service.id")) {
					continue;
				} else if (key.equals("component.id")) {
					continue;
				} else if (key.equals("objectClass")) {
					continue;
				}

				Object obj = srvReference.getProperty(key);
				String attrValue = null;
				if (obj instanceof String[]) {
					String[] values = (String[]) obj;

					attrValue = values[0];
					for (int k = 1; k < values.length; k++) {
						attrValue = attrValue + "," + values[k];
					}
				} else {
					attrValue = obj.toString();
				}

				String attrName = diagData.getName() + "." + key;
				addManifestEntry(attrName + ": " + attrValue);
			}

		}
	}

	private void addManifestEntry(String entry) {
		manifestEntries.append(entry + "\n");
	}

	/**
	 * Create the manifest for the diagnostic dump
	 * 
	 * @param diagSrvReferences
	 * @param zos
	 * @throws DiagnosticException
	 * @throws IOException
	 */
	private void createManifest(ZipOutputStream zos) throws DiagnosticException {
		ZipEntry zipEntry = new ZipEntry("META-INF/MANIFEST.MF");
		try {
			zos.putNextEntry(zipEntry);
		} catch (IOException e1) {
			DiagnosticException xDiag = new DiagnosticException(
					DiagnosticException.EXCEPTION_DUMP_FILE);
			xDiag.initCause(e1);
			throw xDiag;
		}

		PrintStream printStream = new PrintStream(zos);

		/*
		 * print manifest header information
		 */
		printStream.println(IDiagnosticData.ATTR_MANIFEST_VERSION + ": "
				+ IDiagnosticData.MANIFEST_VERSION);
		;
		printStream.println(IDiagnosticData.ATTR_DIAGNOSTIC_MANIFEST_VERSION
				+ ": " + IDiagnosticData.DIAGNOSTIC_MANIFEST_VERSION);
		;
		printStream.println(IDiagnosticData.ATTR_DIAGNOSTIC_VERSION + ": "
				+ IDiagnosticData.DIAGNOSTIC_VERSION);
		printStream.println(IDiagnosticData.ATTR_DIAGNOSTIC_VENDOR + ": "
				+ IDiagnosticData.DIAGNOSTIC_VENDOR);

		printStream.println(manifestEntries.toString());

		try {
			zos.closeEntry();
		} catch (IOException e) {
			DiagnosticException xDiag = new DiagnosticException(
					DiagnosticException.EXCEPTION_DUMP_FILE);
			xDiag.initCause(e);
			throw xDiag;
		}
	}

	/**
	 * Dump the diagnostic data for each service component
	 * 
	 * @param diagData
	 * @param zos
	 *            ZipOutputStream
	 * @throws DiagnosticException
	 */
	private void dumpDiagnosticData(IDiagnosticData diagData,
			ZipOutputStream zos) throws DiagnosticException {

		/*
		 * capture the diagnostic data
		 */
		diagData.capture();

		ZipEntry zipEntry = new ZipEntry(diagData.getName() + "."
				+ diagData.getType());
		try {
			zos.putNextEntry(zipEntry);
			diagData.save(zos);
			zos.closeEntry();
		} catch (IOException e) {
			DiagnosticException xDiag = new DiagnosticException(
					DiagnosticException.EXCEPTION_DUMP_FILE);
			xDiag.initCause(e);
			throw xDiag;
		}
	}

	/**
	 * @see IDiagnosticManager#loadData(File)
	 */
	public IDiagnosticData[] loadData(File file) throws DiagnosticException {
		ArrayList<IDiagnosticData> listDiagData = new ArrayList<IDiagnosticData>();

		try {
			ZipFile zipFile = new ZipFile(file);

			/*
			 * get all ext point registered diagnostic data components
			 */
			IExtensionPoint extPoint = Platform.getExtensionRegistry()
					.getExtensionPoint(
							IDiagnosticData.EXT_POINT_DIAGNOSTIC_DATA);
			if (extPoint != null) {

				IConfigurationElement[] configElements = extPoint
						.getConfigurationElements();
				for (IConfigurationElement cfgElement : configElements) {

					String diagName = cfgElement.getAttribute(ATTR_NAME);
					String diagVersion = cfgElement.getAttribute(ATTR_VERSION);

					IDiagnosticData diagData;
					try {
						diagData = (IDiagnosticData) cfgElement
								.createExecutableExtension("class");
						diagData.setName(diagName);
						diagData.setVersion(diagVersion);

						String zipEntryName = diagData.getName() + "."
								+ diagData.getType();

						ZipEntry zipEntry = zipFile.getEntry(zipEntryName);
						InputStream inStream = zipFile.getInputStream(zipEntry);

						diagData.load(inStream);
						listDiagData.add(diagData);
					} catch (Throwable t) {
						logger.error( 
								"Unable to load diag data for: " + diagName, t);
					}
				}
			}

		} catch (Throwable t) {
			DiagnosticException xDiag = new DiagnosticException(
					"Unable to load diagnostic data.");
			xDiag.initCause(t);
			logger.error(  xDiag.getLocalizedMessage(),
					xDiag);
			throw xDiag;
		}

		IDiagnosticData[] data = (IDiagnosticData[]) listDiagData.toArray(new IDiagnosticData[listDiagData.size()]);
		return data;
	}
}
