/*
 * Copyright (c) 2007-2008 Compuware Corporation and others.
 * 
 * All rights reserved. This program and the accompanying 
 * materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at:
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors:
 *     Compuware Corporation - initial API and implementation
 */
package org.eclipse.corona.internal.diagnostic.data;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.util.Dictionary;
import java.util.Hashtable;

import javax.xml.parsers.ParserConfigurationException;

import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.event.Event;
import org.osgi.service.event.EventConstants;
import org.osgi.service.event.EventHandler;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import org.apache.log4j.Logger;
import org.eclipse.corona.diagnostic.AbstractDiagnosticData;
import org.eclipse.corona.diagnostic.DiagnosticException;
import org.eclipse.corona.diagnostic.IDiagnosticData;
import org.eclipse.corona.diagnostic.data.EventData;
import org.eclipse.corona.internal.diagnostic.Activator;

/**
 * Diagnostic data for OSGi Events
 * 
 * @author doflynn
 */
public class EventDiagnosticData extends AbstractDiagnosticData implements Serializable, EventHandler {

	private Logger logger = Logger.getLogger(EventDiagnosticData.class.getName());
	
	private static final long serialVersionUID = 1L;
	private transient BundleContext ctxBundle;
	private transient ServiceRegistration srvRegEventHandler;
	private transient int qSize = 200;
	private transient EventData[] qTempData;
	private transient int qTempHead = 0;
	private transient int qTempTail = 0;
	private EventData[] qCaptureData;
	private transient int qCaptureHead = 0;
	private transient int qCaptureTail = 0;

	private static final String ATTR_QSIZE = "qsize";

	/**
	 * Default CTOR used by Declarative Services
	 */
	public EventDiagnosticData() {
		super();
	}

	/**
	 * Activate the service component
	 * 
	 * @param ctxComponent
	 */
	protected void activate(ComponentContext ctxComponent ) {
		this.ctxBundle = ctxComponent.getBundleContext();
		super.initComponent(ctxComponent.getProperties());

		/*
		 * get the value of the data capture queue from the
		 * component's deployment descriptor
		 */
		Object obj = ctxComponent.getProperties().get(ATTR_QSIZE);
		if ( obj instanceof String[] ) {
			String strQSize = ((String[])obj)[0];
			if ( strQSize != null  ) {
				qSize = Integer.parseInt(strQSize);
			}
		}

		qTempData = new EventData[qSize];
		
		try {
			// register the default Collaboration event handler
			Dictionary<String, String[]> dict = new Hashtable<String, String[]>();
			String topics[] = { "*" };
			dict.put(EventConstants.EVENT_TOPIC, topics);
			srvRegEventHandler = this.ctxBundle.registerService(EventHandler.class
					.getName(), this, dict);
			
		}
		catch(Throwable t) {
			logger.error("failed to register event handler", t);
		}

	}
	
	protected void deactivate(ComponentContext ctxt)
	{
		srvRegEventHandler.unregister();
	}

	
	public void capture() {
		synchronized (qTempData) {
			qCaptureData = qTempData;
			qCaptureHead = qTempHead;
			qCaptureTail = qTempTail;
			
			qTempData = new EventData[qSize];
			qTempHead = 0;
			qTempTail = 0;
		}
	}

	/**
	 * @throws DiagnosticException 
	 * @see IDiagnosticData#load(InputStream)
	 */
	public void load(InputStream inStream) throws DiagnosticException {
		super.load(inStream);
	}

	/**
	 * @see IDiagnosticData#save(OutputStream)
	 */
	public void save(OutputStream outStream) throws IOException {
		try {
			createXML();
		} catch (ParserConfigurationException e) {
			logger.error("failed to save event diag data", e);
			IOException xIO = new IOException(e.getMessage());
			xIO.initCause(e);
			throw xIO;
		}
		
		super.save(outStream);
		
		cleanUp();
	}

	/**
	 * Handle the OSGI events
	 * 
	 */
	public void handleEvent(Event event) {
		EventData data = new EventData(event);
		
		synchronized (qTempData) {
			qTempData[qTempHead++] = data;
			qTempHead = qTempHead % qSize;
			
			if ( qTempHead == qTempTail ) {
				qTempTail = (++qTempTail % qSize);
			}
		}
	}

	private int qCaptureSize() {
		int size = 0;
		
		if ( qTempHead > qTempTail ) {
			size = qTempHead - qTempTail;
		}
		else {
			size = qSize - qTempHead + qTempTail;
		}
		
		return size;
	}
	
	/**
	 * Create the diagnostic data in XML format
	 * @throws ParserConfigurationException 
	 */
	private void createXML() throws ParserConfigurationException {
		super.initXmlDocument();

		Element dataElement = this.docDiag.getDocumentElement();
		
		Element elementPropertyList = super.createElementPropertyList();
		dataElement.appendChild(elementPropertyList);
		
		int count = (qCaptureHead >= qCaptureTail ? qCaptureHead-qCaptureTail : qSize-qCaptureHead+qCaptureTail-1);
		logger.debug( "...diagnostic event capture count:" +count);
		
		dataElement.setAttribute(ATTR_COUNT, Integer.toString(count) );
		
		/*
		 * add all captured event data to diagnostics
		 */
		for (int i = 0; i < count; i++) {
			EventData data = qCaptureData[i];
			Event event = data.getEvent();

			/*
			 * create the xml element for this event
			 */
			Element eventElement = docDiag.createElement("Event");
			eventElement.setAttribute("topic", event.getTopic() );
			eventElement.setAttribute("timestamp", Long.toString(data.getTimestamp()) );
			dataElement.appendChild(eventElement);

			/*
			 * Add the event's properties
			 */
			Element rootPropertyList = docDiag.createElement("PropertyList");
			eventElement.appendChild(rootPropertyList);
			String[] names = event.getPropertyNames();
			for (int j = 0; j < names.length; j++) {
				Object value = event.getProperty(names[j]);
				Element property = createElementProperty(docDiag, names[j], value);
				rootPropertyList.appendChild(property);
			}
		}
	}

	/**
	 * @see IDiagnosticData.getType()
	 */
	public String getType() {
		return TYPE_XML;
	}
}
