/*
 * Copyright (c) 2007-2008 Compuware Corporation and others.
 * 
 * All rights reserved. This program and the accompanying 
 * materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at:
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors:
 *     Compuware Corporation - initial API and implementation
 */
package org.eclipse.corona.internal.diagnostic.data;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.util.Dictionary;
import java.util.Hashtable;

import javax.xml.parsers.ParserConfigurationException;

import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.event.Event;
import org.osgi.service.event.EventConstants;
import org.osgi.service.event.EventHandler;
import org.osgi.service.log.LogEntry;
import org.osgi.service.log.LogListener;
import org.osgi.service.log.LogReaderService;
import org.osgi.service.log.LogService;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import org.apache.log4j.Logger;
import org.eclipse.corona.diagnostic.AbstractDiagnosticData;
import org.eclipse.corona.diagnostic.DiagnosticException;
import org.eclipse.corona.diagnostic.IDiagnosticData;
import org.eclipse.corona.internal.diagnostic.Activator;

/**
 * Diagnostic data for OSGi Events
 * 
 * @author doflynn
 */
public class LogDiagnosticData extends AbstractDiagnosticData implements Serializable, LogListener {

	private static final String LOG_READER_SERVICE = "LogReaderService";

	private Logger logger = Logger.getLogger(LogDiagnosticData.class.getName());
	
	private static final long serialVersionUID = 1L;
	private transient BundleContext ctxBundle;
	private transient ServiceRegistration srvRegEventHandler;
	private transient int qSize = 200;
	private transient LogEntry[] qTempData;
	private transient int qTempHead = 0;
	private transient int qTempTail = 0;
	private LogEntry[] qCaptureData;
	private transient int qCaptureHead = 0;
	private transient int qCaptureTail = 0;

	private ServiceReference refLogReaderService;
	
	private static final String ATTR_QSIZE = "qsize";
	
	/**
	 * Default CTOR used by Declarative Services
	 */
	public LogDiagnosticData() {
		super();
	}

	/**
	 * Activate the service component
	 * 
	 * @param ctxComponent
	 */
	protected void activate(ComponentContext ctxComponent ) {
		this.ctxBundle = ctxComponent.getBundleContext();
		super.initComponent(ctxComponent.getProperties());

		/*
		 * get the value of the data capture queue from the
		 * component's deployment descriptor
		 */
		Object obj = ctxComponent.getProperties().get(ATTR_QSIZE);
		if ( obj instanceof String[] ) {
			String strQSize = ((String[])obj)[0];
			if ( strQSize != null  ) {
				qSize = Integer.parseInt(strQSize);
			}
		}
		
		qTempData = new LogEntry[qSize];
		
		/*
		 * get the log reader service
		 */
		LogReaderService logReaderService = (LogReaderService)ctxComponent.locateService(LOG_READER_SERVICE);
		if ( logReaderService != null ) {
			logReaderService.addLogListener(this);
		}
	}
	
	protected void deactivate(ComponentContext ctxt)
	{
		if ( refLogReaderService != null ) {
			ctxBundle.ungetService(refLogReaderService);
		}
	}

	
	public void capture() {
		if ( refLogReaderService != null ) {
			
			synchronized (qTempData) {
				qCaptureData = qTempData;
				qCaptureHead = qTempHead;
				qCaptureTail = qTempTail;
				
				qTempData = new LogEntry[qSize];
				qTempHead = 0;
				qTempTail = 0;
			}
		}
	}

	/**
	 * @throws DiagnosticException 
	 * @see IDiagnosticData#load(InputStream)
	 */
	public void load(InputStream inStream) throws DiagnosticException {
		super.load(inStream);
	}

	/**
	 * @see IDiagnosticData#save(OutputStream)
	 */
	public void save(OutputStream outStream) throws IOException {
		if ( refLogReaderService != null ) {			
			try {
				createXML();
			} catch (ParserConfigurationException e) {
				logger.error("failed to save log diag data", e);
				IOException xIO = new IOException(e.getMessage());
				xIO.initCause(e);
				throw xIO;
			}
			
			super.save(outStream);
			
			cleanUp();
		}
	}

	private int qCaptureSize() {
		int size = 0;
		
		if ( qTempHead > qTempTail ) {
			size = qTempHead - qTempTail;
		}
		else {
			size = qSize - qTempHead + qTempTail;
		}
		
		return size;
	}
	
	/**
	 * Create the diagnostic data in XML format
	 * @throws ParserConfigurationException 
	 */
	private void createXML() throws ParserConfigurationException {
		super.initXmlDocument();

		Element dataElement = this.docDiag.getDocumentElement();
		Element elementPropertyList = super.createElementPropertyList();
		dataElement.appendChild(elementPropertyList);
		
		int count = (qCaptureHead >= qCaptureTail ? qCaptureHead-qCaptureTail : qSize-qCaptureHead+qCaptureTail-1);
		logger.debug( "...diagnostic event capture count:" +count);
		
		dataElement.setAttribute(ATTR_COUNT, Integer.toString(count) );
		
		/*
		 * add all captured event data to diagnostics
		 */
		for (int i = 0; i < count; i++) {
			LogEntry data = qCaptureData[i];

			/*
			 * create the xml element for this event
			 */
			Element logElement = docDiag.createElement("LogEntry");
			logElement.setAttribute("level", Integer.toString(data.getLevel()) );
			logElement.setAttribute("timestamp", Long.toString(data.getTime()) );
			logElement.setAttribute("bundle", data.getBundle().getSymbolicName() );

			Text logMessage = docDiag.createTextNode(data.getMessage());
			logElement.appendChild(logMessage);
			
			dataElement.appendChild(logElement);

			/*
			 * add LogEntry's exception information
			 */
			Throwable t = data.getException();
			if ( t != null ) {
				Element xElement = docDiag.createElement("Exception");
				xElement.setAttribute("message", t.getMessage());

				StringBuffer strbuf = new StringBuffer();
				StackTraceElement[] stackTrace = t.getStackTrace();
				for (int j = 0; j < stackTrace.length; j++) {
					strbuf.append( stackTrace[i].toString() + "\n" );
				}
				Text xMessage = docDiag.createTextNode(strbuf.toString());
				xElement.appendChild(xMessage);
				
				logElement.appendChild(xElement);
			}
		}
	}

	/**
	 * @see IDiagnosticData.getType()
	 */
	public String getType() {
		return TYPE_XML;
	}

	/**
	 * Handle LogService entries
	 */
	public void logged(LogEntry entry) {
		synchronized (qTempData) {
			qTempData[qTempHead++] = entry;
			qTempHead = qTempHead % qSize;
			
			if ( qTempHead == qTempTail ) {
				qTempTail = (++qTempTail % qSize);
			}
		}
	}
}
