/*
 * Copyright (c) 2007-2008 Compuware Corporation and others.
 * 
 * All rights reserved. This program and the accompanying 
 * materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at:
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors:
 *     Compuware Corporation - initial API and implementation
 */
package org.eclipse.corona;

import java.util.Enumeration;
import java.util.Iterator;
import java.util.PriorityQueue;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.TreeMap;

import org.apache.log4j.Logger;
import org.eclipse.equinox.app.IApplication;
import org.eclipse.equinox.app.IApplicationContext;
import org.osgi.framework.BundleContext;
import org.osgi.service.application.ApplicationException;
import org.osgi.util.tracker.ServiceTracker;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IContributor;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.corona.internal.Activator;

/**
 * Corona Server Application
 * </p>
 * Platform runtime supports plug-ins which would like to declare 
 * main entry points. That is, programs which would like to run using 
 * the platform runtime but yet control all aspects of execution can 
 * declare themselves as an application. Declared applications can 
 * be run directly from the main platform launcher by specifying 
 * the -application argument where the parameter is the id of an 
 * extension supplied to the applications extension point described 
 * here. This application is instantiated and run by the platform. 
 * Platform clients can also use the platform to lookup and run 
 * multiple applications
 *</P>
 * An attempt will be made to start all of the applications defined
 * by the <i>osgi.applications</i> system property.
 * 
 * see org.eclipse.equinox.internal.app.AppCommands
 */
public class Application implements IApplication {

	private static final String SCHEDULED_FILTER = ".schedule.filter";
	private static final String SCHEDULED_RECURRING = ".schedule.recurring";
	private static final String CORONA_APP = "corona.app.";
	private static final String EXT_POINT_STARTUP = "org.eclipse.corona.startup";
	private static final String EXT_POINT_APPLICATION = "org.eclipse.core.runtime.applications";
	private static final String OSGI_APPLICATIONS = "osgi.applications";
	private static final String OSGI_APPLICATIONS_STARTLEVEL = "osgi.applications.defaultStartLevel";

	private Logger logger = Logger.getLogger(Application.class.getName());
	
//	private LogService logService;
	
	private boolean stopRequest = false;
	private final long STOP_TIMEOUT = (1000 * 5);	// stop timeout

	private BundleContext ctxBundle;
	private IApplicationManager appManager;
	private ServiceTracker srvTrackerAppManager;

	private int startLevel = 5;
	
	/**
	 * This inner class is used to process and order the list of
	 * applications to be automatically started.  
	 * 
	 */
	class AppSyntax implements Comparable<AppSyntax> {
		private Logger logger = Logger.getLogger(AppSyntax.class.getName());
		
		String app;
		String id;
		int level = startLevel;
		String schedFilter;
		boolean schedRecurring;
		
		public AppSyntax() {

		}

		public void setApp(String app) {
			logger.info( "setting service application: "+app);
			
			this.app = app;
			
			/*
			 * parse the app syntax w/ @ 
			 */
			if ( app.contains("@") ) {
				StringTokenizer appTokenizer = new StringTokenizer(app, "@,");
				String appId = appTokenizer.nextToken();
				
				String appLevel = appTokenizer.nextToken();
				setLevel(appLevel);
				
				StringBuffer strbuf = new StringBuffer();
				while( appTokenizer.hasMoreTokens() ) {
					strbuf.append(","+appTokenizer.nextToken());
				}
				String appArgs=strbuf.toString();
				
				this.app = appId+appArgs;
			}

			/*
			 * set the application's ID
			 */
			StringTokenizer appTokenizer = new StringTokenizer(this.app, ",");
			this.id = appTokenizer.nextToken();

		}
		
		public void setLevel( String strLevel ) {
			try {
				this.level = Integer.parseInt(strLevel);
			}
			catch (Throwable t) {
				logger.warn( "invalid application start level: "+strLevel);
			}			
		}
		
		public int getLevel() {
			return level;
		}

		public String getId() {
			return this.id;
		}
		
		public String getApp() {
			return this.app;
		}
		
		public int compareTo(AppSyntax osgiApp ) {
			int result = 0;
			
			if ( this.level < osgiApp.getLevel() ) {
				result = -1;
			}
			if ( this.level > osgiApp.getLevel() ) {
				result = 1;
			}

			return result;
		}

		public String getSchedFilter() {
			return schedFilter;
		}

		public void setSchedFilter(String schedFilter) {
			this.schedFilter = schedFilter;
		}

		public boolean isSchedRecurring() {
			return schedRecurring;
		}

		public void setSchedRecurring(boolean schedRecurring) {
			this.schedRecurring = schedRecurring;
		}
		public void setSchedRecurring(String schedRecurring) {
			this.schedRecurring = Boolean.getBoolean(schedRecurring);
		}

		
	}
	/**
	 * Construct Corona Server Application
	 * 
	 */
	public Application() {

		this.ctxBundle = Activator.getDefault().getBundleContext();

		/*
		 * initialize the default osgi application start level
		 */
		String temp = System.getProperty(OSGI_APPLICATIONS_STARTLEVEL);
		if ( temp != null ) {
			try {
				startLevel = Integer.parseInt(temp);
			}
			catch (Throwable t) {
				logger.warn( "invalid application start level: "+temp);
			}
		}
	}

	/**
	 * Start the Corona server application
	 */
	public Object start(IApplicationContext ctxApplication) throws Exception {
		logger.debug( "entering Application.start()");

		srvTrackerAppManager = new ServiceTracker(ctxBundle, IApplicationManager.class.getName(), null);
		srvTrackerAppManager.open();
		appManager = (IApplicationManager)srvTrackerAppManager.waitForService(10 * 1000);
		if ( appManager == null ) {
			logger.error("unable to get IApplicationManager service");
			return new Integer(-1);
		}
		
		ctxApplication.applicationRunning();

		/*
		 * start the bundles registered via ext point
		 */
		startBundles();

		/*
		 * start other registered server applications
		 */
		try {
			startApplications();
		}
		catch( Throwable t) {
			logger.error("error starting Corona IServiceApplication's", t);
		}
		
		/*
		 * this application doesn't do much.  it just hangs
		 * around to the the other services do their thing.
		 */
		logger.info("waiting for stop() request or all applications to end");
		synchronized(this) {
			
			while( !this.stopRequest  ) {
				this.wait(STOP_TIMEOUT);
				
				/*
				 * wait for all running applications to finish
				 */
				if ( appManager.countRunningApplications() == 0 ) {
					logger.info("no more running applications");

					/*
					 * wait for all scheduled applications to finish
					 */
					if ( appManager.countScheduledApplications() == 0 ) {
						logger.info("no more scheduled applications");
						
						break;
					}
				}
			}
		}
	
		/*
		 * stop the applications we started
		 */
		try {
			appManager.stopAllApplications();
		} catch (CoronaException e) {
			logger.error( e.getMessage());
		}
		
		srvTrackerAppManager.close();		

		logger.debug( "exiting Application.start()");
		return EXIT_OK;
	}

	private void startBundles() {
		logger.info("jump starting "+EXT_POINT_STARTUP+" bundles");
		
		/*
		 * get all registered applications
		 */
		IExtensionPoint extPoint= Platform.getExtensionRegistry().getExtensionPoint(EXT_POINT_STARTUP);
		if ( extPoint != null ) {
			/*
			 * create map of bundles to start
			 */
			TreeMap mapTree = new TreeMap();
			
			/*
			 * get the default bundle start level
			 */
			String defaultLevel = System.getProperty("osgi.bundles.defaultStartLevel", "5");

			IConfigurationElement[] configElements = extPoint.getConfigurationElements();
			for( IConfigurationElement cfgElement : configElements ) {
				
				IExtension extension = cfgElement.getDeclaringExtension();
				
				/*
				 * get bundle's start level
				 */
				String attrLevel = cfgElement.getAttribute("level");
				if (attrLevel == null ) {
					attrLevel = defaultLevel;
				}
				Integer level = new Integer(attrLevel);
				
				/*
				 * add bundle to startup map
				 */
				IContributor contrib = extension.getContributor();
				mapTree.put(level, contrib);
			}

			/*
			 * start all the bundles
			 */
			Iterator itor = mapTree.values().iterator();
			while (itor.hasNext()) {
				IContributor contrib = (IContributor) itor.next();
				
				try {
					Activator.getDefault().startBundle(contrib.getName());
				} catch (Exception e) {
					logger.error( "Unable to start bundle", e);
				}					
			}
		}
	}

	/**
	 * start other registered server applications
	 * @throws ApplicationException 
	 */
	private void startApplications() {
		logger.debug( "entering: Application.startApplications()");
				
		/*
		 * set the priorty start level for the applications
		 */
		PriorityQueue<AppSyntax> appQueue = new PriorityQueue<AppSyntax>();
		
		addOsgiApplications(appQueue);

		addCoronaApplications(appQueue);
		
		AppSyntax[] osgiApplications = (AppSyntax[]) appQueue.toArray(new AppSyntax[appQueue.size()]);

		/*
		 * poke application extension point to wake up plug-in
		 */
		startApplicationExtensionBundle(osgiApplications);
		
		/*
		 * start the applications in order by start level
		 */
		for (int i = 0; i < osgiApplications.length; i++) {
			try {
				AppSyntax srvApp = osgiApplications[i];
				
				/*
				 * Only auto-start those applications whose start level
				 * is NOT 0 (zero)
				 */
				if ( srvApp.getLevel() > 0 ) {
					appManager.startApplication( srvApp.getApp() );					
				}

				/*
				 * Check if the application should be scheduled
				 */
				if ( srvApp.getSchedFilter() != null ) {
					appManager.scheduleApplication(srvApp.getId(), srvApp.getSchedFilter(), srvApp.isSchedRecurring());
				}
				
			} catch (CoronaException e) {
				logger.warn( e.getMessage());
			}
		}
		
		logger.debug( "exiting: Application.startApplications()");
	}


	/**
	 * Add the OSGi service applications defined by the "corona.app.*" properties
	 * 
	 * @param appQueue
	 */
	private void addCoronaApplications(PriorityQueue<AppSyntax> appQueue) {
		logger.info("disovering corona service applications");
		
		/*
		 * process 'corona.app.*' properties
		 */
		Properties properties = System.getProperties();
		Enumeration<Object> enumKeys = properties.keys();
		while( enumKeys.hasMoreElements() ) {
			String name = (String)enumKeys.nextElement();
			if ( name.startsWith(CORONA_APP) ) {

				/*
				 * compute the applicatoin's name
				 */
				String app = name.substring(CORONA_APP.length());
				String appCmd = app;
				
				/*
				 * get application start level
				 */
				String startLevel = properties.getProperty(name);
				if ( startLevel != null ) {
					appCmd+="@"+startLevel;
				}

				/*
				 * get the application's arguments
				 */
				String args = properties.getProperty(app+".args");
				if ( args != null ) {
					appCmd+=","+args;
				}
				
				/*
				 * TODO check for <app>.arg.1 & <app>.arg.2 & <app>.arg.n
				 */

				AppSyntax appSyntax = new AppSyntax();
				appSyntax.setApp(appCmd);
				
				/*
				 * check if Service Application is a scheduled application
				 */
				appSyntax.setSchedFilter(properties.getProperty(app+SCHEDULED_FILTER) );
				appSyntax.setSchedRecurring(properties.getProperty(app+SCHEDULED_RECURRING) );
				
				appQueue.add( appSyntax );
			}
		}
	}

	/**
	 * Add the applications defined by the osgi.applications property
	 * 
	 * @param appQueue
	 */
	private void addOsgiApplications(PriorityQueue<AppSyntax> appQueue) {
		/*
		 * get the 'osgi.applications' to be started
		 */
		String apps = System.getProperty(OSGI_APPLICATIONS);
		if ( apps == null ) {
			logger.info( "'osgi.applications' property not defined");
			return;
		}
		
		/*
		 * process each application to be started
		 */
		StringTokenizer appTokenizer = new StringTokenizer(apps, ";");
		while(appTokenizer.hasMoreTokens()) {
			/*
			 * get the application definition (w/ args)
			 */
			String app = appTokenizer.nextToken();
			
			AppSyntax appSyntax = new AppSyntax();
			appSyntax.setApp(app);
			appQueue.add( appSyntax );
		}
	}
	
	private void startApplicationExtensionBundle(AppSyntax[] osgiApplications) {
		/*
		 * get all registered applications
		 */
		IExtensionPoint extPoint= Platform.getExtensionRegistry().getExtensionPoint(EXT_POINT_APPLICATION);
		if ( extPoint != null ) {
			IConfigurationElement[] appConfigurations = extPoint.getConfigurationElements();
			for( IConfigurationElement appCfg : appConfigurations ) {
				
				IExtension extension = appCfg.getDeclaringExtension();
				String extId = extension.getUniqueIdentifier();
				
				/*
				 * for each application extension, check if it should be started
				 */
				boolean appFound = false;
				for (AppSyntax appStart : osgiApplications ) {
					if ( appStart.getId().equals(extId) ) {
						appFound = true;
						break;						
					}
				}
			
				/*
				 * only start the plug-in that provides the application extension
				 * if it one of the applications defined by osgi.applications 
				 */
				if ( appFound ) {
					try {
						IContributor contrib = extension.getContributor();
						Activator.getDefault().startBundle(contrib.getName());
					} catch (Exception e) {
						logger.error( "Unable to create application extenstion", e);
					}					
				}
			}
		}
	}

	/**
	 * Stop the Corona server application
	 */
	public void stop() {
		logger.debug( "entering Application.stop()");

		logger.info("stopping Corona's application");
		this.stopRequest = true;
		
		synchronized(this) {
			this.notify();
		}
		
		logger.debug( "exiting Application.stop()");
	}

}
