/*
 * Copyright (c) 2007-2008 Compuware Corporation and others.
 * 
 * All rights reserved. This program and the accompanying 
 * materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at:
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors:
 *     Compuware Corporation - initial API and implementation
 */
package org.eclipse.corona;

import java.io.File;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.concurrent.ThreadFactory;

import org.apache.log4j.Logger;
import org.eclipse.equinox.app.IApplication;
import org.eclipse.equinox.app.IApplicationContext;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleException;
import org.osgi.framework.ServiceRegistration;

/**
 * Base abstract class for component server application.
 * 
 * @author doflynn
 */
public abstract class AbstractServiceApplication implements IServiceApplication {

	private Logger logger = Logger.getLogger(AbstractServiceApplication.class.getName());
	
	private static final String WORKING_DIR = "working.dir";
	protected String name;
	private String[] args;
	private Properties properties = new Properties();
	protected File workingDir;
	private ThreadGroup threadGroup;

	private ServiceRegistration srvRegistration;

	private BundleContext ctxBundle;

	/**
	 * Start the component server application. 
	 * <p/> 
	 * The initialize of the
	 * component server application will setup the server's application
	 * environment. Once the initialization is finished, the component server
	 * will be registered as an OSGi service component as well.
	 * 
	 * @param ctxBundle
	 *            The bundle context that this server application belongs to.
	 * @param ctxApplication
	 *            application context
	 * @param name
	 *            of server application
	 * @see IApplication#start(IApplicationContext)
	 */
	@SuppressWarnings("unchecked")
	public Object start(BundleContext ctxBundle,
			IApplicationContext ctxApplication, String name) throws Exception {

		this.ctxBundle = ctxBundle;
		
		Object status = IApplication.EXIT_OK;

		setName(name);

		/*
		 * get the application arguments
		 */
		Map mapArgs = ctxApplication.getArguments();
		if ( mapArgs != null ) {
			this.args = (String[]) mapArgs.get("application.args");
		}
		else {
			this.args = new String[0];
		}

		/*
		 * init the server application environment
		 */
		initProperties();
		initialize();

		srvRegistration = ctxBundle.registerService(IServiceApplication.class.getName(), this,
				properties);
		return status;
	}

	/**
	 * @see IApplication#stop()
	 */
	public void stop() {
		this.srvRegistration.unregister();
	}

	/**
	 * Initialize any application properties defined as arguments
	 */
	private void initProperties() {
		/*
		 * process each application argument, if it has an '=', assume it is a
		 * property.
		 */
		for (int i = 0; i < this.args.length; i++) {
			String arg = args[i];
			/*
			 * check of presence of '='
			 */
			if (arg.contains("=")) {
				try {
					StringTokenizer strtok = new StringTokenizer(arg, "=");
					String name = strtok.nextToken();
					String value = strtok.nextToken();

					this.properties.setProperty(name, value);
				} catch (Exception x) {
					logger.warn( "invalid argument property: " + arg);
					continue;
				}
			}
		}
	}

	/**
	 * Initialize the component server's application environment. <p/>
	 * 
	 */
	protected void initialize() {
		this.threadGroup = new ThreadGroup( getName() );

		initWorkingDir();
	}

	/**
	 * Initialize the server's working directory. <p/> The component's working
	 * directory is defined by the application argument property <i>working.dir</i>.
	 * If this argument property is defined, the component's working directory
	 * will be set to this location. <p/> If no working directory is defined, it
	 * is default to a sub-folder within the <i>workspace</i>. The name of the
	 * sub-folder is the component server's name.
	 */
	protected void initWorkingDir() {
		String name = getProperty(WORKING_DIR);

		if (name == null) {
			logger.info("defaulting application working.dir to workspace");
			String instanceArea = System
					.getProperty("osgi.instance.area.default");
			if (instanceArea != null) {
				File workspace = new File(instanceArea);
				File workingDir = new File(workspace, getName());
				name = workingDir.getAbsolutePath();
			} else {
				logger.warn( "workspace is NOT defined.  unable to set application's working.dir");
			}
		}

		if (name != null) {
			this.workingDir = new File(name);
			if (!this.workingDir.exists()) {
				this.workingDir.mkdirs();
			}
		}

	}

	/**
	 * Get the value of a server application property
	 * 
	 * @param name
	 *            of the property
	 * @return property's value of <code>null</code> if not defined.
	 */
	protected String getProperty(String name) {
		return this.properties.getProperty(name);
	}

	protected void setProperty( String name, String value) {
		this.properties.setProperty(name, value);
	}
	
	/**
	 * Get the application's arguments
	 * 
	 * @return
	 */
	public String[] getArguments() {
		return this.args;
	}

	/**
	 * @see IServiceApplication#getName()
	 */
	public String getName() {
		return name;
	}
	public void setName(String name) {
		this.name = name;
		this.properties.setProperty("name", name);
	}

	/**
	 * @see IServiceApplication#getWorkingDirectory()
	 */
	public File getWorkingDirectory() {
		if ( this.workingDir == null  ) {
			logger.warn( "component server application not properly configured...");
			logger.warn( "...application working.dir is NOT defined, attempting to using 'java.io.tmpdir'");

			File dirTemp = new File( System.getProperty("java.io.tmpdir") );
			this.workingDir = new File(dirTemp, getName());
			if (!this.workingDir.exists()) {
				this.workingDir.mkdirs();
			}
		}

		return this.workingDir;
	}

	/**
	 * @see IServiceApplication#getThreadGroup()
	 */
	public ThreadGroup getThreadGroup() {
		return this.threadGroup;
	}
	
	/**
	 * @see ThreadFactory#newThread(Runnable)
	 */
	public Thread newThread(Runnable run) {
		Thread thread = new Thread(this.threadGroup, run);

		return thread;
	}
	
	/**
	 * Start OSGi Bundles
	 * <p>
	 * Convenience method to allow an OSGi Application to start a set of bundles
	 * </p>
	 * 
	 * @param bundleNames
	 */
	protected void startBundles(String[] bundleNames) {
		Bundle[] bundles = ctxBundle.getBundles();
		
		for (int i = 0; i < bundleNames.length; i++) {
			String bundleName = bundleNames[i];
			
			/*
			 * search list of all registered bundles
			 */
			boolean found = false;
			for (int j = 0; j < bundles.length; j++) {		
				String symbolicName = bundles[j].getSymbolicName();

				if ( symbolicName.equals(bundleName)) {
					// found it!
					found = true;
					
					try {
						// if the bundle is RESOLVED - start it
						if ( bundles[j].getState() == Bundle.RESOLVED ) {
							this.logger.info("application.startBundles(): " + symbolicName);

							bundles[j].start();
						}
					} catch (BundleException e) {
						this.logger.warn( "application.startBundles(): " + symbolicName, e);
					}
				}
			}
			
			// if not found, log it
			if ( !found ) {
				this.logger.warn( "Cound NOT start bundle - not found: "+bundleName);
			}
		}
	}
	
	/**
	 * Stop OSGi Bundles
	 * <p>
	 * Convenience method to allow an OSGi Application to stop a set of bundles
	 * </p>
	 * 
	 * @param bundleNames
	 */
	protected void stopBundles(String[] bundleNames) {
		Bundle[] bundles = ctxBundle.getBundles();
		
		for (int i = 0; i < bundleNames.length; i++) {
			
			/*
			 * search list of all registered bundles
			 */
			for (int j = 0; j < bundles.length; j++) {				
				String symbolicName = bundles[j].getSymbolicName();
				if ( symbolicName.equals(bundleNames[i])) {
					
					// found it!
					try {
						// if the bundle is RESOLVED - start it
						if ( bundles[j].getState() == Bundle.ACTIVE ) {
							this.logger.info("application.stopBundles(): " + symbolicName);

							bundles[j].stop();
						}
					} catch (BundleException e) {
						this.logger.warn( "application.stopBundles(): " + symbolicName, e);
					}
				}
			}			
		}
	}
}
