/*
* Copyright (c) 2005-2007 Compuware Corporation and others.
* All rights reserved. This program and the accompanying
* materials are made available under the terms of the
* Eclipse Public License v1.0 which accompanies this
* distribution, and is available at:
* http://www.eclipse.org/legal/epl-v10.html
*
* Contributors:
*     Compuware Corporation - initial API and implementation
*
*/
package org.eclipse.cosmos.dc.local.registry;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;

import org.eclipse.cosmos.dc.common.domain.Domain;
import org.eclipse.cosmos.dc.common.registry.DataFlow;
import org.eclipse.cosmos.dc.common.registry.DataSet;
import org.eclipse.cosmos.dc.common.registry.DataSource;
import org.eclipse.cosmos.dc.common.registry.DataSourceType;
import org.eclipse.cosmos.dc.common.registry.DimensionSet;
import org.eclipse.cosmos.dc.common.registry.ProviderRegistry;
import org.eclipse.cosmos.dc.mgmt.annotations.ManagedFrameworkAutowire;
import org.eclipse.cosmos.dc.mgmt.annotations.ManagedResource;
import org.eclipse.cosmos.dc.mgmt.common.ContributionManager;

@ManagedResource(autowireName="ProviderRegistry")
public class InMemoryProviderRegistry implements ProviderRegistry {
	
	@ManagedFrameworkAutowire(name="Domain")
	private Domain domain;
	
	public static final Collection<DataSet> EMPTY_LIST = new ArrayList<DataSet>(); 
	
	private HashMap<String, DataFlow> flowMap = new HashMap<String, DataFlow>();
	private HashMap<String, DataSet> setMap = new HashMap<String, DataSet>();
	private HashMap<String, DataSource> sourceMap = new HashMap<String, DataSource>();
	private HashMap<String, DataSourceType> sourceTypeMap = new HashMap<String, DataSourceType>();
	private HashMap<String, DimensionSet> dimSetMap = new HashMap<String, DimensionSet>();
	
	private HashMap<DataFlow, Collection<DataSet>> setsByFlowMap = new HashMap<DataFlow, Collection<DataSet>>(); 
	private HashMap<DataSource, Collection<DataSet>> setsBySourceMap = new HashMap<DataSource, Collection<DataSet>>();
	
	private ArrayList<DataSet> allSets = new ArrayList<DataSet>();
	
	public InMemoryProviderRegistry(){
	}
	
	public void setManager(ContributionManager manager){
		manager.manage(this);	
	}
	
	public Collection<DataSet> getDataSetsForFlow(DataFlow flow) {
		Collection<DataSet> list = setsByFlowMap.get(flow);
		if(list == null) return EMPTY_LIST;
		return list;
	}

	public Collection<DataSet> getDataSetsForSource(DataSource source) {
		Collection<DataSet> list = setsBySourceMap.get(source);
		if(list == null) return EMPTY_LIST;
		return list;
	}

	public boolean registerDataFlow(DataFlow flow) throws Exception {
		DataFlow f = flowMap.get(flow.getName());
		if(f != null) throw new Exception("DataFlow with name " + f.getName() + " already registered.");
		flowMap.put(flow.getName(), flow);
		return true;
	}

	public synchronized boolean registerDataSet(DataSet ds) throws Exception {
		DataSet set = setMap.get(ds.getName());
		if(set != null) throw new Exception("Dataset with name " + ds.getName() + " already registered.");
		setMap.put(ds.getName(), ds);
		Collection<DataSet> sets = setsByFlowMap.get(ds.getDataFlow());
		if(sets == null){
			sets = new ArrayList<DataSet>();
			setsByFlowMap.put(ds.getDataFlow(), sets);
		}
		sets.add(ds);
		
		sets = setsBySourceMap.get(ds.getDataSource());
		if(sets == null){
			sets = new ArrayList<DataSet>();
			setsBySourceMap.put(ds.getDataSource(), sets);
		}
		sets.add(ds);
		allSets.add(ds);
		return true;
	}

	public synchronized boolean updateDataSet(DataSet ds) throws Exception {
		DataSet set = setMap.get(ds.getName());
		if(set == null) throw new Exception("Dataset with name " + ds.getName() + " does not exist.");
		setMap.put(ds.getName(), ds);
		Collection<DataSet> sets = setsByFlowMap.get(ds.getDataFlow());
		//FIXME need to update sets by flow 
		//FIXME need to update sets by source
		//FIXME need to update all sets
		return true;
	}

	public boolean registerDataSource(DataSource src) throws Exception {
		DataSource s = sourceMap.get(src.getName());
		if(s != null) throw new Exception("DataSource with name " + src.getName() + " already registered.");
		sourceMap.put(src.getName(), src);
		return true;
	}

	public boolean registerDataSourceType(DataSourceType src) throws Exception {
		DataSourceType s = sourceTypeMap.get(src.getName());
		if(s != null) throw new Exception("DataSourceType with name " + src.getName() + " already registered.");
		sourceTypeMap.put(src.getName(), src);
		return true;
	}

	public boolean registerDimensionSet(DimensionSet dims) throws Exception {
		DimensionSet s = dimSetMap.get(dims.getName());
		if(s != null) throw new Exception("DimensionSet with name " + dims.getName() + " already registered.");
		dimSetMap.put(dims.getName(), dims);
		return true;
	}

	public DataFlow getDataFlow(String name) {
		return this.flowMap.get(name);
	}

	public DataSet getDataSet(String name) {
		return this.setMap.get(name);
	}

	public DataSource getDataSource(String name) {
		return this.sourceMap.get(name);
	}

	public DataSourceType getDataSourceType(String name) {
		return this.sourceTypeMap.get(name);
	}

	public DimensionSet getDimensionSet(String name) {
		return this.dimSetMap.get(name);
	}

	public Collection<DataSet> getAllDataSets() {
		return allSets;
	}

	public Collection<DataFlow> getAllDataFlows() {
		return flowMap.values();
	}

	public Collection<DataSource> getAllDataSources() {
		return sourceMap.values();
	}

	public Collection<DataSet> getDataSetsForFlow(String flowName) {
		DataFlow flow = flowMap.get(flowName);
		if(flow == null) return EMPTY_LIST;
		return getDataSetsForFlow(flow);
	}

	public Collection<DataSet> getDataSetsForSource(String sourceName) {
		DataSource source = sourceMap.get(sourceName);
		if(source == null) return EMPTY_LIST;
		return getDataSetsForSource(source);
	}
	
	public void setDomain(Domain domain){
		this.domain = domain;
	}

	public Collection<DataSet> getDataSetsForSourceKeyset(String source,
			String keyset) throws Exception {
		// TODO Auto-generated method stub
		return null;
	}

	public DataSet getLatestDataSetForSourceKeyset(String source,
			String keyset) throws Exception {
		// TODO Auto-generated method stub
		return null;
	}

}
