/*
* Copyright (c) 2005-2007 Compuware Corporation and others.
* All rights reserved. This program and the accompanying
* materials are made available under the terms of the
* Eclipse Public License v1.0 which accompanies this
* distribution, and is available at:
* http://www.eclipse.org/legal/epl-v10.html
*
* Contributors:
*     Compuware Corporation - initial API and implementation
*
*/
package org.eclipse.cosmos.dc.local.registry;

import java.io.Reader;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.cosmos.dc.common.domain.Domain;
import org.eclipse.cosmos.dc.common.registry.DataFlow;
import org.eclipse.cosmos.dc.common.registry.DataSet;
import org.eclipse.cosmos.dc.common.registry.DataSource;
import org.eclipse.cosmos.dc.common.registry.DataSourceType;
import org.eclipse.cosmos.dc.common.registry.Dimension;
import org.eclipse.cosmos.dc.common.registry.DimensionSet;
import org.eclipse.cosmos.dc.common.registry.ProviderRegistry;
import org.eclipse.cosmos.dc.common.registry.impl.SimpleDataFlow;
import org.eclipse.cosmos.dc.common.registry.impl.SimpleDataSet;
import org.eclipse.cosmos.dc.common.registry.impl.SimpleDataSource;
import org.eclipse.cosmos.dc.common.registry.impl.SimpleDataSourceType;
import org.eclipse.cosmos.dc.common.registry.impl.SimpleDimension;
import org.eclipse.cosmos.dc.common.registry.impl.SimpleDimensionSet;
import org.eclipse.cosmos.dc.local.registry.persistence.impl.DataFlowImpl;
import org.eclipse.cosmos.dc.local.registry.persistence.impl.DataSetImpl;
import org.eclipse.cosmos.dc.local.registry.persistence.impl.DataSourceImpl;
import org.eclipse.cosmos.dc.local.registry.persistence.impl.DataSourceTypeImpl;
import org.eclipse.cosmos.dc.local.registry.persistence.impl.DimensionImpl;
import org.eclipse.cosmos.dc.local.registry.persistence.impl.KeySetEntryImpl;
import org.eclipse.cosmos.dc.local.registry.persistence.impl.KeySetImpl;
import org.eclipse.cosmos.dc.mgmt.annotations.ManagedFrameworkAutowire;
import org.eclipse.cosmos.dc.mgmt.annotations.ManagedResource;
import org.eclipse.cosmos.dc.mgmt.common.ContributionManager;

import com.ibatis.common.resources.Resources;
import com.ibatis.sqlmap.client.SqlMapClient;
import com.ibatis.sqlmap.client.SqlMapClientBuilder;

@ManagedResource(autowireName="ProviderRegistry")
public class SQLProviderRegistry implements ProviderRegistry {
	
	@ManagedFrameworkAutowire(name="Domain")
	private Domain domain;
	
	public static final Collection<DataSet> EMPTY_LIST = new ArrayList<DataSet>(); 
		
	private SqlMapClient sqlMapper;
	
	public SQLProviderRegistry(){
	}
	
	public void setManager(ContributionManager manager){
		manager.manage(this);	
		ClassLoader prevLoader = Thread.currentThread().getContextClassLoader();
		try {
			Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
			Reader reader = Resources.getResourceAsReader("org/eclipse/cosmos/dc/local/registry/persistence/sql/SqlMapConfig.xml");
			sqlMapper = SqlMapClientBuilder.buildSqlMapClient(reader);
			reader.close();
		}catch(Throwable t){
			t.printStackTrace();
		}finally{
			Thread.currentThread().setContextClassLoader(prevLoader);
		}
	}
	
	//Used for Unit testing...
	public void setSqlMapper(SqlMapClient sqlMapper){
		this.sqlMapper = sqlMapper;
	}

	public Collection<DataSet> getDataSetsForFlow(DataFlow flow){
		DataFlowImpl flowImpl = null;
		Collection<DataSet> returnList = null;
		try {
			flowImpl = (DataFlowImpl)sqlMapper.queryForObject("getDataFlowForName", flow.getName());
			if(flowImpl != null){
				Collection<DataSetImpl> sets = (Collection<DataSetImpl>)sqlMapper.queryForList("getAllDataSetsForFlow",flowImpl.getId());
				returnList = new ArrayList<DataSet>();
				for(DataSetImpl impl : sets){
					returnList.add(this.getDataSet(impl.getName()));
				}
			}
		}
		catch(SQLException se){
			se.printStackTrace();
		}catch(Throwable t){
			t.printStackTrace();
		}
		return returnList;
	}

	public Collection<DataSet> getDataSetsForSource(DataSource source) {
		DataSourceImpl sourceImpl = null;
		Collection<DataSet> returnList = null;
		try {
			sourceImpl = (DataSourceImpl)sqlMapper.queryForObject("getDataSourceForName", source.getName());
			if(sourceImpl != null){
				Collection<DataSetImpl> sets = (Collection<DataSetImpl>)sqlMapper.queryForList("getAllDataSetsForSource",sourceImpl.getId());
				returnList = new ArrayList<DataSet>();
				for(DataSetImpl impl : sets){
					DataSet ds = getDataSet(impl.getName());
					if(ds != null)
						returnList.add(ds);
				}
			}
		}
		catch(SQLException se){
			se.printStackTrace();
		}catch(Throwable t){
			t.printStackTrace();
		}
		return returnList;
	}

	public boolean registerDataFlow(DataFlow df) throws Exception {
		DataFlowImpl flow = null;
		DataSourceImpl source = null;
		KeySetImpl keys = null;
		try {
			flow = (DataFlowImpl)sqlMapper.queryForObject("getDataFlowForName", df.getName());
			if(flow != null){
				//TODO add verification code
			}else{
				source = (DataSourceImpl)sqlMapper.queryForObject("getDataSourceForName", df.getDataSource().getName());
				keys = (KeySetImpl)sqlMapper.queryForObject("getKeySetForName", df.getDimensionSet().getName());
				flow = new DataFlowImpl();
				flow.setName(df.getName());
				flow.setSource(source.getId());
				flow.setKeySet(keys.getId());
				Object setid = sqlMapper.insert("registerDataFlow", flow);
			}
		}
		catch(SQLException se){
			se.printStackTrace();
			return false;
		}catch(Throwable t){
			t.printStackTrace();
			return false;
		}
		return true;
	}

	public synchronized boolean registerDataSet(DataSet ds) throws Exception {
		DataSetImpl set = null;
		DataFlowImpl flow = null;
		DataSourceImpl src = null;
		try {
			set = (DataSetImpl)sqlMapper.queryForObject("getDataSetForName", ds.getName());
			if(set != null){
				//TODO add verification code
			}else{
				src = (DataSourceImpl)sqlMapper.queryForObject("getDataSourceForName", ds.getDataSource().getName());
				flow = (DataFlowImpl)sqlMapper.queryForObject("getDataFlowForName", ds.getDataFlow().getName());
				set = new DataSetImpl();
				set.setName(ds.getName());
				set.setSource(src.getId());
				set.setFlow(flow.getId());
				set.setStart(ds.getStart());
				set.setStop(ds.getStart());
				Object setid = sqlMapper.insert("registerDataSet", set);
			}
		}
		catch(SQLException se){
			se.printStackTrace();
			return false;
		}catch(Throwable t){
			t.printStackTrace();
			return false;
		}
		return true;
	}

	public synchronized boolean updateDataSet(DataSet ds) throws Exception {
		DataSetImpl set = null;
		DataSourceImpl src = null;
		try {
			set = (DataSetImpl)sqlMapper.queryForObject("getDataSetForName", ds.getName());
			if(set == null){
				//TODO add verification code
			}else{
				Object setid = sqlMapper.update("updateDataSet", set);
			}
		}
		catch(SQLException se){
			se.printStackTrace();
			return false;
		}catch(Throwable t){
			t.printStackTrace();
			return false;
		}
		return true;
	}

	public boolean registerDataSource(DataSource ds) throws Exception {
		DataSourceTypeImpl type = null;
		DataSourceImpl src = null;
		try {
			src = (DataSourceImpl)sqlMapper.queryForObject("getDataSourceForName", ds.getName());
			if(src != null){
				//TODO add verification code
			}else{
				type = (DataSourceTypeImpl)sqlMapper.queryForObject("getDataSourceTypeForName", ds.getDataSourceType().getName());
				src = new DataSourceImpl();
				src.setName(ds.getName());
				src.setType(type.getId());
				Object setid = sqlMapper.insert("registerDataSource", src);
			}
		}
		catch(SQLException se){
			se.printStackTrace();
			return false;
		}catch(Throwable t){
			t.printStackTrace();
			return false;
		}
		return true;
	}

	public boolean registerDataSourceType(DataSourceType src) throws Exception {
		DataSourceTypeImpl type = null;
		try {
			type = (DataSourceTypeImpl)sqlMapper.queryForObject("getDataSourceTypeForName", src.getName());
			if(type != null){
				//TODO add verification code
			}else{
				type = new DataSourceTypeImpl();
				type.setName(src.getName());
				Object setid = sqlMapper.insert("registerDataSourceType", type);
			}
		}
		catch(SQLException se){
			se.printStackTrace();
			return false;
		}catch(Throwable t){
			t.printStackTrace();
			return false;
		}
		return true;
	}

	public boolean registerDimensionSet(DimensionSet dimset) throws Exception {
		DimensionImpl impl = new DimensionImpl();
		KeySetImpl keyset = null;
		KeySetEntryImpl keysetEntry = new KeySetEntryImpl();
		try {
			keyset = (KeySetImpl)sqlMapper.queryForObject("getKeySetForName", dimset.getName());
			if(keyset != null){
				//TODO add verification code
			}else{
				keyset = new KeySetImpl();
				keyset.setName(dimset.getName());
				Object setid = sqlMapper.insert("registerKeySet", keyset);
				Collection<Dimension> dims = dimset.getDimensions();
				int i = 1;
				for(Dimension dim : dims){
					impl.setName(dim.getName());
					impl.setType(dim.getType());
					Object dimid = sqlMapper.insert("addDimension", impl);
					keysetEntry.setDimension((Integer)dimid);
					keysetEntry.setKeySet((Integer)setid);
					keysetEntry.setSequence(i++);
					Object entryid = sqlMapper.insert("addKeySetEntry", keysetEntry);
				}
			}
		}
		catch(SQLException se){
			se.printStackTrace();
			return false;
		}catch(Throwable t){
			t.printStackTrace();
			return false;
		}
		return true;
	}

	public DataFlow getDataFlow(String name) {
		DataFlowImpl flow = null;
		SimpleDataFlow retFlow = null;
		try {
			flow = (DataFlowImpl)sqlMapper.queryForObject("getDataFlowForName", name);
			if(flow != null){
				KeySetImpl keyset = (KeySetImpl)sqlMapper.queryForObject("getKeySet",flow.getKeySet());
				DataSourceImpl source = (DataSourceImpl)sqlMapper.queryForObject("getDataSource",flow.getSource());
				retFlow = new SimpleDataFlow(flow.getName(), this.getDataSource(source.getName()),this.getDimensionSet(keyset.getName())); 
			}
		}
		catch(SQLException se){
			se.printStackTrace();
		}catch(Throwable t){
			t.printStackTrace();
		}
		return retFlow;
	}

	public DataSet getDataSet(String name) {
		DataSet dataset = null;
		try {
			DataSetImpl impl = (DataSetImpl)sqlMapper.queryForObject("getDataSetForName", name);
			if(impl != null){
				DataFlowImpl df = (DataFlowImpl)sqlMapper.queryForObject("getDataFlow", impl.getFlow());
				DataFlow flow = getDataFlow(df.getName());
				DataSourceImpl ds = (DataSourceImpl)sqlMapper.queryForObject("getDataSource", impl.getSource());
				DataSource source = getDataSource(ds.getName());
				dataset = new SimpleDataSet(name,flow,source,impl.getStart(),impl.getStop());
			}
		}
		catch(SQLException se){
			se.printStackTrace();
		}catch(Throwable t){
			t.printStackTrace();
		}
		return dataset;
	}

	public DataSource getDataSource(String name) {
		DataSourceTypeImpl type = null;
		DataSourceImpl source = null;
		SimpleDataSource retType = null;
		try {
			source = (DataSourceImpl)sqlMapper.queryForObject("getDataSourceForName", name);
			if(source != null){
				type = (DataSourceTypeImpl)sqlMapper.queryForObject("getDataSourceType", source.getType());
				SimpleDataSourceType dst = new SimpleDataSourceType(type.getName());
				retType = new SimpleDataSource(name, dst);
			}
		}
		catch(SQLException se){
			se.printStackTrace();
		}catch(Throwable t){
			t.printStackTrace();
		}
		return retType;
	}

	public DataSourceType getDataSourceType(String name) {
		DataSourceTypeImpl type = null;
		SimpleDataSourceType retType = null;
		try {
			type = (DataSourceTypeImpl)sqlMapper.queryForObject("getDataSourceTypeForName", name);
			if(type != null){
				retType = new SimpleDataSourceType(name);
			}
		}
		catch(SQLException se){
			se.printStackTrace();
		}catch(Throwable t){
			t.printStackTrace();
		}
		return retType;
	}

	public DimensionSet getDimensionSet(String name) {
		try {
			KeySetImpl keySet = (KeySetImpl)sqlMapper.queryForObject("getKeySetForName", name);
			if(keySet != null){
				List<KeySetEntryImpl> entries = (List<KeySetEntryImpl>)sqlMapper.queryForList("getKeysetEntries", keySet.getId());
				ArrayList<Dimension> dims = new ArrayList<Dimension>();
				for(KeySetEntryImpl entry : entries){
					DimensionImpl dim = (DimensionImpl) sqlMapper.queryForObject("getDimension", entry.getDimension());
					Class dimClass = null;
					SimpleDimension s = new SimpleDimension(dim.getName(), dim.getType());
					dims.add(s);
				}
				SimpleDimensionSet set = new SimpleDimensionSet(keySet.getName(),dims);
				return set;
			}
		} catch (SQLException e) {
			e.printStackTrace();
		}
		return null;
	}

	public Collection<DataSet> getAllDataSets() {
		Collection<DataSet> list = EMPTY_LIST;
		try{
			List<DataSetImpl> sets = sqlMapper.queryForList("getAllDataSets");
			if(!sets.isEmpty()){
				list = new ArrayList<DataSet>();
				for(DataSetImpl set : sets){
					//FIXME really inefficient!!!
					list.add(getDataSet(set.getName()));
				}
			}
		}catch(Throwable t){
			//Oh well....
		}
		return list;
	}

	public Collection<DataFlow> getAllDataFlows() {
		Collection<DataFlow> list = null;
		try{
			List<DataFlowImpl> flows = sqlMapper.queryForList("getAllDataFlows");
			if(!flows.isEmpty()){
				list = new ArrayList<DataFlow>();
				for(DataFlowImpl flow: flows){
					//FIXME really inefficient!!!
					list.add(getDataFlow(flow.getName()));
				}
			}
		}catch(Throwable t){
			//Oh well....
		}
		return list;
	}

	public Collection<DataSource> getAllDataSources() {
		Collection<DataSource> list = null;
		try{
			List<DataSourceImpl> sources = sqlMapper.queryForList("getAllDataSources");
			if(!sources.isEmpty()){
				list = new ArrayList<DataSource>();
				for(DataSourceImpl source : sources){
					//FIXME really inefficient!!!
					list.add(getDataSource(source.getName()));
				}
			}
		}catch(Throwable t){
			//Oh well....
		}
		return list;
	}

	public Collection<DataSet> getDataSetsForFlow(String flowName) {
		DataFlow flow = getDataFlow(flowName);
		if(flow != null){
			return getDataSetsForFlow(flow);
		}
		return null;
	}

	public Collection<DataSet> getDataSetsForSource(String sourceName) {
		DataSource source = getDataSource(sourceName);
		if(source != null){
			return this.getDataSetsForSource(source);
		}
		return null;
	}
	
	public void setDomain(Domain domain){
		this.domain = domain;
	}
	
	
	private DataFlowImpl getDataFlowForSourceKeyset(String source, String keyset){
		DataSourceImpl sourceImpl = null;
		KeySetImpl keySetImpl = null;
		try {
			sourceImpl = (DataSourceImpl)sqlMapper.queryForObject("getDataSourceForName", source);
			keySetImpl = (KeySetImpl)sqlMapper.queryForObject("getKeySetForName", keyset);
		}
		catch(SQLException se){
			se.printStackTrace();
		}catch(Throwable t){
			t.printStackTrace();
		}
		try {
			Map parmMap = new HashMap();
			parmMap.put("source", sourceImpl.getId());
			parmMap.put("keySet", keySetImpl.getId());
			return (DataFlowImpl)sqlMapper.queryForObject("getDataFlowForMap", parmMap);
		}
		catch(SQLException se){
			se.printStackTrace();
		}catch(Throwable t){
			t.printStackTrace();
		}
		
		return null;
	}

	public Collection<DataSet> getDataSetsForSourceKeyset(String source,
			String keyset) throws Exception {
		DataFlowImpl flow = getDataFlowForSourceKeyset(source, keyset);
		return getDataSetsForFlow(flow.getName());
	}

	public DataSet getLatestDataSetForSourceKeyset(String source,
			String keyset) throws Exception {
		DataFlowImpl flow = getDataFlowForSourceKeyset(source, keyset);
		Collection<DataSet> returnList = null;
		try {
			Collection<DataSetImpl> sets = (Collection<DataSetImpl>)sqlMapper.queryForList("getLatestDataSetsForFlow",flow.getId());
			returnList = new ArrayList<DataSet>();
			for(DataSetImpl impl : sets){
				returnList.add(this.getDataSet(impl.getName()));
			}
		}
		catch(SQLException se){
			se.printStackTrace();
		}catch(Throwable t){
			t.printStackTrace();
		}
		if(returnList != null && !returnList.isEmpty()){
			return returnList.iterator().next();
		}
		return null;
	}

}
