/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.cmdbf.services.internal.query.transform.input.artifacts;

import java.io.StringWriter;

import org.eclipse.cosmos.dc.cmdbf.services.common.CMDBfServicesUtil;
import org.eclipse.cosmos.dc.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.IQueryTransformerConstants;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IOperator;

/**
 * Any operator element that is a child of a property value constraint
 * 
 * @author David Whiteman
 */
public class Operator implements IOperator {

	private Boolean negate = null; // optional, default to false

	/**
	 * Type of operator. Valid values can be one of: EQUALS, GREATER_THAN,
	 * LESS_THAN, GREATER_THAN_OR_EQUAL LESS_THAN_OR_EQUAL, LIKE, CONTAINS,
	 * or NOOP
	 */
	protected int type;

	/**
	 * Indicates whether case should be considered in comparison.
	 * This field is ignored for some operators.
	 */
	private Boolean caseSensitive = null;

	/**
	 * Value that is being compared against. This can be any simple type, such as String, datetime, int, or boolean.
	 */
	private Object value;

	public Operator(int type) 
	{
		this.type = type;
	}

	public boolean isNegated() {
		if (primIsNegated() == null) {
			return false;
		}
		return primIsNegated().booleanValue();
	}

	private Boolean primIsNegated() {
		return negate;
	}

	public void setNegate(boolean negate) {
		this.negate = Boolean.valueOf(negate);
	}

	public int getType() {
		return type;
	}

	public boolean isCaseSensitive() {
		if (primIsCaseSensitive() == null) {
			return true;
		}
		return primIsCaseSensitive().booleanValue();
	}

	private Boolean primIsCaseSensitive() {
		return caseSensitive;
	}

	public void setCaseSensitive(boolean caseSensitive) {
		this.caseSensitive = Boolean.valueOf(caseSensitive);
	}

	public Object getValue() {
		return value;
	}

	public void setValue(Object value) {
		this.value = value;
	}

	public void toXML(StringWriter writer, int indent) {
		CMDBfServicesUtil.addIndent(writer, indent);
		String elementName = getElementName();
		writer.write(ICMDBfServicesConstants.OPEN_ANGLE_BRACKET + ICMDBfServicesConstants.CMDBF_PREFIX + elementName);
		if (primIsCaseSensitive() != null) {
			writer.write(ICMDBfServicesConstants.SINGLE_SPACE);
			CMDBfServicesUtil.writeAttribute(writer, IQueryTransformerConstants.CASE_SENSITIVE_ATTRIBUTE, primIsCaseSensitive().toString());
		}
		if (primIsNegated() != null) {
			writer.write(ICMDBfServicesConstants.SINGLE_SPACE);
			CMDBfServicesUtil.writeAttribute(writer, IQueryTransformerConstants.NEGATE_ATTRIBUTE, primIsNegated().toString());
		}
		if (getType() == IS_NULL) {
			writer.write(ICMDBfServicesConstants.FORWARD_SLASH);
		}
		writer.write(ICMDBfServicesConstants.CLOSE_ANGLE_BRACKET);
		if (getType() != IS_NULL) {
			writer.write(getValue().toString());
			writer.write(CMDBfServicesUtil.endTagFor(ICMDBfServicesConstants.CMDBF_PREFIX + elementName));
		}
		writer.write(ICMDBfServicesConstants.nl);
	}

	private String getElementName() {
		switch (type) {
		case LESS_THAN:
			return IQueryTransformerConstants.LESS_ELEMENT;
		case LESS_THAN_OR_EQUAL:
			return IQueryTransformerConstants.LESS_OR_EQUAL_ELEMENT;
		case GREATER_THAN:
			return IQueryTransformerConstants.GREATER_ELEMENT;
		case GREATER_THAN_OR_EQUAL:
			return IQueryTransformerConstants.GREATER_OR_EQUAL_ELEMENT;
		case EQUALS:
			return IQueryTransformerConstants.EQUAL_ELEMENT;
		case LIKE:
			return IQueryTransformerConstants.LIKE_ELEMENT;
		case CONTAINS:
			return IQueryTransformerConstants.CONTAINS_ELEMENT;
		case IS_NULL:
			return IQueryTransformerConstants.IS_NULL_ELEMENT;
		default:
			return "unknown";
		}
	}

	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + (isCaseSensitive() ? 1231 : 1237);
		result = prime * result + (isNegated() ? 1231 : 1237);
		result = prime * result + type;
		result = prime * result + ((value == null) ? 0 : value.hashCode());
		return result;
	}

	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final Operator other = (Operator) obj;
		if (isCaseSensitive() != other.isCaseSensitive())
			return false;
		if (isNegated() != other.isNegated())
			return false;
		if (type != other.type)
			return false;
		if (value == null) {
			if (other.value != null)
				return false;
		} else if (!value.equals(other.value))
			return false;
		return true;
	}
}
