/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts;

import java.util.List;

import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IItemTemplate;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IRelationshipTemplate;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IGraphElement;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IInstanceId;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IItem;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IRecord;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IRecordMetadata;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IRelationship;

/**
 * Interface that allows a custom artifact factory to be supplied
 * for building the query response POJO structure
 * 
 * @provisional
 * @author David Whiteman
 */
public interface IQueryOutputArtifactFactory {

	/**
	 * Answer a new {@link IEdges} instance associated with the relationship
	 * with the given templateId
	 * 
	 * @param templateId
	 * @return
	 */
	public abstract IEdges createEdges(String templateId);

	/**
	 * Answer a new {@link IEdges} instance associated with the relationship
	 * template.
	 * 
	 * @param relationshipTemplate
	 * @return
	 */
	public abstract IEdges createEdges(
			IRelationshipTemplate relationshipTemplate);

	/**
	 * Answer a new {@link IEdges} instance associated with the relationship
	 * template, and adding an initial relationship via {@link IRelationshipConvertible}.
	 * 
	 * @param relationshipTemplate
	 * @param relationshipConvertible
	 * @return
	 */
	public abstract IEdges createEdges(
			IRelationshipTemplate relationshipTemplate,
			IRelationshipConvertible relationshipConvertible);

	/**
	 * Answer a new {@link IEdges} instance associated with the relationship
	 * template, and setting the initial list of relationship elements.
	 * 
	 * @param relationshipTemplate
	 * @param elements List of {@link IRelationship} instances
	 * @return
	 */
	public abstract IEdges createEdges(
			IRelationshipTemplate relationshipTemplate, List<?> elements);

	/**
	 * Answer a new {@link IItem} instance.
	 * 
	 * @return
	 */
	public abstract IItem createItem();

	/**
	 * Answer a new {@link INodes} instance associated with the item
	 * with the given templateId
	 * 
	 * @param templateId
	 * @return
	 */
	public abstract INodes createNodes(String templateId);

	/**
	 * Answer a new {@link INodes} instance associated with the item
	 * template, and setting the initial list of item elements.
	 * 
	 * @param itemTemplate
	 * @return
	 */
	public abstract INodes createNodes(IItemTemplate itemTemplate);

	/**
	 * Answer a new {@link INodes} instance associated with the item
	 * template, and adding an initial item via {@link IItemConvertible}.
	 * 
	 * @param itemTemplate
	 * @param itemConvertible
	 * @return
	 */
	public abstract INodes createNodes(IItemTemplate itemTemplate,
			IItemConvertible itemConvertible);

	/**
	 * Answer a new {@link INodes} instance associated with the item
	 * template, and setting the initial list of item elements.
	 * 
	 * @param itemTemplate
	 * @param elements List of {@link IItem} instances
	 * @return
	 */
	public abstract INodes createNodes(IItemTemplate itemTemplate,
			List<?> elements);

	/**
	 * Answer a new {@link IQueryResult} instance.
	 * 
	 * @return
	 */
	public abstract IQueryResult createQueryResult();

	/**
	 * Answer a new {@link IRelationship} instance.
	 * 
	 * @return
	 */
	public abstract IRelationship createRelationship();

	/**
	 * Answer a new instance of IInstanceId with the indicated attributes
	 * 
	 * @param mdrId URI string containing the MDR's ID
	 * @param localId URI string with the local ID of the instance
	 * @return
	 */
	public abstract IInstanceId createInstanceId(String mdrId, String localId);

	/**
	 * Answer a new record as a child of the indicated parent graph element.
	 * 
	 * @param parent The parent element
	 * @param recordId The record ID
	 * @return A record
	 */
	public abstract IRecord createRecord(IGraphElement parent, String recordId);

	/**
	 * Answer a new record metadata
	 * 
	 * @return A record metadata
	 */
	public abstract IRecordMetadata createRecordMetadata();

	/**
	 * Answer a new instance of {@link IItemTemplate} with the given
	 * id.
	 * 
	 * @param id
	 * @return
	 */
	public abstract IItemTemplate createItemTemplate(String id);

	/**
	 * Answer a new instance of {@link IRelationshipTemplate} with the given
	 * id.
	 * 
	 * @param id
	 * @return
	 */
	public abstract IRelationshipTemplate createRelationshipTemplate(String id);

}