/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.dc.cmdbf.services.common;

import java.util.Hashtable;
import java.util.Map;

import org.eclipse.cosmos.dc.cmdbf.services.common.CMDBfServiceException;
import org.eclipse.cosmos.dc.cmdbf.services.internal.CMDBfMessages;
import org.eclipse.osgi.util.NLS;

/**
 * An abstract notification handler that supports caching.  If
 * caching is enabled, at most one instance of the handler class
 * is created.  
 * 
 * @provisional
 * @author Ali Mehregani
 */
public abstract class AbstractNotificationHandlerFactory implements INotificationHandlerFactory
{
	/**
	 * Constants used as the key of the cache map
	 */
	private static final int ITEM = 0x00;
	private static final int RELATIONSHIP = 0x01;
	
	
	/**
	 * A cached copy of the handlers created 
	 */
	private Map<Integer, Map<Integer, INotificationHandler>> cachedHandlers;
	
	
	public AbstractNotificationHandlerFactory()
	{
		cachedHandlers = new Hashtable<Integer, Map<Integer, INotificationHandler>>();
	}
	
	/**
	 * @see org.eclipse.cosmos.dc.cmdbf.services.common.INotificationHandlerFactory#createItemHandler(int)
	 */
	public IItemNotificationHandler createItemHandler(int type) throws CMDBfServiceException
	{
		return (IItemNotificationHandler) createNotificationHandler(ITEM, type);
	}

	
	/**
	 * @see org.eclipse.cosmos.dc.cmdbf.services.common.INotificationHandlerFactory#createRelationshipHandler(int)
	 */
	public IRelationshipNotificationHandler createRelationshipHandler(int type) throws CMDBfServiceException 
	{
		return (IRelationshipNotificationHandler) createNotificationHandler(RELATIONSHIP, type);
	}
	
	
	private INotificationHandler createNotificationHandler(int entityType, int notificationType) throws CMDBfServiceException
	{
		// Check the cache first
		if (isCachingSupported())
		{
			Map<Integer, INotificationHandler> nestedMap = cachedHandlers.get(entityType);
			INotificationHandler handler = nestedMap == null ? null : nestedMap.get(notificationType);
			if (handler != null)
			{
				return handler;
			}
		}
		
		INotificationHandler handler = null;
		switch (entityType)
		{
			case ITEM:
				switch (notificationType)
				{
					case INotificationHandler.REGISTRATION:
						handler = createRegistrationItemHandler();
						break;
					case INotificationHandler.DEREGISTRATION:
					handler = createDeregistrationItemHandler();
						break;
				}
				break;
			case RELATIONSHIP:
				switch (notificationType)
				{
					case INotificationHandler.REGISTRATION:
						handler = createRegistrationRelationshipHandler();
						break;
					case INotificationHandler.DEREGISTRATION:
						handler = createDeregistrationRelationshipHandler();
						break;
				}
				break;
		}
		
		if (handler != null && isCachingSupported())
		{
			Map<Integer, INotificationHandler> nestedMap = cachedHandlers.get(entityType);
			if (nestedMap == null)
			{
				nestedMap = new Hashtable<Integer, INotificationHandler>();
				cachedHandlers.put(entityType, nestedMap);
			}
			nestedMap.put(notificationType, handler);
		}
		
		return handler;
	}
	
	
	/**
	 * Returns an item handler for registration requests.  Subclasses
	 * can override this method if this type of handler is supported.
	 * 
	 * @return An instance of a registration item handler
	 */
	protected IItemNotificationHandler createRegistrationItemHandler() throws CMDBfServiceException
	{
		throw new CMDBfServiceException (
				CMDBfServiceException.SENDER,
				CMDBfServiceException.REGISTRATION_ERROR,
				NLS.bind (CMDBfMessages.unsupportedHandler, new String[]
                {
					CMDBfMessages.serviceRegistration,
					CMDBfMessages.item
                }));
	}

	
	/**
	 * Returns a relationship handler for registration requests.  Subclasses
	 * can override this method if this type of handler is supported.
	 * 
	 * @return An instance of a registration relationship handler
	 */
	protected IRelationshipNotificationHandler createRegistrationRelationshipHandler() throws CMDBfServiceException
	{
		throw new CMDBfServiceException (
				CMDBfServiceException.SENDER,
				CMDBfServiceException.REGISTRATION_ERROR,
				NLS.bind (CMDBfMessages.unsupportedHandler, new String[]
                {
					CMDBfMessages.serviceRegistration,
					CMDBfMessages.relationship
                }));
	}
	
	
	/**
	 * Returns an item handler for deregistration requests.  Subclasses
	 * can override this method if this type of handler is supported.
	 * 
	 * @return An instance of a deregistration item handler
	 */
	protected IItemNotificationHandler createDeregistrationItemHandler() throws CMDBfServiceException
	{
		throw new CMDBfServiceException (
				CMDBfServiceException.SENDER,
				CMDBfServiceException.REGISTRATION_ERROR,
				NLS.bind (CMDBfMessages.unsupportedHandler, new String[]
                {
					CMDBfMessages.serviceDeregistration,
					CMDBfMessages.item
                }));
	}

	
	/**
	 * Returns an relationship handler for registration requests.  Subclasses
	 * can override this method if this type of handler is supported.
	 * 
	 * @return An instance of a registration relationship handler
	 */
	protected IRelationshipNotificationHandler createDeregistrationRelationshipHandler() throws CMDBfServiceException
	{
		throw new CMDBfServiceException (
				CMDBfServiceException.SENDER,
				CMDBfServiceException.REGISTRATION_ERROR,
				NLS.bind (CMDBfMessages.unsupportedHandler, new String[]
                {
					CMDBfMessages.serviceDeregistration,
					CMDBfMessages.relationship
                }));
	}

	/**
	 * By default caching is enabled.  Subclasses can override
	 * this method to disable caching.
	 * 
	 * @return Indicates whether caching is enabled
	 */
	protected boolean isCachingSupported()
	{
		return true;
	}
}
