/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.dc.cmdbf.services.common;

import org.eclipse.cosmos.dc.cmdbf.services.cmdbfservice.AbstractServiceHandler;
import org.eclipse.cosmos.dc.cmdbf.services.cmdbfservice.IQueryHandler;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.input.artifacts.IRequest;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.output.artifacts.IRegistrationServiceOutputArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.output.artifacts.IResponse;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IGraphElement;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IGraphElementCollection;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IItem;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IRelationship;

/**
 * An abstract class used for the registration and deregistration
 * operations.  This class contains the common ground between the
 * two operations
 * 
 * @provisional
 * @author Ali Mehregani
 */
public abstract class AbstractNotificationOperation extends AbstractServiceHandler
{
	/**
	 * The registration handler factory
	 */
	private INotificationHandlerFactory factory;
	
	/**
	 * The factory for creating the response artifacts
	 */
	private IRegistrationServiceOutputArtifactFactory outputArtifactFactory;

	/**
	 * The constructor
	 * 
	 * @param factory The factory class that will be used to create 
	 * registration handlers
	 */
	public AbstractNotificationOperation (INotificationHandlerFactory factory, IRegistrationServiceOutputArtifactFactory outputArtifactFactory)
	{
		this.factory = factory;
		this.outputArtifactFactory = outputArtifactFactory;
	}
	
	
	/**
	 * Returns the handler factory associated with this
	 * operation.
	 *  
	 * @return
	 */
	protected INotificationHandlerFactory getFactory()
	{
		return factory;
	}
	
	
	/**
	 * This execute method can be shared between the registration and
	 * the deregistration operation.  The method will first process
	 * the items in a request followed by the relationships.
	 * 
	 * @param request The request
	 * @param response The response
	 * @throws CMDBfServiceException 
	 */
	protected void execute (IRequest request, IResponse response, int type) throws CMDBfServiceException
	{
		// Handle the items
		handleGraphElementCollection(request.getItemList(), response, type, true);
		
		// Handle the relationships
		handleGraphElementCollection (request.getRelationshipList(), response, type, false);
	}
	
	private void handleGraphElementCollection(IGraphElementCollection collection, IResponse response, int type, boolean item) throws CMDBfServiceException
	{
		if (collection == null || collection.isEmpty())
		{
			return;
		}
		
		// Create the item registration request
		IItemNotificationHandler itemHandler = null;
		IRelationshipNotificationHandler relationshipHandler = null;
		if (item)
		{
			itemHandler = getFactory().createItemHandler(type);
		}
		else
		{
			relationshipHandler = getFactory().createRelationshipHandler(type);
		}		
		IQueryHandler serviceHandler = itemHandler == null ? relationshipHandler : itemHandler; 		
		if (!serviceHandler.isInitialized())
		{
			serviceHandler.initialize(getInit());
		}
		
		
		IGraphElement[] elements = collection.getElements();
		for (int i = 0; i < elements.length; i++)
		{
			response.addInstanceResponse(itemHandler == null ?
					relationshipHandler.handle(type, (IRelationship)elements[i]) :
					itemHandler.handle(type, (IItem)elements[i]));
		}	
	}


	protected IRegistrationServiceOutputArtifactFactory getOutputArtifactFactory() {
		if (outputArtifactFactory == null) {
			outputArtifactFactory = createArtifactFactory();
		}
		return outputArtifactFactory;
	}


	protected abstract IRegistrationServiceOutputArtifactFactory createArtifactFactory();
}
