/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.validation.internal.artifacts;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.cosmos.rm.validation.internal.common.IValidationConstants;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.validation.internal.databuilders.TypeInheritanceDataBuilderImpl;

/**
 * A document node is used to represent an SML-IF document 
 * 
 * @author Ali Mehregani
 */
public class DocumentNode
{
	/**
	 * The aliases of this document
	 */
	private List<String> aliases;
	
	/**
	 * The edges of this document node are indexed by reference
	 * type.
	 * 
	 * KEY = A {@link String} indicating the uri
	 * VALUE = A {@link Map} where,
	 * 	KEY = A {@link String} indicating the local name of the type	
	 * 	VALUE = A {@link List} storing all edges of type {@link ElementEdge}.
	 */
	private Map<String, Map<String, List<ElementEdge>>> edges;
	
	
	public DocumentNode()
	{
		edges = new Hashtable<String, Map<String, List<ElementEdge>>>();
	}
	
	/**
	 * Returns the aliases of this document
	 * 
	 * @return The aliases of this document node
	 */
	public String[] getAliases()
	{
		if (aliases == null)
			return null;
		
		return (String[])aliases.toArray(new String[aliases.size()]);
	}

	
	/**
	 * Add an alias for this document node.  An alias is used to reference
	 * a document.
	 * 
	 * @param alias The alias to be added
	 */
	public void addAlias(String alias)
	{
		if (aliases == null)
			aliases = new ArrayList<String>();
		
		aliases.add(alias);
	}
	
	
	/**
	 * Adds an element edge to this document
	 * 
	 * @param edge The element edge
	 */
	@SuppressWarnings("unchecked")
	public void addEdge(ElementEdge edge)
	{
		if (edge == null || edge.getReferenceType() == null)
			return;
		
		if (edges == null)
			edges = new Hashtable<String, Map<String, List<ElementEdge>>>();
		
		String uri = edge.getReferenceNameSpace();
		uri = uri == null ? IValidationConstants.EMPTY_STRING : uri;
		
		Map namesPerUri = SMLValidatorUtil.retrieveNestedMap(edges, uri, true);

		List<ElementEdge> referenceEdges = (List<ElementEdge>)namesPerUri.get(edge.getReferenceType());
		if (referenceEdges == null)
			referenceEdges = new ArrayList<ElementEdge>();
		
		referenceEdges.add(edge);
		namesPerUri.put(edge.getReferenceType(), referenceEdges);
	}
	
		
	/**
	 * Return the edges for the reference type passed in.  A valid inheritence tree
	 * must be registered with the data builder for the edges derived from referenceType 
	 * to be returned.
	 * 
	 * @param uri The namespace of the type
	 * @param localName The local name of the type
	 * @return The edges corresponding to the reference type passed in
	 */
	@SuppressWarnings("unchecked")
	public ElementEdge[] getEdges(String uri, String localName)
	{		
		Map inheritanceMap = (Map)SMLValidatorUtil.retrieveDataStructure(TypeInheritanceDataBuilderImpl.ID);
		
		/* Create an empty map */
		if (inheritanceMap == null)
			inheritanceMap = new Hashtable<String, Map<String, TypeNode>>();
		
		List<ElementEdge> referenceEdges = new ArrayList<ElementEdge>();
		TypeNode refType = new TypeNode(localName, uri);
		TypeNode refTypeInQuestion = new TypeNode();
		for (Iterator<String> uriList = edges.keySet().iterator(); uriList.hasNext();)
		{
			String currentURI = uriList.next();
			Map<String, List<ElementEdge>> namesPerUri = edges.get(currentURI);
			
			for (Iterator<String> typeList = namesPerUri.keySet().iterator(); typeList.hasNext();)
			{
				String currentType = (String)typeList.next();			
				refTypeInQuestion.setUri(currentURI);
				refTypeInQuestion.setType(currentType);

				if (TypeInheritanceDataBuilderImpl.isDerivedType(inheritanceMap, refType, refTypeInQuestion))
				{
					referenceEdges.addAll(namesPerUri.get(currentType));
				}
				
			}
		}
		
		return referenceEdges.toArray(new ElementEdge[referenceEdges.size()]);
	}

}
