/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.validation.internal.artifacts;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.Path;

/**
 * Model object to track instances found in validating SML resources.
 * It is used by the SchemaValidator in attempt to preserve the original
 * source to improve the accuracy of line number information in
 * validation error messages.
 * 
 * @author David Whiteman
 */
public class ElementModel extends ElementLocation {

	private String source;
	private String elementName;
	private int endingLine;
	private int endingColumn;
	private String additionalSchemaLocation;
	
	/**
	 * A computed value containing the list of definition aliases
	 * required to validate this instance
	 * As a first step, find all schemaBindings where any of the element's aliases
	 * is bound to a schema.
	 * Second, add all defaultSchema bindings schemas as defined in the defaultSchemaBinding element
	 * Last, for all schemas found in step one, get all imported or included schemas
	 * Note: if the schemaBindings is missing completely, step 1 and 2 are replaced by namespace
	 * match ( find all schemas with a targetNamespace = ns where ns is used by this instance
	 * 
	 */
	private List<String> calculatedValidationSchemas;
	
	/**
	 * the aliases of the definition document
	 */
	private List<String> aliases;
	

	public ElementModel(String name, String filePath, int lineNumber, int columnNumber) {
		super(filePath, lineNumber, columnNumber);
		this.elementName = name;
		
		aliases = new ArrayList<String>();		
	}

	public int getEndingLine() {
		return endingLine;
	}

	public void setEndingLine(int endingLine) {
		this.endingLine = endingLine;
	}

	public String getSource() {
		return source;
	}

	public void setSource(String contents) {
		this.source = contents;
	}

	public String getElementName() {
		return elementName;
	}

	public int getEndingColumn() {
		return endingColumn;
	}

	public void setEndingColumn(int columnNumber) {
		this.endingColumn = columnNumber;
	}

	public int numberOfLines() {
		if (getLineNumber() > getEndingLine()) {
			return 0;
		}
		return getEndingLine() - getLineNumber() + 1;
	}

	public String toString() {
		return getClass().getSimpleName() + " <"+elementName+"> line="+getLineNumber()+", col="+getColumnNumber();
	}

	public void setAdditionalSchemaLocation(String string) {
		additionalSchemaLocation = string;
	}

	public String getAdditionalSchemaLocation() {
		return additionalSchemaLocation;
	}
	
	public boolean equals(Object o)
	{
		if (!(o instanceof ElementModel))
			return false;
		
		ElementModel elementModel = (ElementModel)o;		
		return 	elementName.equals(elementModel.getElementName()) &&
				new Path (getFilePath()).equals(new Path(elementModel.getFilePath())) &&
				source.equals(elementModel.getSource());
	}
	
	/**
	 * @return the aliases
	 */
	public List<String> getAliases() {
		return aliases;
	}

	/**
	 * @param aliases the aliases to set
	 */
	public void setAliases(List<String> aliases) {
		this.aliases.addAll(aliases);
	}

	/**
	 * @return the calculatedValidationSchemas
	 */
	public List<String> getCalculatedValidationSchemas() {
		return calculatedValidationSchemas;
	}

	/**
	 * @param calculatedValidationSchemas the calculatedValidationSchemas to set
	 */
	public void setCalculatedValidationSchemas(List<String> calculatedValidationSchemas) {
		this.calculatedValidationSchemas = calculatedValidationSchemas;
	}
	
}
