/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.validation.internal.artifacts;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.MissingResourceException;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.cosmos.rm.validation.internal.SMLActivator;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidationMessages;
import org.eclipse.osgi.util.NLS;


/**
 * A resource wrapper is used to represent a resource inside or outside of 
 * the Eclipse environment
 * 
 * @author Ali Mehregani
 */
public class ResourceWrapper
{

	/**
	 * This field is set iff the validator is running 
	 * inside the Eclipse environment
	 */
	private IResource eclipseResource;
	
	/**
	 * This field is set iff the validator is running
	 * outside the Eclipse environment 
	 */
	private File resource;

	/**
	 * True iff the validator is running inside the Eclipse 
	 * environment
	 */
	private boolean inEclipse;
	
		
	/**
	 * The constructor
	 *  
	 * @param path The path of the resource
	 * @param inEclipse Must be true iff running in Eclipse
	 */
	public ResourceWrapper(String path, boolean inEclipse)
	{
		this.inEclipse = inEclipse;
		if (inEclipse)
		{
			IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
			IPath resourcePath = new Path(path);
			eclipseResource = root.getFile(new Path(path));
			eclipseResource = eclipseResource != null && eclipseResource.exists() ? eclipseResource : root.getFileForLocation(resourcePath);
			eclipseResource = eclipseResource != null && eclipseResource.exists() ? eclipseResource : root.getFolder(new Path(path));		
			if (!eclipseResource.exists())
			{ 
				throw new MissingResourceException(NLS.bind(SMLValidationMessages.errorMissingFile, path), null, null);
			}			
		}
		else
		{
			resource = new File(path);
			if (!resource.exists())
			{
				throw new MissingResourceException(NLS.bind(SMLValidationMessages.errorMissingFile, path), null, null);
			}
		}
	}


	public ResourceWrapper(IResource eclipseResource)
	{
		this.eclipseResource = eclipseResource;
		this.inEclipse = true;
	}

	
	public ResourceWrapper(File resource)
	{
		this.resource = resource;
		this.inEclipse = false;
	}
	
	
	public boolean isDirectory()
	{
		return inEclipse ? eclipseResource instanceof IContainer : resource.isDirectory();
	}


	public ResourceWrapper[] getChildren()
	{
		ResourceWrapper[] childrenWrapped = new ResourceWrapper[0];
		if (isDirectory())
		{
			if (inEclipse)
			{
				IResource[] children;
				try
				{
					children = ((IContainer)eclipseResource).members();
					childrenWrapped = new ResourceWrapper[children.length]; 
					for (int i = 0; i < children.length; i++)
					{
						childrenWrapped[i] = new ResourceWrapper(children[i]);
					}
				}
				catch (CoreException e)
				{
					childrenWrapped = new ResourceWrapper[0];
					SMLActivator.logError(e);
				}
			}
			else
			{
				File[] children = resource.listFiles();
				childrenWrapped = new ResourceWrapper[children.length]; 
				for (int i = 0; i < children.length; i++)
				{
					childrenWrapped[i] = new ResourceWrapper(children[i]);
				}
			}
		}
		
		return childrenWrapped;
	}


	public String getName()
	{
		return inEclipse ? eclipseResource.getName() : resource.getName();
	}


	public boolean isFile()
	{
		return inEclipse ? eclipseResource instanceof IFile : resource.isFile();
	}


	public InputStream getContents()
	{
		if (!isFile())
			return null;
		
		try
		{
			if (!inEclipse) {
				return new FileInputStream(resource);
			}
			// Need to handle the case where we are in Eclipse, but have opened
			// an external file
			if (((IFile)eclipseResource).exists()) {
				return ((IFile)eclipseResource).getContents();
			} else {
				return new FileInputStream(eclipseResource.getFullPath().toOSString());
			}
		} 
		catch (FileNotFoundException e)
		{		
			e.printStackTrace();
		} 
		catch (CoreException e)
		{
			SMLActivator.logError(e);
		}
		return null;
	}


	public String getFullPath()
	{
		return inEclipse ? eclipseResource.getFullPath().toString() : resource.getPath();
	}


	public String getAbsolutePath()
	{
		if (!inEclipse) {
			return resource.getAbsolutePath();
		}
		// Checking to see if the resource exists in the workspace,
		// or is an external file
		if (eclipseResource.exists()) {
			return eclipseResource.getRawLocation().toString();
		} else {
			return eclipseResource.getFullPath().toOSString();
		}
	}


	public boolean exists()
	{
		return inEclipse ? eclipseResource.exists() : resource.exists();
	}


	public Object getResource()
	{
		return inEclipse ? (Object)eclipseResource : resource;
	}
	
	public boolean equals(Object o)
	{
		if (!(o instanceof ResourceWrapper) || getResource() == null)
			return false;
		
		return getResource().equals(((ResourceWrapper)o).getResource());
	}
	
	public int hashCode() {
		return getResource().hashCode();
	}

	public String toString() {
		return "ResourceWrapper: " + getFullPath();
	}
}
