/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.validation.internal.artifacts;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * A data structure that stores the schematron rule bindings
 * 
 * @author Ali Mehregani
 */
public class RuleBindings
{
	/**
	 * Used to store untargeted schematron nodes
	 */
	private static final String UNTARGETED = "untargeted";
		
	/**
	 * The data structure associated with this builder
	 * KEY = taget namespace of the schema containing the schematron rule OR {@link #UNTARGETED}
	 * VALUE = An object of type {@link Map} that contains:
	 * 	KEY = The document alias, the element name, or the element type
	 * 	VALUE = The schematron node that should be applied to the element
	 */
	protected Map<String, Map<String, List<Schematron>>> data;

	/**
	 * Globally bound rules
	 * (i.e. rules bound to every document in the model)
	 */
	private List<Schematron> globallyBoundRules;

	
	public RuleBindings()
	{
		data = new HashMap<String, Map<String, List<Schematron>>>();
		globallyBoundRules = new ArrayList<Schematron>();
	}
	
	
	/**
	 * Binds a rule with the document alias passed in
	 * 
	 * @param alias The document alias
	 * @param schematronRule The schematron rule
	 */
	public void bindRule(String alias, Schematron schematronRule)
	{
		bindRule (UNTARGETED, alias, schematronRule);
	}
	
	
	/**
	 * Binds a rule with a namespace and a localname
	 * 
	 * @param uri The namespace
	 * @param localName The local name
	 * @param schematronRule The schematron rule
	 */
	public void bindRule (String uri, String localName, Schematron schematronRule)
	{
		Map<String, List<Schematron>> schematronRuleMappings = null;		
		
		if ((schematronRuleMappings = data.get(uri)) == null)
		{
			schematronRuleMappings = new HashMap<String, List<Schematron>>();
			data.put(uri, schematronRuleMappings);
		}
		// A local name maps to a list of schematron rules, as there can
		// be more than one rule mapped to an instance document
		// https://bugs.eclipse.org/bugs/show_bug.cgi?id=179106
		List<Schematron> rules = schematronRuleMappings.get(localName);
		if (rules == null) {
			rules = new ArrayList<Schematron>();
			schematronRuleMappings.put(localName, rules);
		}
		rules.add(schematronRule);
	}
	
	
	/**
	 * Adds a globally bound rule
	 * 
	 * @param schematronRule The rule
	 */
	public void addGloballyBoundRule(Schematron schematronRule)
	{
		globallyBoundRules.add(schematronRule);
	}
	
	
	/**
	 * Returns a rule that's been bound to an alias via the ruleBindings
	 * element
	 * 
	 * @param alias The alias
	 * @return The list of associated rules
	 */
	public List<Schematron> getBoundRules (String alias)
	{
		return getBoundRules(UNTARGETED, alias);
	}
	
	
	/**
	 * Returns a rule that's been bound to an element name or a element 
	 * type declaration.
	 *  
	 * @param uri The URI
	 * @param localName The local name
	 * @return The list of schematron rules
	 */
	public List<Schematron> getBoundRules (String uri, String localName)
	{
		Map<String, List<Schematron>> schematronRules = data.get(uri);
		if (schematronRules == null)
			return null;
		
		return schematronRules.get(localName);
	}
	
	/**
	 * Returns the list of the rules bound to all documents in the model.
	 * This occurs when a ruleBinding element is encountered in a SML-IF
	 * document that specifies a ruleAlias but does not specify a documentAlias.
	 * 
	 * @return The globally bound rules
	 */
	public List<Schematron> getGloballyBoundRules()
	{
		return globallyBoundRules;
	}
	
	/**
	 * Returns true if the SML-IF document being parsed contains a
	 * ruleBindings element
	 * 
	 * @return true if the ruleBindings element is present; false
	 * otherwise
	 */
	public boolean isRuleBindingPresent()
	{						
		return (data.size() > 0);
	}
	
}
