/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.validation.internal.artifacts;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.eclipse.cosmos.rm.validation.internal.common.AbstractValidationOutput.ValidationMessage;

/**
 * Artifact that contains schematron information
 * 
 * @author sleeloy
 * @author Ali Mehregani
 */
public class Schematron
{
	/**
	 * Stores the schematron fragments, excluding the patterns
	 */
	private StringBuffer fragment;

	/**
	 * Contains the rules associated with a schematron
	 */
	private List<Pattern> patterns;

	/**
	 * The prefix map
	 */
	private Map<String, String> prefixMap;

	private int lineNumber = ValidationMessage.NO_LINE_NUMBER;

	public Schematron(StringBuffer schematron, List<Pattern> patterns, Map<String, String> prefixMap, int lineNumber)
	{
		super();
		this.fragment = schematron;
		this.patterns = new ArrayList<Pattern>(patterns.size());
		this.patterns.addAll(patterns);		
		this.prefixMap = prefixMap;
		this.lineNumber = lineNumber;
	}

	public List<Pattern> getPatterns()
	{
		return patterns;
	}


	/**
	 * Returns XML fragment that represents the schematron
	 * 
	 * @return XML fragment that represents the schematron
	 */
	public StringBuffer getFragment()
	{
		return fragment;
	}
	
	public int getLineNumber()
	{
		return lineNumber;
	}

	/**
	 * Sets XML fragment that represents the schematron
	 * 
	 * @param schematron
	 *            XML fragment that represents the schematron
	 */
	public void setFragment(StringBuffer schematron)
	{
		this.fragment = schematron;
	}

	/**
	 * Returns a prefix map used to resolve namespace uris
	 * 
	 * @return a prefix map used to resolve namespace uris
	 */
	public Map<String, String> getPrefixMap()
	{
		return prefixMap;
	}

	/**
	 * Sets prefix map used to resolve namespace uris
	 * 
	 * @param prefixMap
	 *            prefix map used to resolve namespace uris
	 */
	public void setPrefixMap(Map<String, String> prefixMap)
	{
		this.prefixMap = prefixMap;
	}
	
	
	public static class Pattern
	{
		private StringBuffer fragment;
		private List<Rule> rules;

		public Pattern()
		{
			rules = new ArrayList<Rule>();
			fragment = new StringBuffer();
		}
		
		
		public Pattern(List<Rule> rules)
		{
			this.rules = new ArrayList<Rule>(rules.size());
			this.rules.addAll(rules);			
		}

		/**
		 * @return the rules
		 */
		public List<Rule> getRules()
		{
			return rules;
		}

		/**
		 * @param rules the rules to set
		 */
		public void setRules(List<Rule> rules)
		{
			this.rules = rules;
		}

		public void addRule(Rule rule)
		{
			rules.add(rule);
		}


		/**
		 * @return the fragment
		 */
		public StringBuffer getFragment()
		{
			return fragment;
		}


		/**
		 * @param fragment the fragment to set
		 */
		public void setFragment(StringBuffer fragment)
		{
			this.fragment = fragment;
		}		
		
		
	}
	
	public static class Rule
	{
		private String context;
		private StringBuffer fragment;
		private boolean matchContainsDeref;
		private int lineNumber = ValidationMessage.NO_LINE_NUMBER;
		
		public Rule(String context, StringBuffer fragment)
		{
			this.context = context;
			this.fragment = fragment;
		}
		
		public Rule()
		{
			fragment = new StringBuffer();
		}
		
		/**
		 * @return the context
		 */
		public String getContext()
		{
			return context;
		}
		/**
		 * @param context the context to set
		 */
		public void setContext(String context)
		{
			this.context = context;
		}
		/**
		 * @return the fragment
		 */
		public StringBuffer getFragment()
		{
			return fragment;
		}
		
		public int getLineNumber() {
			return lineNumber;
		}
		
		/**
		 * @param fragment the fragment to set
		 */
		public void setFragment(StringBuffer fragment)
		{
			this.fragment = fragment;
		}
		
		public boolean getMatchContainsDeref()
		{
			return matchContainsDeref;
		}
		
		public void setMatchContainsDeref(boolean matchContainsDeref)
		{
			this.matchContainsDeref = matchContainsDeref;
		}

		public void setLineNumber(int lineNumber) {
			this.lineNumber = lineNumber;
		}
	}
}
