/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.validation.internal.common;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.MissingResourceException;

import org.eclipse.cosmos.rm.validation.internal.artifacts.ResourceWrapper;
import org.eclipse.cosmos.rm.validation.internal.core.IValidator;

/**
 * A validation output is used to write messages while performing 
 * the validation process.  
 * 
 * @author Ali Mehregani
 */
public abstract class AbstractValidationOutput implements IValidationOutput<String, Object>
{	
	private ResourceWrapper associatedResource;
	
	private int errorCount = 0;
	
	private List<IValidationMessage> previousMessages = new ArrayList<IValidationMessage>();
	
	/**
	 * Indicates whether warnings should be suppressed   
	 */
	private boolean suppressWarnings;
	
	/**
	 * @see org.eclipse.cosmos.rm.validation.internal.common.IValidationOutput#initialize(java.util.Map)
	 */
	public void initialize(Map<String, Object> attributes)
	{
		errorCount = 0;
		previousMessages.clear();
		String fileName = (String) attributes.get(IValidator.ATTRIBUTE_INSTANCE);
		if (fileName != null)
		{
			try
			{
				suppressWarnings = attributes.get(IValidator.ATTRIBUTE_SUPPRESS_WARNINGS) == null ? false :
					((Boolean)attributes.get(IValidator.ATTRIBUTE_SUPPRESS_WARNINGS)).booleanValue();
				associatedResource = new ResourceWrapper(fileName, IValidator.VALUE_ENV_ECLIPSE.equals(attributes.get(IValidator.ATTRIBUTE_ENV)));
			}
			catch (MissingResourceException mre)
			{
				associatedResource = null;
			}
		}
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.validation.internal.common.IValidationOutput#reportMessage(org.eclipse.cosmos.rm.validation.internal.common.AbstractValidationOutput.ValidationMessage)
	 */
	public void reportMessage(IValidationMessage validationMessage)
	{
		if (validationMessage == null)
			return; 
		
		int severity = validationMessage.getAttribute(IValidationMessage.ATTRIBUTE_SEVERITY, -1);		
		switch (severity)
		{
			case IValidationMessage.SEVERITY_ERROR:
			{
				errorCount++;
				break;
			}
			case IValidationMessage.SEVERITY_WARNING:
			{
				if (suppressWarnings)
				{
					return;
				}
				break;
			}
		}
			
		
		if (associatedResource != null && validationMessage.getAttribute(ValidationMessage.ATTRIBUTE_RESOURCE, null) == null) 
			validationMessage.setAttribute(ValidationMessage.ATTRIBUTE_RESOURCE, associatedResource);		
		
		// Prevent duplicate messages that can occur with the some of the
		// validation that processes element node references
		// https://bugs.eclipse.org/bugs/show_bug.cgi?id=188975
		if (previousMessages.contains(validationMessage)) {
			return;
		}
		previousMessages.add(validationMessage);
		
		writeMessageToSource(validationMessage);		
	}
	
	/**
	 * The final method invoked to write the validation message to
	 * the output source.
	 * 
	 * @param validationMessage The validation message
	 */
	public abstract void writeMessageToSource(IValidationMessage validationMessage);
	
	
	/**
	 * @see org.eclipse.cosmos.rm.validation.internal.common.IValidationOutput#close()
	 */
	public void close()
	{
		
	}
	
	public int getErrorCount() 
	{
		return errorCount;
	}
	
	/**
	 * Represents a verification message, which is simply described as a set
	 * of unstructured attributes.
	 * 
	 * @author Ali Mehregani
	 */
	public static class ValidationMessage implements IValidationMessage
	{
		/**
		 * Indicates a severity of info
		 */
		public static final int SEVERITY_INFO = 0;
		
		/**
		 * Indicates a severity of warning
		 */
		public static final int SEVERITY_WARNING = 1;
		
		/**
		 * Indicates a severity of error
		 */
		public static final int SEVERITY_ERROR = 2;
		
		/**
		 * The message attribute.  The value of this attribue is expected
		 * to be a {@link String}
		 */
		public static final String ATTRIBUTE_MESSAGE = "attribute.message";
		
		/**
		 * The severity attribute.  The value of this attribute is expected
		 * to be one of: <br/>
		 * SEVERITY_INFO <br/>
		 * SEVERITY_WARNING <br/>
		 * SEVERITY_ERROR <br/>
		 */
		public static final String ATTRIBUTE_SEVERITY = "attribute.severity";
		
		/**
		 * An object of type IResource indicating the resource associated with this message
		 */
		public static final String ATTRIBUTE_RESOURCE = "attribute.resource";
		
		/**
		 * An integer indicating the line number associated with this message
		 */
		public static final String ATTRIBUTE_LINE_NUMBER = "attribute.line.number";
		
		/**
		 * Value indicating no line number has been supplied to the validation message
		 */
		public static final int NO_LINE_NUMBER = -1;
		
		/**
		 * The attributes of the message
		 */
		private Map<String, Object> attributes; 
				
		
		public ValidationMessage()
		{
			attributes = new Hashtable<String, Object>();
		}
				
		/**
		 * Set a string attribute
		 * 
		 * @param key The key 
		 * @param value The value
		 */
		public void setAttribute(String key, String value)
		{
			setAttribute(key, (Object)value);
		}
		
		/**
		 * Set an integer attribute
		 * 
		 * @param key The key 
		 * @param value The value
		 */
		public void setAttribute(String key, int value)
		{
			setAttribute(key, new Integer(value));
		}		
		
		/**
		 * Set a boolean attribute
		 * 
		 * @param key The key 
		 * @param value The value
		 */
		public void setAttribute(String key, boolean value)
		{
			setAttribute(key, new Boolean(value));
		}		
		
		/**
		 * Set an object attribute
		 * 
		 * @param key The key 
		 * @param value The value
		 */
		public void setAttribute(String key, Object value)
		{
			if (key == null || value == null)
				return;
			
			attributes.put(key, value);	
		}
		
		
		/**
		 * Retrieve an integer attribute
		 * 
		 * @param key The key 
		 * @param defaultValue The default value
		 */
		public int getAttribute(String key, int defaultValue)
		{
			Object value = attributes.get(key);
			if (value instanceof Integer)
				return ((Integer)value).intValue();
			return defaultValue;
		}		
		
		/**
		 * Retrieve a boolean attribute
		 * 
		 * @param key The key 
		 * @param defaultValue The default value
		 */
		public boolean getAttribute(String key, boolean defaultValue)
		{
			Object value = attributes.get(key);
			if (value instanceof Boolean)
				return ((Boolean)value).booleanValue();
			return defaultValue;
		}
				
		/**
		 * Retrieve a string attribute
		 * 
		 * @param key The key 
		 * @param defaultValue The default value
		 */
		public String getAttribute(String key, String defaultValue)
		{
			Object value = attributes.get(key);
			if (value instanceof String)
				return (String)value;
			return defaultValue;
		}
		
		/**
		 * Retrieve an object attribute
		 * 
		 * @param key The key 
		 */
		public Object getAttribute(String key)
		{
			return attributes.get(key);
		}
		
		public String toString() 
		{
			StringBuffer buffer = new StringBuffer();
			buffer.append("MSG: "+getAttribute(ValidationMessage.ATTRIBUTE_MESSAGE, "<none>"));
			return buffer.toString();
		}

		public int hashCode() {
			final int PRIME = 31;
			int result = 1;
			result = PRIME * result + ((attributes == null) ? 0 : attributes.hashCode());
			return result;
		}

		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (!(obj instanceof ValidationMessage))
				return false;
			final ValidationMessage other = (ValidationMessage) obj;
			if (attributes == null) {
				if (other.attributes != null)
					return false;
			} else if (!attributes.equals(other.attributes))
				return false;
			return true;
		}
	}
	
	/**
	 * The validation factory class is used to generate validation
	 * messages.
	 * 
	 * @author Ali Mehregani
	 */
	public static class ValidationMessageFactory
	{
		public static IValidationMessage createErrorMessage(String filePath, int lineNumber, String message)
		{
			return createMessage(IValidationMessage.SEVERITY_ERROR, filePath, lineNumber, message);
		}
		
		public static IValidationMessage createErrorMessage(int lineNumber, String message)
		{
			return createMessage(IValidationMessage.SEVERITY_ERROR, null, lineNumber, message);
		}
		
		public static IValidationMessage createErrorMessage(String filePath,
				int lineNumber, int columnNumber, String localizedMessage) {
			return createMessage(IValidationMessage.SEVERITY_ERROR, filePath, lineNumber, columnNumber, localizedMessage);
		}

		public static IValidationMessage createWarningMessage(String filePath, int lineNumber, String message)
		{
			return createMessage(IValidationMessage.SEVERITY_WARNING, filePath, lineNumber, message);
		}
		
		public static IValidationMessage createWarningMessage(int lineNumber, String message)
		{
			return createMessage(IValidationMessage.SEVERITY_WARNING, null, lineNumber, message);
		}
		
		public static IValidationMessage createInfoMessage(String filePath, int lineNumber, String message)
		{
			return createMessage(IValidationMessage.SEVERITY_INFO, filePath, lineNumber, message);
		}
		
		public static IValidationMessage createInfoMessage(int lineNumber, String message)
		{
			return createMessage(IValidationMessage.SEVERITY_INFO, null, lineNumber, message);
		}
	
		private static IValidationMessage createMessage(int severity, String filePath, int lineNumber, String message)
		{
			return createMessage(severity, filePath, lineNumber, -1, message);
		}
		
		private static IValidationMessage createMessage(int severity, String filePath, int lineNumber, int columnNumber, String message)
		{
			if (message == null)
				return null;
			
			IValidationMessage validationMessage = new ValidationMessage();
			validationMessage.setAttribute(IValidationMessage.ATTRIBUTE_SEVERITY, severity);
			validationMessage.setAttribute(IValidationMessage.ATTRIBUTE_LINE_NUMBER, lineNumber);
			validationMessage.setAttribute(IValidationMessage.ATTRIBUTE_COLUMN_NUMBER, columnNumber);
			validationMessage.setAttribute(IValidationMessage.ATTRIBUTE_MESSAGE, message);
			if (filePath != null)
				validationMessage.setAttribute(IValidationMessage.ATTRIBUTE_RESOURCE, filePath);
			
			return validationMessage;
		}

	}

	public ResourceWrapper getAssociatedResource ()
	{
		return associatedResource;
	}
	
	public void setAssociatedResource(ResourceWrapper resource)
	{
		this.associatedResource = resource;
	}
}
