/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.validation.internal.content;

import java.io.BufferedReader;
import java.io.CharConversionException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.core.runtime.content.IContentDescriber;
import org.eclipse.core.runtime.content.IContentDescription;

/**
 * The abstract XML Content describer provides generic mechanisms for
 * determining the content of an XML document
 * 
 * @author Ali Mehregani
 */
public abstract class AbstractXMLContentDescriber implements IContentDescriber
{
	/**
	 * Indicates that the content search should continue 
	 */
	public static final int CONTINUE = 10;
	
	/**
	 * @see org.eclipse.core.runtime.content.IContentDescriber#describe(java.io.InputStream, org.eclipse.core.runtime.content.IContentDescription)
	 */
	public int describe(InputStream contents, IContentDescription description) throws IOException
	{
		BufferedReader br = new BufferedReader(new InputStreamReader(contents));
		
		String currentLine = null;
		boolean commentHit = false;
		int inx = -1;
		try {
			while ((currentLine = br.readLine()) != null)
			{
				currentLine = currentLine.trim();
				if (currentLine.length() <= 0)
				{
					continue;
				}
				else if (currentLine.startsWith("<?"))
				{
					int openAngleInx = currentLine.substring(1).indexOf('<') + 1;
					if (openAngleInx > 0)
					{
						if (currentLine.length() > openAngleInx + 4 && !"<!--".equals(currentLine.substring(openAngleInx, openAngleInx + 4)))
						{
							currentLine = currentLine.substring(openAngleInx);
							int type = determineType(currentLine);
							if (type != CONTINUE)
								return type;						
						}					
					}
					continue;				
				}
				else if (currentLine.startsWith("<!--"))
				{
					int endOfComment = currentLine.indexOf("-->");
					if (endOfComment >= 0)
					{
						if (endOfComment+4 < currentLine.length())
						{		
							int type;
							if ((type = determineType(currentLine.substring(endOfComment+3))) == CONTINUE)
								continue;
							return type;						
						}
						continue;
					}
					
					
					commentHit = true;
				}
				else if (commentHit && (inx = currentLine.indexOf("-->")) >= 0)
				{
					commentHit = false;
					if (inx < currentLine.length() - 4)
					{
						int type;
						if ((type = determineType(currentLine.substring(inx).trim())) == CONTINUE)
							continue;
						return type;
					}
				}
				else if (!commentHit)				
				{
					int type = determineType(currentLine);
					if (type == CONTINUE)
						continue;
					return type;

				}
			}
		} catch (CharConversionException e) {
			// Can be thrown when file has binary content
		}
		
		return IContentDescriber.INVALID;
	}

	
	/**
	 * Used to determine the type of the content based on the line passed in
	 * 
	 * @param line The line being processed
	 */
	public abstract int determineType(String line);


	/**
	 * @see org.eclipse.core.runtime.content.IContentDescriber#getSupportedOptions()
	 */
	public QualifiedName[] getSupportedOptions()
	{		
		return null;
	}
	
	
	protected String retrieveElementName (String line)
	{
		if (line == null || line.length() <= 0 || line.charAt(0) != '<')
			return null;
		
		int inx = line.indexOf(' ');
		line = inx < 0 ? line.substring(1) : line.substring(1, inx);
		inx = line.indexOf(':');
		return inx < 0 ? line : line.substring(inx + 1);
	}

}
