/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.validation.internal.core;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.StringTokenizer;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.cosmos.rm.validation.internal.artifacts.ResourceWrapper;
import org.eclipse.cosmos.rm.validation.internal.common.AbstractValidationOutput;
import org.eclipse.cosmos.rm.validation.internal.common.ISMLConstants;
import org.eclipse.cosmos.rm.validation.internal.common.IValidationConstants;
import org.eclipse.cosmos.rm.validation.internal.common.SMLIFIdentity;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.validation.internal.common.AbstractValidationOutput.ValidationMessage;
import org.eclipse.cosmos.rm.validation.internal.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidatorUtil.RemoteRetrievalException;
import org.eclipse.cosmos.rm.validation.internal.databuilders.DataBuilderRegistry;
import org.eclipse.cosmos.rm.validation.internal.databuilders.IDataBuilder;
import org.eclipse.cosmos.rm.validation.internal.databuilders.IdentityDataBuilder;
import org.eclipse.osgi.util.NLS;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXNotRecognizedException;
import org.xml.sax.SAXNotSupportedException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * Implements a custom SAX Default Handler that will delegate ContentHandler
 * calls to IDataBuilder instances.
 * 
 * @author sleeloy
 * @author Ali Mehregani
 */
public class FoundationBuilder extends DefaultHandler implements IFoundationBuilder
{
	/**
	 * The SAX parser
	 */
	private SAXParser saxParser;

	/**
	 * List of IDataBuilder instances
	 */
	private IDataBuilder<?>[] dataStructureBuilders;

	/**
	 * The validation logger
	 */
	private AbstractValidationOutput validationLogger;

	/**
	 * Custom xml schema validator
	 */
	private IXMLValidator xmlValidation;

	/**
	 * Indicates whether the XML file should be validated
	 */
	private boolean validateXML;

	/**
	 * The parsing phase that the parser is in
	 */
	private byte phase = ISMLConstants.START_PHASE;

	/**
	 * The locator used to indicate the column and line number of a problem
	 */
	private Locator locator;
	
	/**
	 * A flag used to indicate when a document element has been hit
	 */
	private boolean documentElementHit;
	
	/**
	 * Indicates whether the input is a set of SML unit document or an SML-IF 
	 * document.  
	 */
	private boolean smlIfInput;
	
	/**
	 * Indicates the status of this main data structure builder.  true indicates
	 * a passed status and false indicates a failed status.
	 */
	private boolean status;
	
	/**
	 * The validation listeners
	 */
	private Collection<IValidationListener> validationListeners;

	/**
	 * Set to true if the validator is running inside the Eclipse environment
	 */
	private boolean inEclipse;
	
	/**
	 * The validation attributes
	 */
	private Map<String, Object> attributes;
	
	/**
	 * The processed import statements
	 */
	private Map<String, String> processedImports;
	
	/**
	 * The imported schema files that are pending to be processed
	 */
	private Map<String, String> pendingProcess;
	
	/**
	 * The absolute path of the current resource
	 */
	private String currentResource;
	
	
	public FoundationBuilder() 
	{
		super();
		status = true;
		processedImports = new Hashtable<String,String>();
		pendingProcess = new Hashtable<String, String>();
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.validation.internal.core.IValidator#initialize(java.util.Map)
	 */
	public void initialize(Map<String, Object> validationAttribute)
	{
		validationLogger = (AbstractValidationOutput) validationAttribute.get(ATTRIBUTE_OUTPUT);
		smlIfInput = !(validationAttribute.get(IValidator.ATTRIBUTE_INPUT_TYPE) instanceof String) || !IValidator.VALUE_SML_UNIT.equals(validationAttribute.get(IValidator.ATTRIBUTE_INPUT_TYPE));				
		inEclipse = IValidator.VALUE_ENV_ECLIPSE.equals(validationAttribute.get(ATTRIBUTE_ENV));			
		this.attributes = validationAttribute;
		
		Object validationClass = (validationAttribute.get(IValidator.ATTRIBUTE_VALIDATION_XML));
		
		// if no validation class then we should do no XML schema validation
		validateXML = true;
		if (validationClass == null)
		{
			validateXML = false;
		} 
		else if (IXMLValidator.DEFAULT_VALIDATION_XML_LITERAL.equals( validationClass ))
		{
			// should validate and use the default validator
			xmlValidation = null;
		} 
		else
		{
			// should use the custom xml schema validator
			IValidator[] validation;
			try
			{
				validation = ValidationFactory.createValidator(IXMLValidator.class, validationClass);
				xmlValidation = validation.length > 0 ? (IXMLValidator) validation[0] : null;
				xmlValidation.initialize(validationAttribute);
			} 
			catch (InstantiationException e)
			{
				validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.errorInstantiating, validationClass)));
			} 
			catch (IllegalAccessException e)
			{
				validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.errorInstantiating, validationClass)));
			} 
			catch (ClassNotFoundException e)
			{
				validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.errorClassNotFound, validationClass)));
			}

		}
		createSAXParser(smlIfInput ? (String) validationAttribute.get(ATTRIBUTE_SML_IF_SCHEMA) : null);
	}

	/**
	 * Configures the default handler and associates the schema with the SAX
	 * Parser
	 * 
	 * @param xsd file location of the schema file to validate
	 */
	protected void createSAXParser(String xsd)
	{
		try
		{
			// if there is no schema then we should not validate
			if (xsd == null)
				validateXML = false;

			InputStream schemaAsStream = (xsd == null) ? null : this.getClass().getClassLoader().getResourceAsStream(xsd);
			saxParser = SMLValidatorUtil.createSAXParser(schemaAsStream, validateXML, this, validationLogger);
		} 
		catch (ParserConfigurationException e)
		{
			fatalError(e);
		} 
		catch (SAXNotRecognizedException e)
		{			
			fatalError(ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.errorUnsupportedSAXProperty1, e.getLocalizedMessage()));
		} 
		catch (SAXNotSupportedException e)
		{			
			fatalError(ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.errorUnsupportedSAXProperty1, e.getLocalizedMessage()));
		} 
		catch (SAXException e)
		{
			fatalError(e);
		}
	}

	public void error(SAXParseException exception)
	{
		status = false;
		validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(exception.getLineNumber(), exception.getLocalizedMessage() + SMLValidationMessages.commonLine + exception.getLineNumber() + SMLValidationMessages.commonColumn + exception.getColumnNumber()));
	}

	protected void fatalError(Exception exception)
	{
		fatalError(ValidationMessage.NO_LINE_NUMBER, exception.getLocalizedMessage());
	}

	protected void fatalError(int lineNumber, String message)
	{
		status = false;
		validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(lineNumber, message));
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.validation.internal.core.IValidator#validate()
	 */
	@SuppressWarnings("unchecked")
	public boolean validate()
	{
		IProgressMonitor monitor = (IProgressMonitor)attributes.get(IValidator.ATTRIBUTE_PROGRESS_MONITOR);
		if (monitor != null)
		{
			monitor.setTaskName(SMLValidationMessages.validationBuildingStructures);
		}
		
		Exception exception = null;
		String currentDocument = null;
		int lineNumber = -1;
		
		
		try
		{
			// Retrieve the builders
			retrieveBuilders();
			
			// Set their message outputter
			for (int i = 0; i < dataStructureBuilders.length; i++)
			{
				IDataBuilder dataBuilder = dataStructureBuilders[i];
				dataBuilder.setMessageOutputter(validationLogger);
			}
			
			// The input is an SML if document 
			String input = (String)attributes.get(IValidator.ATTRIBUTE_INSTANCE);
			if (smlIfInput)
			{
				currentDocument = input;
				saxParser.parse(new InputSource(new FileInputStream(new File(currentDocument))), this);
			}
			// Otherwise the input is expected to be a set of comma separated files and directories
			else
			{
				StringTokenizer inputs = new StringTokenizer(input, ISMLConstants.COMMA); //$NON-NLS-1$
				List<ResourceWrapper> instanceList = new ArrayList<ResourceWrapper>(); 					
				
				// Look up the identity
				SMLIFIdentity identity = (SMLIFIdentity)SMLValidatorUtil.retrieveDataStructure(IdentityDataBuilder.ID);
				SMLIFIdentity metaIdentity = (SMLIFIdentity)attributes.get(IValidator.ATTRIBUTE_IDENTITY);
				if (identity != null) 
				{
					if (metaIdentity != null)
					{
						identity.copy(metaIdentity);
					}
					else
					{
						identity.setBaseURI(IValidationConstants.DUMMY_BASE_URI);
					}
				}
				
				// Send the rule binding data
				phase = ISMLConstants.START_PHASE;
				Map<String, List<String>> ruleBindings = (Map<String, List<String>>)attributes.get(IValidator.ATTRIBUTE_RULE_BINDINGS);
				if (ruleBindings != null && ruleBindings.size() > 0)
				{
					elementNotification(ISMLConstants.SMLIF_URI, ISMLConstants.RULEBINDINGS_ELEMENT, ISMLConstants.RULEBINDINGS_ELEMENT, 
							IValidationConstants.EMPTY_ATTRIBUTES, false, false, true);
					for (Iterator<String> aliases = ruleBindings.keySet().iterator(); aliases.hasNext();)
					{
						String alias =  aliases.next();
						List<String> rules = ruleBindings.get(alias);
						if (rules != null)
						{
							for (int i = 0, ruleCount = rules.size(); i < ruleCount; i++)
							{
								ruleNotification(alias, (String)rules.get(i));
							}
						}
					}
					elementNotification(ISMLConstants.SMLIF_URI, ISMLConstants.RULEBINDINGS_ELEMENT, ISMLConstants.RULEBINDINGS_ELEMENT, 
							IValidationConstants.EMPTY_ATTRIBUTES, false, false, false);
				}
				
				
				
				// Start the definition element
				phase = ISMLConstants.DEFINITIONS_PHASE;
				elementNotification(ISMLConstants.SMLIF_URI, ISMLConstants.DEFINITIONS_ELEMENT, ISMLConstants.DEFINITIONS_ELEMENT, 
						IValidationConstants.EMPTY_ATTRIBUTES, true, false, true);
				
				String contextRoot = (String)attributes.get(IValidator.ATTRIBUTE_REPOSITORY_ROOT);
				contextRoot = contextRoot == null ? "" : contextRoot;
				
				// For every input
				while (inputs.hasMoreTokens())
				{
					currentDocument = inputs.nextToken();
					ResourceWrapper resource = retrieveResource(currentDocument);
					if (resource == null)
						return false;

					String parentFolder = findParentFolder(contextRoot, resource.getFullPath());
					traverseRecursively(instanceList, resource, true, parentFolder);
					Map<String,String> processPending = (Map<String, String>)((Hashtable)pendingProcess).clone();
					
					if (processPending == null)
					{
						continue;
					}
										
					for (Iterator<String> namespaces = processPending.keySet().iterator(); namespaces.hasNext();) 
					{
						String namespace = (String) namespaces.next();
						String path = (String)processPending.get(namespace);
						
						if (path == null || processedImports.get(namespace) != null)
						{
							continue;
						}
						
						try
						{
							traverseRecursively(instanceList, new ResourceWrapper (path, inEclipse), true, parentFolder);														
						}
						catch (MissingResourceException mre)
						{
							validationLogger.reportMessage(ValidationMessageFactory.createWarningMessage(ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.warningMissingResource, path)));							
						}						
					}
					
					processedImports.putAll(processPending);
					for (Iterator<String> namespaces = processPending.keySet().iterator(); namespaces.hasNext();)
					{
						String namespace = namespaces.next();
						processedImports.put(namespace, pendingProcess.get(namespace));
						pendingProcess.remove(namespace);
					}
				}

				// Terminate the definition element
				elementNotification(ISMLConstants.SMLIF_URI, ISMLConstants.DEFINITIONS_ELEMENT, ISMLConstants.DEFINITIONS_ELEMENT, 
						IValidationConstants.EMPTY_ATTRIBUTES, true, false, false);
				
				// Start the instance element
				phase = ISMLConstants.INSTANCES_PHASE;
				elementNotification(ISMLConstants.SMLIF_URI, ISMLConstants.INSTANCES_ELEMENT, ISMLConstants.DEFINITIONS_ELEMENT, 
						IValidationConstants.EMPTY_ATTRIBUTES, false, true, true);
				
				// For every instance document
				for (int i = 0, instanceCount = instanceList.size(); i < instanceCount; i++)
				{
					ResourceWrapper currentInstanceResource = (ResourceWrapper)instanceList.get(i);
					traverseRecursively(instanceList, currentInstanceResource, false, findParentFolder(contextRoot, currentInstanceResource.getFullPath()));
				}
				
				// Terminate the instance element
				elementNotification(ISMLConstants.SMLIF_URI, ISMLConstants.INSTANCES_ELEMENT, ISMLConstants.DEFINITIONS_ELEMENT, 
						IValidationConstants.EMPTY_ATTRIBUTES, false, true, false);
			}
			
			return true;
		} 
		catch (IOException e)
		{
			exception = e;
		} 
		catch (SAXParseException e)
		{			
			lineNumber = e.getLineNumber();
			exception = e;			
		} 
		catch (SAXException e)
		{			
			exception = e;			
		} 
		catch (CoreException e)
		{
			exception = e;
		} 
		catch (ParserConfigurationException e)
		{
			exception = e;
		}

		if (exception != null)
		{
			fatalError(lineNumber, exception.getLocalizedMessage());
		}
		
		return false;
	}

	
	private void ruleNotification(String alias, String ruleAlias) throws SAXException
	{
		fireNotifications(ISMLConstants.SMLIF_URI, new String[]{
				ISMLConstants.RULEBINDING_ELEMENT,
				ISMLConstants.DOCUMENTALIAS_ELEMENT
			}, true);		
		characters(alias.toCharArray(), 0, alias.length());		
		elementNotification(ISMLConstants.SMLIF_URI, ISMLConstants.DOCUMENTALIAS_ELEMENT, ISMLConstants.DOCUMENTALIAS_ELEMENT, 
				IValidationConstants.EMPTY_ATTRIBUTES, false, false, false);				
		elementNotification(ISMLConstants.SMLIF_URI, ISMLConstants.RULEALIAS_ELEMENT, ISMLConstants.DOCUMENTALIAS_ELEMENT, 
				IValidationConstants.EMPTY_ATTRIBUTES, false, false, true);
		characters(ruleAlias.toCharArray(), 0, ruleAlias.length());		
		fireNotifications(ISMLConstants.SMLIF_URI, new String[]{
				ISMLConstants.RULEALIAS_ELEMENT,
				ISMLConstants.RULEBINDING_ELEMENT
			}, false);	
	}

	/**
	 * Given a root path and a resource path, the following method will determine the
	 * resource path minus the intersection and the filename: <br/>
	 *                
	 *                ***
	 * |_____________|___|____________|
	 *  -------root-------
	 *               -----resource----
	 * 
	 * <br/>
	 * The intersection is marked by '***'              
	 * @param root The root path
	 * @param resource The resource path
	 * @return The resource path minus the intersection and the file name
	 */
	private String findParentFolder(String root, String resource)
	{
		// Make sure that both paths use forward slashes
		root = root.replace('\\', '/');
		resource = resource.replace('\\', '/');
		
		IPath rootPath = new Path(root);
		IPath resourcePath = new Path(resource);
		
		if (inEclipse)
		{
			// Find the work-space directory and exclude it from the root path
			IPath workspace = ResourcesPlugin.getWorkspace().getRoot().getLocation();			
			rootPath = rootPath.removeFirstSegments(workspace.segmentCount());			
		}
		
		for (int i = 0, segCount = rootPath.segmentCount(); i < segCount; i++)
		{
			if (resourcePath.segment(0).equals(rootPath.segment(i)))
			{
				resourcePath = resourcePath.removeFirstSegments(1);
			}
			else
			{
				break;
			}
		}
		
		String parentFolder = resourcePath.removeLastSegments(1).toString();
		String device = resourcePath.getDevice();
		if (device != null && parentFolder.startsWith(device))
		{
			parentFolder = parentFolder.substring(device.length());
		}
		
		return parentFolder;
	}

	private void traverseRecursively(List<ResourceWrapper> instanceList, ResourceWrapper resource, boolean isDefinition, String parentFolder) throws CoreException, IOException, SAXException, ParserConfigurationException
	{
		if (resource.isDirectory())
		{
			ResourceWrapper[] children = resource.getChildren();
			parentFolder = (parentFolder.length() > 0 ? parentFolder + "/" : IValidationConstants.EMPTY_STRING) + resource.getName(); //$NON-NLS-1$
			for (int i = 0; i < children.length; i++)
			{
				traverseRecursively(instanceList, children[i], isDefinition, parentFolder);
			}			
		}
		else if (resource.isFile())
		{	
			int type = SMLValidatorUtil.identifyDocumentType(resource);
			currentResource = resource.getAbsolutePath();
			
			switch (type)
			{
				// definition
				case ISMLConstants.TYPE_DEFINITION:
					addDocument (parentFolder, resource);
					break;
				
				// instance 
				case ISMLConstants.TYPE_INSTANCE:
					if (isDefinition)
					{
						instanceList.add(resource);
						return;
					}
					addDocument (parentFolder, resource);					
					break;
				default:
					break;
			}
		}
	}

	private void addDocument(String parentFolder, ResourceWrapper resource) throws SAXException, ParserConfigurationException, IOException, CoreException
	{
		// Add the aliases
		addAliases(parentFolder, resource);
		
		// Add the data	 
		processData(resource);
		
		elementNotification(ISMLConstants.SMLIF_URI, 
				ISMLConstants.DOCUMENT_ELEMENT, 
				ISMLConstants.DOCUMENT_ELEMENT, 
				IValidationConstants.EMPTY_ATTRIBUTES, 
				false, false, false);
	}

	@SuppressWarnings("unchecked")
	private void addAliases(String parentFolder, ResourceWrapper resource) throws SAXException
	{
		// The document and the doc info section
		fireNotifications(ISMLConstants.SMLIF_URI, new String[]{
				ISMLConstants.DOCUMENT_ELEMENT,
				ISMLConstants.DOC_INFO_ELEMENT, 
				ISMLConstants.ALIASES_ELEMENT							
			}, true);
		
		Map<String, List<String>> fileAliasMap = (Map<String, List<String>>)attributes.get(IValidator.ATTRIBUTE_ALIASES);
		boolean definedAliases = false;
		if (fileAliasMap != null)
		{
			List<String> aliases = fileAliasMap.get(resource.getName());						
			if (aliases != null)
			{	
				definedAliases = aliases.size() > 0;
				for (int i = 0, aliasCount = aliases.size(); i < aliasCount; i++)
				{
					String alias = aliases.get(i);
					notifyAlias(alias);							
				}
			}
		}
		
		// If there are no aliases defined in the meta-data, then use the
		// path as the alias
		if (!definedAliases)
		{
			notifyAlias(((parentFolder.length() > 0 ? parentFolder + "/" : IValidationConstants.DUMMY_BASE_URI) + resource.getName()));
		}
							
		fireNotifications(ISMLConstants.SMLIF_URI, new String[]{						
				ISMLConstants.ALIASES_ELEMENT,						
				ISMLConstants.DOC_INFO_ELEMENT					
			}, false);
	}

	private void notifyAlias(String alias) throws SAXException
	{
		elementNotification(ISMLConstants.SMLIF_URI, ISMLConstants.ALIAS_ELEMENT, 
				ISMLConstants.ALIAS_ELEMENT, IValidationConstants.EMPTY_ATTRIBUTES, false, false, true);
		characters(alias.toCharArray(), 0, alias.length());
		elementNotification(ISMLConstants.SMLIF_URI, ISMLConstants.ALIAS_ELEMENT, 
				ISMLConstants.ALIAS_ELEMENT, IValidationConstants.EMPTY_ATTRIBUTES, false, false, false);
	}

	private void processData(ResourceWrapper resource) throws SAXException, ParserConfigurationException, IOException, CoreException
	{
		InputStream fileInputStream = resource.getContents();
		try
		{
			elementNotification(ISMLConstants.SMLIF_URI, 
					ISMLConstants.DATA_ELEMENT, 
					ISMLConstants.DATA_ELEMENT, 
					IValidationConstants.EMPTY_ATTRIBUTES, false, false, true);
			
			validationLogger.setAssociatedResource(resource);
			for (int i = 0; i < dataStructureBuilders.length; i++)
			{
				IDataBuilder<?> dataBuilder = dataStructureBuilders[i];
				dataBuilder.setFilePath(resource.getFullPath());
			}
			
			saxParser.parse(fileInputStream, this);			
			elementNotification(ISMLConstants.SMLIF_URI, 
					ISMLConstants.DATA_ELEMENT, 
					ISMLConstants.DATA_ELEMENT, 
					IValidationConstants.EMPTY_ATTRIBUTES, false, false, false);
		}
		finally
		{
			if (fileInputStream != null)
				fileInputStream.close();
		}
	}

	private void fireNotifications(String uri, String[] elements, boolean open) throws SAXException
	{
		for (int i = 0; i <elements.length; i++)
		{
			elementNotification(uri, elements[i], elements[i], IValidationConstants.EMPTY_ATTRIBUTES, false, false, open);
		}
	}

	private ResourceWrapper retrieveResource(String path)
	{
		try
		{
			return new ResourceWrapper (path, inEclipse);
		}
		catch (MissingResourceException mre)
		{
			validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(ValidationMessage.NO_LINE_NUMBER, mre.getMessage()));
			return null;			
		}
	}
	
	
	private void retrieveBuilders()
	{
		if (dataStructureBuilders == null)
		{
			Collection<IDataBuilder<?>> dataBuilders = DataBuilderRegistry.instance().getDataStructureBuilders();
			dataStructureBuilders = new IDataBuilder[dataBuilders.size()];
			dataBuilders.toArray(dataStructureBuilders);
		}
	}

	public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException
	{
		boolean startDefinitionsPhase = false;
		boolean startInstancesPhase = false;
		
		if (ISMLConstants.SMLIF_URI.equals(uri))
		{
			if (phase != ISMLConstants.INSTANCES_PHASE)
			{
				if ((phase != ISMLConstants.DEFINITIONS_PHASE) && (ISMLConstants.DEFINITIONS_ELEMENT.equals(localName)))
				{
					phase = ISMLConstants.DEFINITIONS_PHASE;
					startDefinitionsPhase = true;
				} else if (ISMLConstants.INSTANCES_ELEMENT.equals(localName))
				{
					phase = ISMLConstants.INSTANCES_PHASE;
					startInstancesPhase = true;
				}
			}
			
			if ((phase == ISMLConstants.DEFINITIONS_PHASE || phase == ISMLConstants.INSTANCES_PHASE))
			{
				if (ISMLConstants.DOCUMENT_ELEMENT.equals(localName))
				{
					documentElementHit = true;
				}
				else if (documentElementHit && ISMLConstants.LOCATOR_ELEMENT.equals(localName))
				{
					retrieveRemoteDocument(attributes);
				}
			}
		}
		else if (ISMLConstants.SCHEMA_URI.equals(uri))
		{
			if (ISMLConstants.IMPORT_ELEMENT.equals(localName))
			{
				String namespace = attributes.getValue(ISMLConstants.NAMESPACE_ATTRIBUTE);
				String location = attributes.getValue(ISMLConstants.SCHEMA_LOCATION_ATTRIBUTE);
				
				if (namespace != null && location != null && currentResource != null 
					&& processedImports.get(namespace) == null && location.indexOf("//") < 0)
				{
					int inx = Math.max(currentResource.lastIndexOf('/'), currentResource.lastIndexOf('\\')); 
					String parentFolder = inx >= 0 ? currentResource.substring(0, inx) : currentResource;
					location = location.startsWith(parentFolder) ? location : parentFolder + "/" + location;
					pendingProcess.put(namespace, location);					
				}
			}
		}
		
		// build datastructures
		elementNotification (uri, localName, qName, attributes, startDefinitionsPhase, startInstancesPhase, true);
	}

	private void retrieveRemoteDocument(Attributes attributes) throws SAXException
	{
		String href = attributes.getValue(ISMLConstants.XLINK_URI, ISMLConstants.HREF_ATTRIBUTE);
		Node node = null;
		Exception exception = null;
		try
		{
			node = SMLValidatorUtil.retrieveRemoteDocument(href);			
			Node dataNode = node.getOwnerDocument().createElementNS(ISMLConstants.SMLIF_URI, ISMLConstants.DATA_ELEMENT);
			dataNode.appendChild(node);
			node = dataNode;						
		} 
		catch (RemoteRetrievalException e)
		{
			exception = e;
		}
		
		if (exception != null || node == null)
		{
			validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(locator.getLineNumber(), NLS.bind(SMLValidationMessages.errorMissingDocument, String.valueOf(locator.getLineNumber()))));						
			return;
		}
		walkDocument(node);
	}

	private void walkDocument(Node node) throws SAXException
	{
		Attributes atts = toAttributes(node.getAttributes());
		domNotification(node, atts, true);
		
		NodeList children = node.getChildNodes();
		for (int i = 0, childCount = children.getLength(); i < childCount; i++)
		{
			walkDocument (children.item(i));			
		}
		
		domNotification(node, atts, false);
	}

	private void domNotification(Node node, Attributes atts, boolean start) throws SAXException
	{
		switch (node.getNodeType())
		{
			case Node.ELEMENT_NODE:
				elementNotification(node.getNamespaceURI(), node.getLocalName(), node.getNodeName(), 
						atts, false, false, start);
				break;
			case Node.TEXT_NODE:
			case Node.COMMENT_NODE:
				if (!start)
					break;
				String value = node.getNodeValue();
				if (value == null)
					break;				
				characters(value.toCharArray(), 0, value.length());
				break;
			default:
				break;
		}
	}

	private Attributes toAttributes(final NamedNodeMap attributes)
	{				
		AttributeWrapper attributeWrapper = new AttributeWrapper (attributes);
		return attributeWrapper;
	}

	public void endElement(String uri, String localName, String qName) throws SAXException
	{
		boolean endDefinitionsPhase = false;
		boolean endInstancesPhase = false;
		
		if (ISMLConstants.SMLIF_URI.equals(uri))
		{
			if ((phase == ISMLConstants.DEFINITIONS_PHASE) && (ISMLConstants.DEFINITIONS_ELEMENT.equals(localName)))
			{
				phase = ISMLConstants.DEFINITIONS_INSTANCES_PHASE;
				endDefinitionsPhase = true;
			} 
			else if ((phase == ISMLConstants.INSTANCES_PHASE) && (ISMLConstants.INSTANCES_ELEMENT.equals(localName)))
			{
				phase = ISMLConstants.DEFINITIONS_INSTANCES_PHASE;
				endInstancesPhase = true;
			}
			
			
			if ((phase == ISMLConstants.DEFINITIONS_PHASE || phase == ISMLConstants.INSTANCES_PHASE) && ISMLConstants.DOCUMENT_ELEMENT.equals(localName))
			{
				documentElementHit = false;				
			}
		}
		elementNotification (uri, localName, qName, null, endDefinitionsPhase, endInstancesPhase, false);
	}
	
	
	private void elementNotification (String uri, String localName, String qName, Attributes attributes, 
									  boolean definitionPhase, boolean instancePhase, boolean start) throws SAXException
	{
		for (int i = 0; i < dataStructureBuilders.length; i++)
		{
			IDataBuilder<?> dataBuilder = dataStructureBuilders[i];
			if (definitionPhase)
			{
				if (start) {
					dataBuilder.startDefinitions();
				} else {
					dataBuilder.endDefinitions();
				}
			}
			if (instancePhase)
			{
				if (start) {
					dataBuilder.startInstances();
				} else {
					dataBuilder.endInstances();
				}
			}
			if ((dataBuilder.getPhase() == ISMLConstants.UNKNOWN_PHASE) || (dataBuilder.getPhase() == phase) || (dataBuilder.getPhase() == ISMLConstants.DEFINITIONS_INSTANCES_PHASE && (phase == ISMLConstants.INSTANCES_PHASE || phase == ISMLConstants.DEFINITIONS_PHASE)))
			{
				if (start) {
					dataBuilder.startElement(uri, localName, qName, attributes);
				} else {
					dataBuilder.endElement(uri, localName, qName);
				}
			}
		}
		
		if (xmlValidation != null)
		{
			if (start)
				xmlValidation.startElement(uri, localName, qName, attributes);
			else
				xmlValidation.endElement(uri, localName, qName);
		}
	}

	public void characters(char[] ch, int start, int len) throws SAXException
	{
		eventNotification (EVENT_CHARACTER, ch, new int[]{start, len}, null);
	}

	public void endDocument() throws SAXException
	{
		eventNotification (EVENT_END_DOCUMENT, null, null, null);
	}

	public void endPrefixMapping(String arg0) throws SAXException
	{
		eventNotification (EVENT_END_PREFIX, null, null, new String[]{arg0});
	}

	public void unparsedEntityDecl(String arg0, String arg1, String arg2, String arg3) throws SAXException
	{
		super.unparsedEntityDecl(arg0, arg1, arg2, arg3);
	}

	public void ignorableWhitespace(char[] arg0, int arg1, int arg2) throws SAXException
	{
		eventNotification (EVENT_IGNORABLE_WHITESPACE, arg0, new int[]{arg1, arg2}, null);
	}

	public void processingInstruction(String arg0, String arg1) throws SAXException
	{
		eventNotification (EVENT_PROCESSING_INSTRUCTIONS, null, null, new String[]{arg0, arg1});
	}

	public void skippedEntity(String arg0) throws SAXException
	{
		eventNotification (EVENT_SKIPPED_ENTITY, null, null, new String[]{arg0});
	}

	public void startDocument() throws SAXException
	{
		eventNotification (EVENT_START_DOCUMENT, null, null, null);
	}

	public void startPrefixMapping(String arg0, String arg1) throws SAXException
	{
		eventNotification(EVENT_START_PREFIX, null, null, new String[]{arg0, arg1});
	}
	
	public void setDocumentLocator(Locator locator)
	{
		this.locator = locator;
		retrieveBuilders();
		for (int i = 0; i < dataStructureBuilders.length; i++)
		{
			dataStructureBuilders[i].setDocumentLocator(locator);
		}
		
		
		if (xmlValidation != null)
		{
			xmlValidation.setDocumentLocator(locator);
		}
	}
	
	private void eventNotification (int event, char[] characters, int[] integerParameters, String[] stringParameters) throws SAXException
	{
		retrieveBuilders();
		for (int i = 0; i < dataStructureBuilders.length; i++)
		{
			byte builderPhase = dataStructureBuilders[i].getPhase();
			if ((builderPhase == ISMLConstants.UNKNOWN_PHASE || 
				 builderPhase == phase || 
				 builderPhase == ISMLConstants.DEFINITIONS_INSTANCES_PHASE && (phase == ISMLConstants.INSTANCES_PHASE || phase == ISMLConstants.DEFINITIONS_PHASE)) &&
				(isEventSupported(dataStructureBuilders[i], event)))
			{
				handleEvent (dataStructureBuilders[i], event, characters, integerParameters, stringParameters);				
			}
		}		
		
		if (xmlValidation != null)
		{
			handleEvent (xmlValidation, event, characters, integerParameters, stringParameters);
		}
	}
	
	
	public void comment(char[] ch, int start, int len) throws SAXException 
	{
		eventNotification (EVENT_COMMENT, ch, new int[]{start, len}, null);
	}
	
	
	/**
	 * Returns true if dataBuilder supports the event passed
	 * in.
	 * 
	 * @param dataBuilder The data builder
	 * @param event The event to be checked
	 * @return true if dataBuilder supports event; false otherwise
	 */
	private boolean isEventSupported(IDataBuilder<?> dataBuilder, int event)
	{
		int supportedEvents = dataBuilder.getSupportedEvents();
		return (supportedEvents & (1 << event)) > 0;
	}

	private void handleEvent (ContentHandler contentHandler, int event, char[] characters, int[] integerParameters, String[] stringParameters) throws SAXException
	{
		switch (event)
		{
			case EVENT_COMMENT:
				((IDataBuilder<?>)contentHandler).comment(characters, integerParameters[0], integerParameters[1]);
				break;
			case EVENT_CHARACTER:
				contentHandler.characters(characters, integerParameters[0], integerParameters[1]);
				break;
			case EVENT_START_DOCUMENT:
				contentHandler.startDocument();
				break;
			case EVENT_END_DOCUMENT:
				contentHandler.endDocument();
				break;
			case EVENT_START_PREFIX:
				contentHandler.startPrefixMapping(stringParameters[0], stringParameters[1]);
				break;
			case EVENT_END_PREFIX:
				contentHandler.endPrefixMapping(stringParameters[0]);
				break;
			case EVENT_IGNORABLE_WHITESPACE:
				contentHandler.ignorableWhitespace(characters, integerParameters[0], integerParameters[1]);
				break;
			case EVENT_SKIPPED_ENTITY:
				contentHandler.skippedEntity(stringParameters[0]);
				break;
			case EVENT_PROCESSING_INSTRUCTIONS:
				contentHandler.processingInstruction(stringParameters[0], stringParameters[1]);
				break;
			default:
				break;
		}
	}
	
	
	private class AttributeWrapper implements Attributes
	{
		private NamedNodeMap attributes;
		private Map<String, Integer> qNameInx;
		private Map<String, Integer> uriInx;
		
		public AttributeWrapper(NamedNodeMap attributes)
		{
			this.attributes = attributes;
			qNameInx = new Hashtable<String, Integer>();
			uriInx = new Hashtable<String, Integer>();
			
			if (attributes == null)
				return;
			for (int i = 0, attributeCount = attributes.getLength(); i < attributeCount; i++)
			{
				Node att = attributes.item(i);
				Integer inx = new Integer(i);
				qNameInx.put(att.getNodeName(), inx);
				String key = att.getNamespaceURI() == null ? IValidationConstants.EMPTY_STRING : att.getNamespaceURI();
				key += att.getLocalName();
				uriInx.put(key, inx);
			}
		}
		
		public int getIndex(String qName)
		{			
			Integer inx = (Integer)qNameInx.get(qName);
			return inx == null ? -1 : inx.intValue();
		}

		public int getIndex(String uri, String localName)
		{				
			Integer inx = (Integer)uriInx.get(uri == null ? localName : uri+localName);
			return inx == null ? -1 : inx.intValue();
		}

		public int getLength()
		{
			return attributes.getLength();
		}

		public String getLocalName(int index)
		{
			return attributes.item(index).getLocalName();
		}

		public String getQName(int index)
		{				
			return attributes.item(index).getNodeName();
		}

		public String getType(int index)
		{
			return null;
		}

		public String getType(String qName)
		{
			return null;
		}

		public String getType(String uri, String localName)
		{				
			return null;
		}

		public String getURI(int index)
		{
			return attributes.item(index).getNamespaceURI();
		}

		public String getValue(int index)
		{
			Node node = index >= 0 && index < attributes.getLength() ? attributes.item(index) : null;
			return node == null ? null : node.getNodeValue();
		}

		public String getValue(String qName)
		{
			Node node = attributes.getNamedItem(qName);
			return node == null ? null : node.getNodeValue();
		}

		public String getValue(String uri, String localName)
		{
			Node node = attributes.getNamedItemNS(uri, localName); 
			return node == null ? null : node.getNodeValue();
		}
		
	}
	
	public boolean getStatus()
	{
		return status;
	}

	public void addValidationListener(IValidationListener listener) {
		if (listener == null) {
			throw new NullPointerException();
		}
		if (validationListeners == null) {
			validationListeners = new ArrayList<IValidationListener>();
		}
		if (!validationListeners.contains(listener)) {
			validationListeners.add(listener);
		}
	}

	public void endCDATA() throws SAXException {
		// do nothing
	}

	public void endDTD() throws SAXException {
		// do nothing
	}

	public void endEntity(String arg0) throws SAXException {
		// do nothing
	}

	public void startCDATA() throws SAXException {
		// do nothing
	}

	public void startDTD(String arg0, String arg1, String arg2) throws SAXException {
		// do nothing
	}

	public void startEntity(String arg0) throws SAXException {
		// do nothing
	}
}
