/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.validation.internal.core;

import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.cosmos.rm.validation.internal.common.SMLIFIdentity;


/**
 * The generic interface that is shared between the different phase of 
 * the validation process.
 * 
 * @see IXMLValidator
 * @see ISMLValidator
 * @see ISchematronValidation
 * @author Ali Mehregani
 */
public interface IValidator
{
	/**
	 * This attribute indicates the environment that the validation process
	 * is being done in.  The value of this attribute can be one of the following
	 * constants:<br/>
	 * 
	 * VALUE_ENV_ECLIPSE <br/>
	 * VALUE_ENV_STANDALONE <br/>
	 * Default value is VALUE_ENV_STANDALONE
	 */
	public static final String ATTRIBUTE_ENV = "attribute.environment";
		
	/**
	 * If {@link #ATTRIBUTE_INPUT_TYPE} is set to {@value #VALUE_SML_UNIT}, then the
	 * expected value of this attribute is a set of comma separated SML unit documents; 
	 * otherwise the input is expected to be the path of an SML-IF document.
	 */
	public static final String ATTRIBUTE_INSTANCE = "validation.instance";
	
	/**
	 * Stores the class that will be used to perform the XML validation
	 */
	public static final String ATTRIBUTE_VALIDATION_XML = "attribute.validation.xml";
	
	/**
	 * Stores the classes that will be used to perform the SML (schema extension) validations
	 */
	public static final String ATTRIBUTE_VALIDATION_SML = "attribute.validation.sml";
	
	/**
	 * Stores the class that will be used to perform the schematron validation
	 */
	public static final String ATTRIBUTE_VALIDATION_SCHEMATRON = "attribute.validation.schematron";
	
	/**
	 * A path to a file which the message will be written to
	 */
	public static final String ATTRIBUTE_FILE_OUTPUT = "attribute.file.output";
	
	/**
	 * Indicates the location of the SML-IF schema.  
	 */
	public static final String ATTRIBUTE_SML_IF_SCHEMA = "validation.schema.smlif";
	
	/**
	 * Indicates the location of the schematron schema.  
	 */
	public static final String ATTRIBUTE_SCHEMATRON_SCHEMA = "validation.schema.schematron";
		
	/**
	 * Indicates the input type.  Valid values are:
	 * <ul>
	 * 	<li> VALUE_SML_UNIT </li>
	 *  <li> VALUE_SML_IF </li>
	 * </ul>
	 * The default value is VALUE_SML_IF
	 */
	public static final String ATTRIBUTE_INPUT_TYPE = "validation.input.type";
		
	/**
	 * Indicates the validation output attribute
	 */
	public static final String ATTRIBUTE_OUTPUT = "validation.output";
	
	/**
	 * Indicates the structured schematron output attribute
	 */
	public static final String ATTRIBUTE_STRUCTURED_OUTPUT = "validation.structured.output";;

	/**
	 * Stores the repository root directory
	 */
	public static final String ATTRIBUTE_REPOSITORY_ROOT = "validation.repository.root";
	
	/**
	 * Indicates whether the validation should stop after encountering the first error.
	 */
	public static final String ATTRIBUTE_FAST_FAIL_POLICY = "validation.policy.fastfail";

	/**
	 * This attribute is used to suppress warnings.  The value of this attribute is
	 * expected to be of type {@link Boolean}.  If the value of the {@link Boolean} is
	 * true, then warnings will be suppressed; otherwise they will be displayed. 
	 */
	public static final String ATTRIBUTE_SUPPRESS_WARNINGS = "validation.suppress.warnings";
	
	/**
	 * Stores the progress monitor used to indicate the progress of a validation process.
	 * The value is either expected to be null or of type {@link IProgressMonitor}  
	 */
	public static final String ATTRIBUTE_PROGRESS_MONITOR = "validation.progress";
	
	/**
	 * The SML-IF identity.  The value is expected to be of type
	 * {@link SMLIFIdentity}.
	 */
	public static final String ATTRIBUTE_IDENTITY = "validation.identity";
	
	/**
	 * Rule bindings.  The value is expected to be of type
	 * {@link Map} where KEY = alias and the value = A {@link List} of rule bindings
	 */
	public static final String ATTRIBUTE_RULE_BINDINGS = "validation.rule.bindings";
	
	/**
	 * The aliases.  The value is expected to be of type
	 * {@link Map} where KEY = file path and the value = A {@link List} of aliases
	 */
	public static final String ATTRIBUTE_ALIASES = "validation.aliases";
	
	/**
	 * Indicates that the validation is being performed in Eclipse
	 */
	public static final String VALUE_ENV_ECLIPSE = "value.environment.eclipse";
	
	/**
	 * Indicates that the validation is being performed in standalone mode
	 */
	public static final String VALUE_ENV_STANDALONE = "value.environment.standalone";
	
	/**
	 * Indicates that the input is a set of SML documents that will need to be
	 * validated.
	 */
	public static final String VALUE_SML_UNIT = "value.sml.unit";
	
	/**
	 * Indicates that the input type is an SML-IF document that will need to be
	 * validated.
	 */
	public static final String VALUE_SML_IF = "value.sml.if";
	
	/**
	 * Initialize this validation class based on the set of attributes
	 * passed in.  See the ATTRIBUTE_* available.
	 * 
	 * @param validationAttribute The validation outputs
	 */
	public void initialize(Map<String, Object> validationAttribute);

	
	/**
	 * Performs the validation and returns true if successful and
	 * false otherwise.
	 * 
	 * @return true if successful; false otherwise
	 */
	public boolean validate();
	
	public void addValidationListener(IValidationListener listener);

}
