/*******************************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.rm.validation.internal.databuilders;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Stack;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.eclipse.cosmos.rm.validation.internal.SMLActivator;
import org.eclipse.cosmos.rm.validation.internal.artifacts.ElementLocation;
import org.eclipse.cosmos.rm.validation.internal.common.ISMLConstants;
import org.eclipse.cosmos.rm.validation.internal.common.IValidationConstants;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.validation.internal.common.AbstractValidationOutput.ValidationMessage;
import org.eclipse.cosmos.rm.validation.internal.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.validation.internal.core.IFoundationBuilder;
import org.eclipse.osgi.util.NLS;
import org.w3c.dom.Comment;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.Text;
import org.xml.sax.Attributes;

/**
 * This class will build a map containing all instance documents in a DOM format.
 * The documents will be identified by the document alias
 * 
 * @author Ali Mehregani
 */
public class DocumentDOMBuilder extends AbstractDataBuilder<Map<String, Object>>
{
	/**
	 * The ID of this builder
	 */
	public static final String ID = SMLActivator.PLUGIN_ID + ".DocumentDOMBuilder";
	
	/**
	 * The key used for documents that don't have aliases
	 */
	public static final String NO_ALIAS = "no_alias";
	
	/**
	 * The root document
	 */
	private Document document;
	
	/**
	 * Stores the document nodes
	 * 
	 * KEY = A string representing a document alias
	 * VALUE = An object of type {@link Node} representing the root document node
	 */
	private Map<String, Object> domDocumentMap;

	/**
	 * Stores the document nodes
	 * 
	 * KEY = A string representing a document alias
	 * VALUE = An object of type {@link Node} representing the root the definition document node
	 */
	private Map<String, Object> domDefinitionDocumentMap;
	
	/**
	 * Stores a list of documents nodes without an alias
	 */
	private List<Node> documentsWithNoAlias;
	
	/**
	 * The current root element of the document
	 */
	private Node rootElement;
	
	/**
	 * Keeps track of the elements of the current document element
	 */
	private Stack<Element> elementStack = new Stack<Element>();
	
	/**
	 * The current alias list
	 */
	private List<String> aliasesList;
	
	/**
	 * Current alias
	 */
	private String currentAlias;
	
	/**
	 * Indicates whether a CDATA section is reached 
	 */
	private boolean cdataStarted;

	/**
	 * Indicate that the alias element is hit
	 */
	private boolean aliasElementHit;
	
	/**
	 * Indicates that the document element is hit
	 */
	private boolean documentElementHit;
	
	/**
	 * Indicates that the data element is hit
	 */
	private boolean dataElementHit;

	/**
	 * Indicates that the instance element is hit
	 */
	private boolean instanceElementHit;

	/**
	 * Indications when the definition element has been hit
	 */
	private boolean definitionElementHit;
	
	/**
	 * Map of line numbers to elements 
	 */
	private Map<Element, ElementLocation> lineNumberMap;
	
	/**
	 * The schemaComplete attribute on the definitions element
	 */
	private boolean isSchemaComplete;
	
	

	public DocumentDOMBuilder()
	{
		domDocumentMap = new Hashtable<String, Object>();
		domDefinitionDocumentMap = new Hashtable<String, Object>();
		documentsWithNoAlias = new ArrayList<Node>();
		domDocumentMap.put(NO_ALIAS, documentsWithNoAlias);
		
		currentAlias = IValidationConstants.EMPTY_STRING;
		aliasesList = new ArrayList<String>();
		lineNumberMap = new Hashtable<Element, ElementLocation>();
		instanceElementHit = false;
		definitionElementHit = false;
		isSchemaComplete = false;		
		
		super.addEvents(new int[]{IFoundationBuilder.EVENT_CHARACTER, IFoundationBuilder.EVENT_COMMENT});
	}

	
	/**
	 * @see org.eclipse.cosmos.rm.validation.internal.databuilders.AbstractDataBuilder#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
	 */
	public void startElement(String uri, String localName, String qName, Attributes attributes)
	{		
		/* The document and data element is hit */
		if (dataElementHit)
		{			
			if (document == null)
			{
				DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
				factory.setNamespaceAware(true);
				factory.setIgnoringElementContentWhitespace(true);
				
				try
				{			
					document = factory.newDocumentBuilder().newDocument();			
				}
				catch (ParserConfigurationException e)
				{					
					setErrorMessage(ValidationMessageFactory.createErrorMessage(ValidationMessage.NO_LINE_NUMBER, SMLValidationMessages.errorCreatingDocument));
					setStructureValidity(false);
					return;
				}
			}
			
			Element element = document.createElementNS(uri, qName);
			if (locator != null) {
				setLocation(element, new ElementLocation(getFilePath(), locator.getLineNumber()));
			}
			
			// Add all attributes to the element
			// If this is a namespace import, we need to make sure
			// it contains a schemaLocation
			boolean importElement = ISMLConstants.IMPORT_ELEMENT.equals(localName);
			boolean needSchemaLocation = importElement;
			boolean smlNamespace = false;
			for (int i = 0; i < attributes.getLength(); i++)
			{
				if (importElement && ISMLConstants.SCHEMA_LOCATION_ATTRIBUTE.equals(attributes.getLocalName(i))) {
					needSchemaLocation = false;
				}
				if (importElement && ISMLConstants.NAMESPACE_ATTRIBUTE.equals(attributes.getLocalName(i)) && ISMLConstants.SML_URI.equals(attributes.getValue(i))) {
					smlNamespace = true;
				}	
				
				String attributeUri = attributes.getURI(i);
				if (attributeUri == null || attributeUri.length() <= 0)
				{
					element.setAttribute(attributes.getQName(i), attributes.getValue(i));
				}
				else
				{
					element.setAttributeNS(attributeUri, attributes.getQName(i), attributes.getValue(i));
				}				
			}
			// add in schema location so SML validation works properly
			// https://bugs.eclipse.org/bugs/show_bug.cgi?id=177811
			if (smlNamespace && needSchemaLocation) {
				element.setAttribute(ISMLConstants.SCHEMA_LOCATION_ATTRIBUTE, ISMLConstants.SML_URI);
			}
			
			if (rootElement == null)
			{
				rootElement = element;
			}
			else
			{
				if(!elementStack.isEmpty())
					((Element) elementStack.peek()).appendChild(element);
			}
			elementStack.push(element);
		}
		
		
		if (ISMLConstants.SMLIF_URI.equals(uri))
		{
			/* The alias element is hit */
			if (ISMLConstants.ALIAS_ELEMENT.equals(localName))
			{
				aliasElementHit = true;
			}
			/* This test is only needed because of the associated JUnit */
			else if (ISMLConstants.INSTANCES_ELEMENT.equals(localName))
			{
				instanceElementHit = true;
			}
			else if (ISMLConstants.DEFINITIONS_ELEMENT.equals(localName))
			{
				definitionElementHit = true;
				
				int idx = attributes.getIndex(ISMLConstants.SCHEMA_COMPLETE_ATTR);
				if(idx != -1)
					isSchemaComplete = Boolean.parseBoolean(attributes.getValue(idx));
				
			}
			/* The document element is hit */
			else if (/*instanceElementHit &&*/ ISMLConstants.DOCUMENT_ELEMENT.equals(localName))
			{
				documentElementHit = true;				
			}
			/* The data element is hit */
			else if (documentElementHit && ISMLConstants.DATA_ELEMENT.equals(localName))
			{
				dataElementHit = true;
			}
		}
	}

	
	/**
	 * @see org.eclipse.cosmos.rm.validation.internal.databuilders.AbstractDataBuilder#endElement(java.lang.String, java.lang.String, java.lang.String)
	 */
	public void endElement(String uri, String localName, String qName)
	{
		if (ISMLConstants.SMLIF_URI.equals(uri))
		{
			/* The alias element is hit */
			if (ISMLConstants.ALIAS_ELEMENT.equals(localName))
			{
				if (currentAlias.length() >  0)
				{
					URI aliasURI = null;
					try
					{
						aliasURI = new URI(currentAlias);
					}
					catch (URISyntaxException use)
					{
						// Ignore
					}
					
					// Aliases are expected to be unique
					if (domDocumentMap.get(currentAlias) != null)
					{
						getMessageOutputter().reportMessage(ValidationMessageFactory.createErrorMessage(
								locator.getLineNumber(), NLS.bind(SMLValidationMessages.errorDuplicateAlias, currentAlias)));
					}
					// An alias cannot have a fragment component
					else if (aliasURI != null && aliasURI.getFragment() != null)
					{
						getMessageOutputter().reportMessage(ValidationMessageFactory.createErrorMessage(
								locator.getLineNumber(), SMLValidationMessages.errorAliasHasFragment));
					}
					
					aliasesList.add(currentAlias);
				}
				currentAlias = IValidationConstants.EMPTY_STRING;
				aliasElementHit = false;
			}
			else if (instanceElementHit && ISMLConstants.INSTANCES_ELEMENT.equals(localName))
			{
				instanceElementHit = false;				
			}
			else if (definitionElementHit && ISMLConstants.DEFINITIONS_ELEMENT.equals(localName))
			{
				definitionElementHit = false;				
			}
			/* The document element is hit */
			else if (documentElementHit && ISMLConstants.DOCUMENT_ELEMENT.equals(localName))
			{
				documentElementHit = false;				
			}
			/* The data element is hit */
			else if (dataElementHit && ISMLConstants.DATA_ELEMENT.equals(localName))
			{
				dataElementHit = false;
				
				/* Iterate through the aliases and add them to the map */
				if (rootElement != null)
				{
					// We end up with an extra text node we need to back up over
					if (rootElement.getLastChild() != null && (rootElement.getLastChild().getNodeType() == Node.TEXT_NODE) && ("".equals( rootElement.getLastChild().getNodeValue().trim() ))) {
						rootElement.removeChild(rootElement.getLastChild());
					}
					document.appendChild(rootElement);
					if (instanceElementHit) 
					{
						if (aliasesList.size() <= 0)
						{
							documentsWithNoAlias.add(rootElement);							
						}
						
						for (int i = 0, aliasCount = aliasesList.size(); i < aliasCount; i++) 
						{							
							domDocumentMap.put(aliasesList.get(i), rootElement);							
						}						
					}
					else if(definitionElementHit)
					{
						for (int i = 0, aliasCount = aliasesList.size(); i < aliasCount; i++) 
						{
							domDefinitionDocumentMap.put(aliasesList.get(i), rootElement);
						}						
						
					}
				}
				aliasesList.clear();
				rootElement = null;
				document = null;
			}
		}
		
		if (dataElementHit)
		{
			elementStack.pop();		
		}
	}

	/**
	 * @see org.xml.sax.helpers.DefaultHandler#characters(char[], int, int)
	 */
	public void characters(char[] characters, int start, int length)
	{		
		if (aliasElementHit)
		{
			currentAlias += new String(characters, start, length);
		}
		else if (dataElementHit)
		{		
			if (rootElement == null)
				return;
			
			String buffValue = new String(characters, start, length);
			boolean isBufferJustWhiteSpace = IValidationConstants.EMPTY_STRING.equals(buffValue.trim());
			boolean isBufferEmpty = buffValue.length() <= 0;
			
			if ((cdataStarted && isBufferEmpty) || (!cdataStarted && isBufferJustWhiteSpace))
			{
				cdataStarted = false;
				return;
			}
					
			Text text = document.createTextNode(buffValue);
			cdataStarted = true;
			
			if (elementStack.size() <= 0)
			{
				rootElement.appendChild(text);
			}
			else
			{
				Node node = (Node) elementStack.peek();
				node.getOwnerDocument().adoptNode(text);
				node.appendChild(text);
			}
		}
	}
	
	/**
	 * Create a comment DOM node
	 */
	public void comment(char[] characters, int start, int length)
	{	
		if (dataElementHit)
		{		
			if (rootElement == null)
				return;
			
			String buffValue = new String(characters, start, length);
					
			Comment text = document.createComment(buffValue);
			cdataStarted = true;
			
			if (elementStack.size() <= 0)
			{
				rootElement.appendChild(text);
			}
			else
			{
				((Node) elementStack.peek()).appendChild(text);
			}
		}
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.validation.internal.databuilders.IDataBuilder#getDataStructure()
	 */
	public Map<String, Object> getDataStructure()
	{
		return domDocumentMap;
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.validation.internal.databuilders.AbstractDataBuilder#getPhase()
	 */
	public byte getPhase()
	{
		return ISMLConstants.UNKNOWN_PHASE;
	}

	
	public ElementLocation getLocation(Element element) 
	{
		return (ElementLocation) lineNumberMap.get(element);
	}
	
	public void setLocation(Element element, ElementLocation location) {
		lineNumberMap.put(element, location);
	}


	/**
	 * @return the isSchemaComplete
	 */
	public boolean isSchemaComplete() {
		return isSchemaComplete;
	}


	/**
	 * @return the domDefinitionDocumentMap
	 */
	public Map<String, Object> getDefinitionDocumentMap() {
		return domDefinitionDocumentMap;
	}
}
