/*******************************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.rm.validation.internal.databuilders;

import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.LineNumberReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.cosmos.rm.validation.internal.SMLActivator;
import org.eclipse.cosmos.rm.validation.internal.artifacts.ElementLocation;
import org.eclipse.cosmos.rm.validation.internal.artifacts.ElementModel;
import org.eclipse.cosmos.rm.validation.internal.artifacts.ElementSchemaModel;
import org.eclipse.cosmos.rm.validation.internal.common.ISMLConstants;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidatorUtil;
import org.xml.sax.Attributes;

/**
 * This class will obtain the source for all instance and
 * XML schema documents for the purpose of Schema validation,
 * also obtaining line number information.
 * 
 * @author David Whiteman
 */
public class ElementSourceBuilder extends AbstractDataBuilder<Object>
{
	/**
	 * The ID of this builder
	 */
	public static final String ID = SMLActivator.PLUGIN_ID + ".ElementSourceHandler";
	
	private Map<String, ElementModel> schemas;

	/**
	 * Indicates that the document element is hit
	 */
	private boolean documentElementHit;
	
	/**
	 * Indicates that the data element is hit
	 */
	private boolean dataElementHit;

	/**
	 * Indicates that the instance element is hit
	 */
	private boolean instanceElementHit;

	private boolean definitionElementHit;

	private ElementModel currentElementModel;

	private ElementLocation lastDataLocation;

	private List<ElementModel> instances;

	private String smlifFile;

	private boolean inEclipse;

	public ElementSourceBuilder(boolean inEclipse, String smlifFile)
	{
		schemas = new HashMap<String, ElementModel>();
		instances = new ArrayList<ElementModel>();
		instanceElementHit = false;
		definitionElementHit = false;
		this.smlifFile = smlifFile;
		this.inEclipse = inEclipse;
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.validation.internal.databuilders.AbstractDataBuilder#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
	 */
	public void startElement(String uri, String localName, String qName, Attributes attributes)
	{		
		/* The document and data element is hit */
		if (dataElementHit)
		{			
			if (!ISMLConstants.SCHEMATRON_URI.equals(uri))
			{
				if (lastDataLocation != null) {
					if (ISMLConstants.SCHEMA_ELEMENT.equals(localName)) {
						ElementSchemaModel schemaModel = new ElementSchemaModel(qName, elementFilePath(), Math.max(lastDataLocation.getLineNumber(), 0), lastDataLocation.getColumnNumber());
						schemaModel.setTargetNamespace(attributes.getValue(ISMLConstants.TARGET_NAMESPACE_ATTRIBUTE));
						for (int i = 0; i < attributes.getLength(); i++) {
							if (attributes.getQName(i).startsWith(ISMLConstants.XML_NS_ATTRIBUTE) && !(attributes.getValue(i).equals(schemaModel.getTargetNamespace()))) {
								schemaModel.addPrerequisiteNamespace(attributes.getValue(i));
							}
						}
						currentElementModel = schemaModel;
					} else {
						currentElementModel = new ElementModel(qName, elementFilePath(), Math.max(lastDataLocation.getLineNumber(), 0), lastDataLocation.getColumnNumber());
					}
					lastDataLocation = null;
				}
			}
			
			boolean importElement = false;
			if (ISMLConstants.SCHEMA_URI.equals(uri))
			{
				if (ISMLConstants.IMPORT_ELEMENT.equals(localName))
				{
					importElement = true;
					String namespace = attributes.getValue(ISMLConstants.NAMESPACE_ATTRIBUTE);
					if (namespace != null && currentElementModel instanceof ElementSchemaModel)
					{
						((ElementSchemaModel)currentElementModel).addImportedSchema(namespace);
					}
				}			
			}

			
			boolean needSchemaLocation = importElement && (currentElementModel != null);
			boolean smlNamespace = false;
			if (importElement)
			{
				for (int i = 0; i < attributes.getLength(); i++)
				{
					if (ISMLConstants.SCHEMA_LOCATION_ATTRIBUTE.equals(attributes.getLocalName(i))) {
						needSchemaLocation = false;
					}
					if (ISMLConstants.NAMESPACE_ATTRIBUTE.equals(attributes.getLocalName(i)) && ISMLConstants.SML_URI.equals(attributes.getValue(i))) {
						smlNamespace = true;
					}
				}
			}
						
			// add in schema location so SML validation works properly
			// https://bugs.eclipse.org/bugs/show_bug.cgi?id=177811
			if (smlNamespace && needSchemaLocation) {
				StringBuffer buffer = new StringBuffer(" "+ISMLConstants.SCHEMA_LOCATION_ATTRIBUTE);
				buffer.append(ISMLConstants.EQUAL_SIGN).append(ISMLConstants.DOUBLE_QUOTE).append(ISMLConstants.SML_URI).append(ISMLConstants.DOUBLE_QUOTE);
				currentElementModel.setAdditionalSchemaLocation(buffer.toString());
			}			
		}
		
		
		if (ISMLConstants.SMLIF_URI.equals(uri))
		{
			/* This test is only needed because of the associated JUnit */
			if (ISMLConstants.INSTANCES_ELEMENT.equals(localName))
			{
				instanceElementHit = true;
			}
			else if (ISMLConstants.DEFINITIONS_ELEMENT.equals(localName))
			{
				definitionElementHit = true;
			}
			/* The document element is hit */
			else if (/*instanceElementHit &&*/ ISMLConstants.DOCUMENT_ELEMENT.equals(localName))
			{
				documentElementHit = true;				
			}
			/* The data element is hit */
			else if (documentElementHit && ISMLConstants.DATA_ELEMENT.equals(localName))
			{
				dataElementHit = true;
				if (locator != null) {
					lastDataLocation = new ElementLocation(null, locator.getLineNumber() + 1, locator.getColumnNumber() + 1);
				} else {
					lastDataLocation = new ElementLocation(null, 0, 0);
				}
			}
		}
	}

	private String elementFilePath() {
		// If we are processing an SML-IF file, getFilePath() will be null
		return smlifFile == null ? getFilePath() : smlifFile;
	}

	
	/**
	 * @see org.eclipse.cosmos.rm.validation.internal.databuilders.AbstractDataBuilder#endElement(java.lang.String, java.lang.String, java.lang.String)
	 */
	public void endElement(String uri, String localName, String qName)
	{
		if (ISMLConstants.SMLIF_URI.equals(uri))
		{
			if (instanceElementHit && ISMLConstants.INSTANCES_ELEMENT.equals(localName))
			{
				instanceElementHit = false;				
			}
			else if (definitionElementHit && ISMLConstants.DEFINITIONS_ELEMENT.equals(localName))
			{
				definitionElementHit = false;				
			}
			/* The document element is hit */
			else if (documentElementHit && ISMLConstants.DOCUMENT_ELEMENT.equals(localName))
			{
				documentElementHit = false;				
			}
			/* The data element is hit */
			else if (dataElementHit && ISMLConstants.DATA_ELEMENT.equals(localName))
			{
				dataElementHit = false;
				addModelToList();
			}
		}
		
		if (ISMLConstants.SCHEMA_URI.equals(uri) && ISMLConstants.SCHEMA_ELEMENT.equals(localName)) {
			addSchemaToList();
		}
	}

	private void addModelToList() {
		if (currentElementModel == null) {
			return;
		}
		processCurrentElementModel();
		
		if (!instances.contains(currentElementModel))
		{
			instances.add(currentElementModel);
		}
		currentElementModel = null;
	}

	private void processCurrentElementModel() {
		if (locator != null) {
			currentElementModel.setEndingLine(locator.getLineNumber());
		}
		getElementSource(currentElementModel);
	}

	private void addSchemaToList() {
		if (currentElementModel == null) {
			return;
		}
		processCurrentElementModel();
		
		// The schemas map is keyed by the targetNamespace value of the schema.
		// This allows the schema validator to look up the prerequsite schemas in
		// the map using this key.
		schemas.put(((ElementSchemaModel) currentElementModel).getTargetNamespace(), currentElementModel);
		currentElementModel = null;
	}

	private void getElementSource(ElementModel elementModel) {
		LineNumberReader in = null;
		try {
			String path = inEclipse ? SMLValidatorUtil.retrieveAbsolutePath(elementModel.getFilePath()) : elementModel.getFilePath();			
			in = new LineNumberReader(new FileReader(path));
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		try {
			String currentLine = in.readLine();
			
			while (in.getLineNumber() < elementModel.getLineNumber()) {
				currentLine = in.readLine();
			}
			
			StringBuffer contents = new StringBuffer();
			if (smlifFile != null) {
				contents.append(ISMLConstants.XML_PREAMBLE + ISMLConstants.nl);
			}
			String firstLine = null;
			if (currentLine.indexOf('<') == -1) {
				firstLine = (currentLine = in.readLine()) + ISMLConstants.nl;
			} else {
				int prevTagOnLine = currentLine.lastIndexOf('>', elementModel.getColumnNumber());
				firstLine = currentLine.substring(prevTagOnLine + 1)+ISMLConstants.nl;
			}
			in.setLineNumber(elementModel.getLineNumber());
			contents.append(firstLine);
			boolean endFound = false;
//			elementModel.setLineNumber(in.getLineNumber());
//			elementModel.setColumnNumber(firstLine.indexOf('<'));
			while (!endFound && (currentLine != null)) {
				currentLine = in.readLine();
				if (currentLine == null) {
					break;
				}
				// look for import element to tack SML schema location onto for validation
				if ((elementModel.getAdditionalSchemaLocation() != null) && (currentLine.matches(".*[^/][\\s]*"+ISMLConstants.IMPORT_ELEMENT+".*"))) {
					int endTagIndex = currentLine.indexOf('>', currentLine.indexOf(ISMLConstants.IMPORT_ELEMENT));
					if (endTagIndex == -1) {
						currentLine += elementModel.getAdditionalSchemaLocation();
					} else {
						if (currentLine.charAt(endTagIndex - 1) == '/') {
							// The import element is an empty element (i.e. the tag ends in "/>", so back up to the '/')
							endTagIndex--;
						}
						currentLine = currentLine.substring(0, endTagIndex) + elementModel.getAdditionalSchemaLocation() + currentLine.substring(endTagIndex);
					}
				}
				endFound = currentLine.matches(".*/[\\s]*"+elementModel.getElementName()+"[\\s]*>.*");
				if (endFound) {
					// this handles the case where there is more after the end tag of the element
					// for example, </xsi:schema> </data> appearing on the same line
					int endTagIndex = currentLine.indexOf(elementModel.getElementName());
					// find the first angle bracket immediately after the element name
					int rightBracket = currentLine.indexOf('>', endTagIndex);
					elementModel.setEndingLine(in.getLineNumber());
					elementModel.setEndingColumn(rightBracket);
					contents.append(currentLine.substring(0, rightBracket+1) + ISMLConstants.nl);
				} else {
					contents.append(currentLine + ISMLConstants.nl);
				}
			} 
			elementModel.setSource(contents.toString());
			in.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	
	public Map<String, ElementModel> getSchemas()
	{
		return schemas;
	}

	public List<ElementModel> getInstances()
	{
		return instances;
	}

	/**
	 * @see org.eclipse.cosmos.rm.validation.internal.databuilders.AbstractDataBuilder#getPhase()
	 */
	public byte getPhase()
	{
		return ISMLConstants.UNKNOWN_PHASE;
	}

	public Object getDataStructure() {
		return null;
	}
}
