/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.validation.internal.databuilders;

import org.eclipse.cosmos.rm.validation.internal.common.IValidationMessage;
import org.eclipse.cosmos.rm.validation.internal.common.IValidationOutput;
import org.eclipse.cosmos.rm.validation.internal.core.IFoundationBuilder;
import org.xml.sax.ContentHandler;


/**
 * This interface is used to represent a data builder which will construct
 * a data structure while an SML document is being parsed. 
 * 
 * @author sleeloy
 * @author Ali Mehregani
 */
public interface IDataBuilder<T> extends ContentHandler
{
	/**
	 * Returns the data structure.  The type of the structure must be specified as
	 * part of the specification of the class that implements this interface.
	 * 
	 * @return The data structure
	 */
	public T getDataStructure();

	
	/**
	 * This method will be called when the parser starts parsing the 
	 * definition part of the document 
	 */
	public void startDefinitions();
	
	
	/**
	 * This method will be called when the parser starts parsing the instances 
	 * part of the document 
	 */
	public void startInstances();
	
	
	/**
	 * This method will be called when the parser ends parsing the definitions 
	 * part of the document 
	 */
	public void endDefinitions();
	
	
	/**
	 * This method will be called when the parser ends parsing the instances part 
	 * of the document 
	 */
	public void endInstances();
	
	
	/**
	 * Invoked when to notify a builder when comments are reached.
	 * 
	 * @param chars The characters
	 * @param start The starting index
	 * @param length The length of the message passed in
	 */
	public void comment(char[] chars, int start, int length);
	
	
	/**
	 * Specifies the phase that this data builder is active during parsing.  The parser parses the sml 
	 * document in two parts: Definitions and Instances phases.  The valid values are as follows:
	 * <pre>
	 * ISMLConstants.DEFINITIONS_PHASE -databuilder is active during the parsing of the definitions elements
	 * ISMLConstants.INSTANCES_PHASE -databuilder is active during the parsing of the instances elements
	 * ISMLConstants.DEFINITIONS_INSTANCES_PHASE - databuilder is active during the parsing of the definitions and instances elements
	 * </pre>
	 *    
	 * @return a phase that this databuilder will be active
	 */
	public byte getPhase(); 
	
	
	/**
	 * Returns true if this structure is valid.  An error message is usually
	 * set if this method returns false.
	 */
	public boolean isStructureValid();
	
	
	/**
	 * Sets the structure validity
	 *  
	 * @param validity The validity of the structure
	 */
	public void setStructureValidity(boolean validity);
	
	
	/**
	 * Sets an error message in case the structure is not valid
	 * 
	 * @param error The error message
	 */
	public void setErrorMessage(IValidationMessage message);

	
	/**
	 * Returns the error message set.  This value can be null if no error
	 * message has been set.
	 * 
	 * @return The error message
	 */
	public IValidationMessage getErrorMessage();
	
	
	/**
	 * Sets the path of the current file being processed
	 * 
	 * @param filePath The path
	 */
	public void setFilePath(String filePath);
	
	
	/**
	 * Retrieves the path of the current file being processed
	 * 
	 * @return The path of the file being processed
	 */
	public String getFilePath();
	
	
	/**
	 * This integer value indicates the events that are supported
	 * by this data builder.  See the event constants defined by
	 * {@link IFoundationBuilder}.
	 * 
	 * @return An integer indicating the events that are supported
	 */
	public int getSupportedEvents();
	
	
	/**
	 * Setter method for the validation outputter.  Data builders
	 * can use the outputter to display warning/error messages
	 * while constructing their data structure.
	 * 
	 * @param messageOutputter The outputter
	 */
	public void setMessageOutputter (IValidationOutput<?,?> messageOutputter);
	
	
	/**
	 * Returns the validation outputter associated with this
	 * data builder.
	 * 
	 * @return The associated outputter of this data builder
	 */
	public IValidationOutput<?,?> getMessageOutputter();
}
