/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.validation.internal.databuilders;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.cosmos.rm.validation.internal.SMLActivator;
import org.eclipse.cosmos.rm.validation.internal.artifacts.ConstraintNode;
import org.eclipse.cosmos.rm.validation.internal.common.ISMLConstants;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.validation.internal.common.AbstractValidationOutput.ValidationMessage;
import org.eclipse.cosmos.rm.validation.internal.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.validation.internal.util.ParserHelper;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

/**
 * Constructs a data structure that stores all schema elements that
 * have an associated target* constraint. The data staructure is a Map
 * that is indexed by the name attribute of an <xs:element.
 * 
 * @author sleeloy
 * @author Ali Mehregani
 */
public class TargetSchemaBuilder extends AbstractDataBuilder<Map<String, ConstraintNode>>
{
	public static final String ID = SMLActivator.PLUGIN_ID + ".TargetSchemaBuilder";
	protected Map<String, ConstraintNode> dataStructure;
    protected boolean uriElementFound = false; 	
	
	public TargetSchemaBuilder(){		
		dataStructure = new HashMap<String, ConstraintNode>();
	}

	public Map<String, ConstraintNode> getDataStructure() {
		return dataStructure;
	}

	public void startSchema(String uri) {
	}
	
	@SuppressWarnings("unchecked")
	public void endSchema(String uri) 
	{
		Map<String, Map<String, String>> substitionMap = (Map<String, Map<String, String>>)SMLValidatorUtil.retrieveDataStructure(SubstitutionBuilder.ID);
		

		//need to make sure that substitionElements are also considered
		Iterator<String> iter = substitionMap.keySet().iterator();
		while (iter.hasNext())
		{
			String namespaceURI = iter.next();
			String elementNameNonNS = ParserHelper.removeNameSpace(namespaceURI);
			ConstraintNode targetSchemaNode = (ConstraintNode)dataStructure.get(elementNameNonNS);
			
			
			if (targetSchemaNode != null)
			{
				Object substitionName = substitionMap.get(namespaceURI);
				ConstraintNode subTargetSchemaNode = (ConstraintNode)dataStructure.get(substitionName);
				targetSchemaNode.setSubstitutionGroup(subTargetSchemaNode);				
			}
			else
			{
				Map<String, String> substitutionMap = substitionMap.get(namespaceURI);
				for (Iterator<String> subtitutions = substitutionMap.keySet().iterator(); subtitutions.hasNext();)
				{
					String elementName = subtitutions.next();
					String substitutedElement = (String)substitutionMap.get(elementName);
					substitutedElement = substitutedElement == null ? null : ParserHelper.removeNameSpace(substitutedElement);
					ConstraintNode subTargetSchemaNode = (ConstraintNode)dataStructure.get(substitutedElement);	
					if (subTargetSchemaNode != null)
					{
						dataStructure.put(elementNameNonNS, subTargetSchemaNode);
					}					
				}
											
			}
		}
		
	}
	

	public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException 
	{
		super.startElement(uri, localName, qName, attributes);
		if (!ISMLConstants.SCHEMA_URI.equals(uri))
		{
			return;
		}
		
		
		if (ISMLConstants.SCHEMA_ELEMENT.equals(localName))
		{
			startSchema(uri);
		}
		else if (ISMLConstants.ELEMENT_ELEMENT.equals(localName))
		{
			String elementName = attributes.getValue(ISMLConstants.NAME_ATTRIBUTE);
			String targetElement = attributes.getValue(ISMLConstants.SML_URI, ISMLConstants.TARGETELEMENT_ATTRIBUTE);
			String targetType = attributes.getValue(ISMLConstants.SML_URI, ISMLConstants.TARGETTYPE_ATTRIBUTE);
			String targetRequired = attributes.getValue(ISMLConstants.SML_URI, ISMLConstants.TARGETREQUIRED_ATTRIBUTE);
			if (targetElement != null || targetType != null || targetRequired != null)
			{ 
				ConstraintNode schemaNode = new ConstraintNode(elementName, getFilePath(), locator == null ? ValidationMessage.NO_LINE_NUMBER : locator.getLineNumber(), locator == null ? ValidationMessage.NO_LINE_NUMBER : locator.getColumnNumber());;
				schemaNode.setTargetElement(SMLValidatorUtil.toQName(super.getPrefixMap(), super.getTargetNamespace(), targetElement));
				schemaNode.setTargetType(SMLValidatorUtil.toQName(super.getPrefixMap(), super.getTargetNamespace(), targetType));
				schemaNode.setTargetRequired(targetRequired);
				dataStructure.put(elementName, schemaNode);
			}		
		}			
		// Display a warning if the target* constraints are specified on any other element
		else if (isTargetConstraintPresent(attributes))
		{
			getMessageOutputter().reportMessage(
					ValidationMessageFactory.createWarningMessage(
							locator.getLineNumber(), SMLValidationMessages.targetBadDeclaration));
		}
	}

	private boolean isTargetConstraintPresent(Attributes attributes)
	{
		return 	attributes.getValue(ISMLConstants.SML_URI, ISMLConstants.TARGETELEMENT_ATTRIBUTE) != null ||
				attributes.getValue(ISMLConstants.SML_URI, ISMLConstants.TARGETTYPE_ATTRIBUTE) != null ||
				attributes.getValue(ISMLConstants.SML_URI, ISMLConstants.TARGETREQUIRED_ATTRIBUTE) != null;
	}

	public void endElement(String uri, String localName, String qName) throws SAXException 
	{
		super.endElement(uri, localName, qName);	
		if (ISMLConstants.SCHEMA_ELEMENT.equals(localName))
		{
			endSchema(uri);
		}
	}

	public byte getPhase() {
		return ISMLConstants.DEFINITIONS_PHASE;
	}	

}
