/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.validation.internal.reference;

import java.util.StringTokenizer;

import org.eclipse.cosmos.rm.validation.internal.common.SMLValidationMessages;
import org.eclipse.osgi.util.NLS;

/**
 * This class compiles an XPointer expression and flags any syntactical errors that
 * are detected.  This is a limited implementation of the XPointer standard as required by
 * the W3C recommnedation.  The SML specification does not require a full implementation
 * of the XPointer standard.
 * 
 * @author Ali Mehregani
 */
public class XPointer
{
	/**
	 * Compiles an XPointer expression and returns an instance
	 * of {@link XPointerExpression} which can be used to evaluate
	 * the expression.
	 * 
	 * @param expression The string representation of the XPointer expression.
	 * The expression is expected to take one of the following formats:
	 * 
	 * <ul>
	 * 	<li>
	 * 	Full XPointer: The expression is assumed to be a full XPointer if it 
	 * 	contains an unescaped bracket; otherwise
	 * 	</li>
	 * 	<li>
	 * 	Child Sequence: The expression is assumed to be a child sequence if it
	 * 	contains an unescaped forward slash ('/'); otherwise
	 * 	</li>
	 * 	<li>
	 * 	Bare Name: The express is assumed to be a bare name
	 * 	</li>
	 * </ul>
	 * 
	 * 
	 * @return An instance of {@link XPointerExpression} which can be used
	 * to evaluate the XPointer expression.
	 * 
	 * @throws XPointerSyntaxException If syntax error is detected
	 */
	public static XPointerExpression compile(String expression) throws XPointerSyntaxException
	{
		XPointerExpression xpointerExpression;
		if (unescapedCharExists(expression, '('))
		{
			xpointerExpression = parseFullXPointer(expression);
		}
		else if (unescapedCharExists(expression, '/'))
		{
			xpointerExpression = parseChildSequence(expression);
		}
		else
		{
			xpointerExpression = new XPointerExpression(XPointerExpression.BARE_NAME_TYPE);
			xpointerExpression.setBareName(expression);
		}
		
		return xpointerExpression;
	}

	private static XPointerExpression parseChildSequence(String expression) throws XPointerSyntaxException
	{
		XPointerExpression xpointerExpression = new XPointerExpression(XPointerExpression.CHILD_SEQUENCE_TYPE);
		StringTokenizer st = new StringTokenizer(expression, "/");
		int count = 0;
		while (st.hasMoreTokens())
		{
			String currentToken = st.nextToken();
			if (isNumber(currentToken))
			{
				xpointerExpression.addChildSequence(Integer.parseInt(currentToken));
			}
			else
			{
				/* The first sequence is permitted to be a name */
				if (count == 0)
				{
					xpointerExpression.setBareName(currentToken);					
				}
				else
				{
					throw new XPointerSyntaxException(NLS.bind(SMLValidationMessages.errorCSNotNumber, new String[]{currentToken, expression}));
				}
			}
			count++;
		}
		return xpointerExpression;
		
	}

	
	/**
	 * Returns true iff 'str' is a valid number
	 * 
	 * @param str The input string
	 * @return true iff 'str' is a number; false otherwise
	 */
	private static boolean isNumber(String str)
	{
		if (str == null)
		{
			return false;
		}
		
		for (int i = 0, charCount = str.length(); i < charCount; i++)
		{
			if (!Character.isDigit(str.charAt(i)))
				return false;
		}
		
		return str.length() > 0;
	}
	
	
	private static XPointerExpression parseFullXPointer(String expression) throws XPointerSyntaxException
	{
		XPointerExpression xpointerExpression = new XPointerExpression(XPointerExpression.FULL_XPOINTER_TYPE);
		StringBuffer scheme = new StringBuffer(), schemeExpression = new StringBuffer();
		IXScheme currentScheme = null;
		
		/* Indicates the current read state (i.e. false = scheme and true = scheme expression) */
		boolean readState = false, unescapedChar = false;
		char currentChar = 0, lastChar = 0;
		for (int i = 0, charCount = expression.length(); i < charCount; i++)
		{
			currentChar = expression.charAt(i);
			unescapedChar = '^' == lastChar;
			if (readState)
			{
				if (!unescapedChar && ')' == currentChar)
				{
					readState = false;
					currentScheme.setExpression(schemeExpression.toString());
					schemeExpression = new StringBuffer();
				}
				else
				{
					if (unescapedChar || '^' != currentChar)
						schemeExpression.append(currentChar);
				}
			}
			else
			{
				if (!unescapedChar && '(' == currentChar)
				{
					readState = true;
					currentScheme = createScheme(scheme.toString().trim());
					xpointerExpression.addScheme(currentScheme);
					scheme = new StringBuffer();
				}
				else
				{
					if (!unescapedChar && currentChar == ')')
						throw new XPointerSyntaxException (NLS.bind(SMLValidationMessages.errorUnbalancedBracket, expression));
					
					if (unescapedChar || '^' != currentChar)
						scheme.append(currentChar);
				}
			}
			lastChar = currentChar;
		}
		
		if (scheme.toString().length() > 0)
		{
			throw new XPointerSyntaxException (NLS.bind(SMLValidationMessages.errorMissingExpression, scheme.toString()));
		}
		
		if (schemeExpression.toString().length() > 0)
		{
			throw new XPointerSyntaxException (NLS.bind(SMLValidationMessages.errorUnclosedExpression, schemeExpression.toString()));
		}
		
		return xpointerExpression;		
	}

	private static IXScheme createScheme(String scheme)
	{
		if (XPointerScheme.SCHEME.equals(scheme))
		{
			return new XPointerScheme();
		}
		else if (XmlnsScheme.SCHEME.equals(scheme))
		{
			return new XmlnsScheme();
		}
		else if (SMLXPath1Scheme.SCHEME.equals(scheme))
		{
			return new SMLXPath1Scheme();
		}
		
		return NullScheme.instance;
	}

	private static boolean unescapedCharExists(String expression, char character)
	{
		char previousCharacter = 0;
		for (int i = 0, charCount = expression.length(); i < charCount; i++)
		{
			if (character == expression.charAt(i) && previousCharacter != '^')
				return true;
			
			previousCharacter = expression.charAt(i);
		}
		return false;
	}
	
	
	/**
	 * Represents a null scheme.  Unrecognized schemes are declared as instances
	 * of this class.
	 * 
	 * @author Ali Mehregani
	 */
	public static class NullScheme extends AbstractScheme
	{
		private static final IXScheme instance = new NullScheme();
		
		public Object evaluate(Object context)
		{
			return context;
		}

		public String getType()
		{			
			return null;
		}
		
	}
}
