/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.validation.internal.smlvalidators;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.xpath.XPathFunctionException;

import org.eclipse.cosmos.rm.validation.internal.artifacts.DocumentNode;
import org.eclipse.cosmos.rm.validation.internal.artifacts.ElementTypeMap;
import org.eclipse.cosmos.rm.validation.internal.artifacts.TypeNode;
import org.eclipse.cosmos.rm.validation.internal.common.IValidationOutput;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.validation.internal.common.AbstractValidationOutput.ValidationMessage;
import org.eclipse.cosmos.rm.validation.internal.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.validation.internal.core.AbstractSMLValidator;
import org.eclipse.cosmos.rm.validation.internal.databuilders.AcyclicDataTypesList;
import org.eclipse.cosmos.rm.validation.internal.databuilders.DataBuilderRegistry;
import org.eclipse.cosmos.rm.validation.internal.databuilders.DocumentDOMBuilder;
import org.eclipse.cosmos.rm.validation.internal.databuilders.ElementTypeMapDataBuilder;
import org.eclipse.cosmos.rm.validation.internal.databuilders.IDataBuilder;
import org.eclipse.cosmos.rm.validation.internal.databuilders.ReferenceGraphDataBuilder;
import org.eclipse.cosmos.rm.validation.internal.databuilders.TypeInheritanceDataBuilderImpl;
import org.eclipse.cosmos.rm.validation.internal.reference.DerefXPathFunction;
import org.eclipse.osgi.util.NLS;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;


/**
 * This class is used to validate the acyclic sml extension 
 * 
 * @author Ali Mehregani
 */
public class AcyclicValidator extends AbstractSMLValidator
{	

	/**
	 * A list of elements representing the graph nodes 
	 */
	private List<ResolvedReference> graphNodes;
	
	/**
	 * The inheritance hierarchy 
	 */
	private Map<String, Map<String, TypeNode>> inheritanceHierarchy;

	/**
	 * The element type map
	 */
	private ElementTypeMap typeMap;
	
	/**
	 * Constructor
	 */
	public AcyclicValidator()
	{
		graphNodes = new ArrayList<ResolvedReference>();
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.validation.internal.launcher.IValidator#initialize(java.util.Map)
	 */
	public void initialize(Map<String, Object> validationAttribute)
	{
		super.initialize(validationAttribute);
		
		/* Register the right data builders with the registry */
		DataBuilderRegistry databuilderRegistry = DataBuilderRegistry.instance();
		databuilderRegistry.registerDataStructureBuilder(AcyclicDataTypesList.ID, new AcyclicDataTypesList());
		databuilderRegistry.registerDataStructureBuilder(ReferenceGraphDataBuilder.ID, new ReferenceGraphDataBuilder());
	}
	

	/**
	 * @see org.eclipse.cosmos.rm.validation.internal.launcher.IValidator#validate()
	 */
	@SuppressWarnings("unchecked")
	public boolean validate()
	{
		setTaskName(SMLValidationMessages.validationAcyclic);
	
		// Start the validation process for the acyclic extension		
		Iterator<TypeNode> acyclicTypes = (Iterator<TypeNode>)SMLValidatorUtil.retrieveDataStructure(AcyclicDataTypesList.ID);
		Map<String, DocumentNode> referenceGraph = (Map<String, DocumentNode>)SMLValidatorUtil.retrieveDataStructure(ReferenceGraphDataBuilder.ID);				
		IDataBuilder inheritanceDataBuilder = DataBuilderRegistry.instance().getDataStructureBuilder(TypeInheritanceDataBuilderImpl.ID);
		Map<String, Object> domDocuments = (Map<String, Object>)SMLValidatorUtil.retrieveDataStructure(DocumentDOMBuilder.ID);
		typeMap = (ElementTypeMap)SMLValidatorUtil.retrieveDataStructure(ElementTypeMapDataBuilder.ID);
		
		IValidationOutput<String, Object> output = getValidationOutput();	
		
		// We can't continue if the reference graph is not available 
		if (referenceGraph == null || acyclicTypes == null)
		{
			// Output error 
			output.reportMessage(ValidationMessageFactory.createErrorMessage(ValidationMessage.NO_LINE_NUMBER, SMLValidationMessages.acyclicUnknown));
			output.reportMessage(ValidationMessageFactory.createErrorMessage(ValidationMessage.NO_LINE_NUMBER,
								referenceGraph == null ? SMLValidationMessages.acyclicNoReferenceGraph : 
								acyclicTypes == null ? SMLValidationMessages.acyclicNoAcyclicTypeList : 
								SMLValidationMessages.acyclicNoInheritanceMap));
			if (shouldAbortOnError()) {
				return false;
			}
		}
		
		// Check the validity of the inheritance map 
		if (!inheritanceDataBuilder.isStructureValid())
		{
			output.reportMessage(inheritanceDataBuilder.getErrorMessage());
			if (shouldAbortOnError()) {
				return false;
			}
		}
		
		inheritanceHierarchy = (Map<String, Map<String, TypeNode>>) inheritanceDataBuilder.getDataStructure();
		
		String[] documentAlias = new String[referenceGraph.size()];
		int counter = 0;
		for (Iterator<String> iterator = referenceGraph.keySet().iterator(); iterator.hasNext();)
		{			
			documentAlias[counter++] = iterator.next();			
		}
		
		// Iterate through the acyclic element types.  These are all complex 
		// types CT that have sml:acyclic = "true"/"1"
		for (Iterator<TypeNode> iter = acyclicTypes; iter.hasNext();)
		{
			TypeNode elementType = iter.next();
			
			// Because there is no guarantee that the graph is connected, we'll need to iterate
			// through each document alias. 
			for (int i = 0; i < documentAlias.length; i++)
			{
				graphNodes.clear();
				Node documentNode = (Node)domDocuments.get(documentAlias[i]);
				
				// Traverse references to determine if a cycle can be detected
				if (isCyclePresent(documentNode, elementType))
				{					
					output.reportMessage(ValidationMessageFactory.createErrorMessage(ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.acyclicCycleFound, elementType.getType())));
					StringBuffer cycle = new StringBuffer();
					for (int j = 0, nodesCount = graphNodes.size(); j < nodesCount; j++)
					{
						cycle.append(j > 0 ? ", " : "");						
						cycle.append(graphNodes.get(j).reference);						
					}					
					if (cycle.length() > 0)
					{
						output.reportMessage(ValidationMessageFactory.createErrorMessage(ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.acyclicDetectedCycle, cycle.toString())));
					}
					
					return false;
				}
			}			
		}
		
		return true;
	}

	
	/**
	 * Given a node N and an acyclic complex type CT, determine if there
	 * is a cycle present by traversing through the references of node N.
	 * 
	 * @param node The node N
	 * @param elementType The acyclic complex type CT
	 * @return true if a cycle is detected; false otherwise
	 */
	private boolean isCyclePresent(Node node, TypeNode elementType)
	{
		ResolvedReference[] resolvedReferences = resolveReferences (node, elementType); 
		for (int i = 0; i < resolvedReferences.length; i++)
		{
			if (graphNodes.contains(resolvedReferences[i]))
			{
				graphNodes.add(resolvedReferences[i]);			
				return true;
			}
			
			graphNodes.add(resolvedReferences[i]);			
			if (isCyclePresent(resolvedReferences[i].resolvedReference, elementType))
			{
				return true;
			}
			graphNodes.remove(resolvedReferences[i]);
		}
		return false;
	}

	
	/**
	 * Given node N, determine all nodes referenced by instances of
	 * CT.
	 * 
	 * @param node The node N
	 * @param elementType The complex type CT
	 * @return All nodes that are resolved references of instances of CT
	 */
	private ResolvedReference[] resolveReferences(Node node, TypeNode elementType)
	{
		if (node == null)
		{
			return new ResolvedReference[0];
		}
		
		// Is N of type CT?
		if (checkInstanceType(node, elementType))
		{
			ResolvedReference resolvedReference = resolveReference(node);
			return resolvedReference == null ? new ResolvedReference[0] : new ResolvedReference[]{resolvedReference};
		}
		// Otherwise check if N contains CT
		else
		{
			NodeList children = node.getChildNodes();
			List<ResolvedReference> references = new ArrayList<ResolvedReference>();
			findReferences(references, children, elementType);			
			return references.toArray(new ResolvedReference[references.size()]);
		}
	}

	
	private void findReferences(List<ResolvedReference> references, NodeList nodeList, TypeNode elementType)
	{
		for (int i = 0, childCount = nodeList.getLength(); i < childCount; i++)
		{
			Node child = nodeList.item(i);
			if (checkInstanceType(child, elementType))
			{
				ResolvedReference resolvedReference = resolveReference(child);
				if (resolvedReference != null)
				{
					references.add(resolvedReference);
				}					
			}
			
			if (child.hasChildNodes())
			{
				findReferences(references, child.getChildNodes(), elementType);
			}
		}
	}

	/**
	 * Returns true if node N is an instance of the complex type CT
	 * 
	 * @param node Node N to be checked
	 * @param elementType The complex type CT
	 * @return true if N is an instance of CT
	 */
	@SuppressWarnings("unchecked")
	private boolean checkInstanceType(Node node, TypeNode elementType)
	{
		String type = typeMap.getType(node.getNamespaceURI(), node.getLocalName());
		String namespace = node.getNamespaceURI() == null ? "" : node.getNamespaceURI();
		Map<String, TypeNode> nestedMap = (Map<String, TypeNode>)SMLValidatorUtil.retrieveNestedMap(inheritanceHierarchy, namespace, false);
		
		if (!namespace.equals(elementType.getUri()))
		{
			return false;
		}
		
		// Walk through the type hierarchy of N		
		while (type != null)
		{			
			if (type.equals(elementType.getType()))
			{
				return true;
			}
			
			TypeNode typeNode = nestedMap == null ? null : nestedMap.get(type);
			type = typeNode == null ? null : typeNode.getType();
		}
		
		return false;
	}
	
	
	/**
	 * Given node N, resolve the SML reference R contained by N.
	 * 
	 * @param node Node N
	 * @return Node T, where T is the resolved SML reference R contained by N
	 */
	private ResolvedReference resolveReference(Node node)
	{
		// Extract reference
		String reference = SMLValidatorUtil.extractReference(node);
		if (reference == null)
		{
			return null;
		}
		List<String> arguments = new ArrayList<String>();
		arguments.add(reference);
		
		
		try
		{
			// The output is expected to be one element
			Object output = DerefXPathFunction.instance().evaluate(arguments);
			NodeList result;
			if (output instanceof NodeList && (result = (NodeList)output).getLength() == 1)
			{
				return new ResolvedReference(reference, result.item(0));
			}
		} 
		catch (XPathFunctionException e)
		{
			// Do nothing
		}		
		
		return null;
	}

	
	private static class ResolvedReference
	{
		private String reference;
		private Node resolvedReference;
		
		public ResolvedReference(String reference, Node resolvedReference)
		{
			this.reference = reference;
			this.resolvedReference = resolvedReference;
		}		
		
		
		@Override
		public boolean equals(Object o)
		{
			if (!(o instanceof ResolvedReference))
			{
				return false;
			}
			
			ResolvedReference other = (ResolvedReference)o;
			return resolvedReference == null ? other.resolvedReference == null : resolvedReference.equals(other.resolvedReference);
		}
	}
}
