/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.validation.internal.smlvalidators;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;

import org.eclipse.cosmos.rm.validation.internal.artifacts.IdentityConstraintStructure;
import org.eclipse.cosmos.rm.validation.internal.common.IValidationConstants;
import org.eclipse.cosmos.rm.validation.internal.common.IValidationOutput;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.validation.internal.common.AbstractValidationOutput.ValidationMessage;
import org.eclipse.cosmos.rm.validation.internal.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.validation.internal.core.AbstractSMLValidator;
import org.eclipse.cosmos.rm.validation.internal.databuilders.DataBuilderRegistry;
import org.eclipse.cosmos.rm.validation.internal.databuilders.DocumentDOMBuilder;
import org.eclipse.cosmos.rm.validation.internal.databuilders.ElementDeclarationBuilder;
import org.eclipse.cosmos.rm.validation.internal.databuilders.IdentityConstraintDataBuilder;
import org.eclipse.cosmos.rm.validation.internal.databuilders.ReferenceExtractor;
import org.eclipse.cosmos.rm.validation.internal.databuilders.IdentityConstraintDataBuilder.IdentityConstraint;
import org.eclipse.osgi.util.NLS;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * This class is used to validate SML identity constraints.  i.e.:
 * <ul>
 * 	<li> key </li>
 * 	<li> keyref </li>
 * 	<li> unique </li>
 * </ul>
 * 
 * @author Ali Mehregani
 */
public class IdentityConstraintValidator extends AbstractSMLValidator
{
	/**
	 * Constructor
	 */
	public IdentityConstraintValidator()
	{
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.validation.internal.launcher.IValidator#initialize(java.util.Map)
	 */
	public void initialize(Map<String, Object> validationAttribute)
	{
		super.initialize(validationAttribute);
		
		/* Register the right data builders with the registry */
		DataBuilderRegistry databuilderRegistry = DataBuilderRegistry.instance();		
		databuilderRegistry.registerDataStructureBuilder(DocumentDOMBuilder.ID, new DocumentDOMBuilder());
		databuilderRegistry.registerDataStructureBuilder(IdentityConstraintDataBuilder.ID, new IdentityConstraintDataBuilder());
		databuilderRegistry.registerDataStructureBuilder(ReferenceExtractor.ID, new ReferenceExtractor());
		databuilderRegistry.registerDataStructureBuilder(ElementDeclarationBuilder.ID, new ElementDeclarationBuilder());
	}
	

	/**
	 * @see org.eclipse.cosmos.rm.validation.internal.launcher.IValidator#validate()
	 */
	@SuppressWarnings("unchecked")
	public boolean validate()
	{
		boolean isValidationSuccess = true;
		setTaskName(SMLValidationMessages.validationIdentity);		
		
		/* Retrieve the data structures required as part of the validation */		
		DocumentDOMBuilder builder = (DocumentDOMBuilder)DataBuilderRegistry.instance().getDataStructureBuilder(DocumentDOMBuilder.ID);
		Map<String, Object> domDocuments = builder == null ? null : (Map<String, Object>) builder.getDataStructure();
		IdentityConstraintStructure identityStruct = (IdentityConstraintStructure)SMLValidatorUtil.retrieveDataStructure(IdentityConstraintDataBuilder.ID);
		SMLValidatorUtil.retrieveDataStructure(ReferenceExtractor.ID);
		
		IValidationOutput output = getValidationOutput();		
		
		if (domDocuments == null || identityStruct == null)
		{
			output.reportMessage(ValidationMessageFactory.createErrorMessage(ValidationMessage.NO_LINE_NUMBER, SMLValidationMessages.identityMissingStructures));
			isValidationSuccess = false;
		}
		
		Map<String, List<int[]>> constrainedInstances = identityStruct.getConstrainedInstances();
		
		/* For each constrained instance contained in a document with an alias */
		for (Iterator<String> documentAliases = constrainedInstances.keySet().iterator(); documentAliases.hasNext();)
		{
			String documentAlias = documentAliases.next();
			Node node = (Node)domDocuments.get(documentAlias);
			if (node == null)
				continue;
			
			List<int[]> indices = constrainedInstances.get(documentAlias);
			for (int i = 0, indexCount = indices.size(); i < indexCount; i++)
			{
				String error = validateIdentityConstraints (identityStruct, node, documentAlias, (int[])indices.get(i));
				if (error != null)
				{										
					output.reportMessage(SMLValidatorUtil.createValidationMessage(builder, node, error));
					isValidationSuccess = false;
					if (shouldAbortOnError()) 
					{
						return false;
					}
				}
			}
		}
		
		/* For each orphan document that does not contain an alias */
		Map<String, List<int[]>> orphanedConstrainedInstances = identityStruct.getOrphanedConstrainedInstances();
		List<Node> orphanedDOMDocuments = (List<Node>)domDocuments.get(DocumentDOMBuilder.NO_ALIAS);
		int orphanSize = orphanedDOMDocuments == null ? 0 : orphanedDOMDocuments.size();
		for (Iterator<String> orphanIndices = orphanedConstrainedInstances.keySet().iterator(); orphanIndices.hasNext();)
		{
			int orphanIndex = Integer.parseInt(orphanIndices.next());
			if (orphanIndex < 0 || orphanIndex >= orphanSize)
				continue;
			
			Node node = orphanedDOMDocuments.get(orphanIndex);
			if (node == null)
				continue;
			
			List<int[]> listOfIndices = orphanedConstrainedInstances.get(String.valueOf(orphanIndex));
			for (int i = 0, indicesCount = listOfIndices.size(); i < indicesCount; i++)
			{
				int[] indices = (int[])listOfIndices.get(i);		
				String error = validateIdentityConstraints (identityStruct, node, null, indices);
				if (error != null)
				{
					output.reportMessage(SMLValidatorUtil.createValidationMessage(builder, node,error));
					isValidationSuccess = false;
					if (shouldAbortOnError()) {
						return false;
					}
				}
			}			
		}
		
		return isValidationSuccess;
	}
	

	private String validateIdentityConstraints (IdentityConstraintStructure identityStruct, Node context, String documentAlias, int[] indices)
	{
		context = findNode(context, indices);			
		IdentityConstraint[] constraints = context == null ? null : identityStruct.retrieveConstraint(context.getNamespaceURI(), context.getLocalName());
		if (context == null || constraints == null)
			return null;
								
		Map<String, List<String>> documentConstraints = new Hashtable<String, List<String>>();
		IdentityConstraint currentConstraint = null;
		String error = null;
		try
		{
			/* For every constraint */
			for (int i = 0; i < constraints.length; i++)
			{												
				currentConstraint = constraints[i];
				XPathExpression selector = null;
				List<String> fields = null;
				List<String> fieldValues = new ArrayList<String>();
				
				synchronized(SMLValidatorUtil.xpath)
				{
					if (constraints[i].getNamespaceContext() != null)
					{
						SMLValidatorUtil.xpath.setNamespaceContext(constraints[i].getNamespaceContext());
					}
					
					selector = SMLValidatorUtil.xpath.compile(currentConstraint.getSelector());													
					fields = currentConstraint.getFields();
					NodeList selectorList = (NodeList)selector.evaluate(context, XPathConstants.NODESET);
					if (selectorList == null || selectorList.getLength() <= 0)
						continue;
										
					fieldValues = new ArrayList<String>();
					/* For every node in the scope of the selector xpath expression */
					for (int j = 0, nodeCount = selectorList.getLength(); j < nodeCount; j++)
					{
						String fieldsPerNode = "";
						
						/* For every field of the constraint */
						for (int k = 0, fieldCount = fields.size(); k < fieldCount; k++)
						{
							Node currentNode = selectorList.item(j);
							String currentField = extractField(currentNode, (String)fields.get(k));
							fieldsPerNode += currentField == null ? IValidationConstants.EMPTY_STRING : (fieldsPerNode.length() > 0 ? ", " + currentField : currentField);
							
							/* The field must exist if the constraint is a key */
							if (currentField == null && currentConstraint.getType() == IdentityConstraint.KEY_TYPE)
							{
								return NLS.bind(SMLValidationMessages.identityMissingField, new String[]{(String)fields.get(k), currentConstraint.getName(), documentAlias == null ? SMLValidationMessages.commonNoAlias : documentAlias});
							}
						}
												
						fieldValues.add(fieldsPerNode);
					}										
				}
				
				documentConstraints.put(currentConstraint.getName(), fieldValues);
				error = validateConstraint(documentConstraints, constraints[i], fieldValues, documentAlias);
				if (error != null)
					return error;
			}
		}
		catch (XPathExpressionException e)
		{		
			return NLS.bind(
							SMLValidationMessages.identityXPathError, 
							new String[]{currentConstraint.getSelector(), currentConstraint.getName()}) + 
							IValidationConstants.LINE_SEPARATOR + (e.getMessage() == null ? IValidationConstants.EMPTY_STRING : e.getMessage()
							); 
		}
		
		return null;
	}
	
	private String validateConstraint(Map<String, List<String>> constraintMap, IdentityConstraint constraint, List<String> fields, String documentAlias)
	{ 
		switch (constraint.getType())
		{
			case IdentityConstraint.UNIQUE_TYPE:
			case IdentityConstraint.KEY_TYPE:				
				
				Hashtable<String, String> indexedHashtable = new Hashtable<String, String>();
				for (int i = 0, listSize = fields.size(); i < listSize; i++)
				{
					if (indexedHashtable.get(fields.get(i)) != null)
						return NLS.bind(SMLValidationMessages.identityDuplicateKey, new String[]{constraint.getName(), documentAlias == null ? SMLValidationMessages.commonNoAlias : documentAlias, fields.get(i).toString()});
					indexedHashtable.put(fields.get(i), IValidationConstants.EMPTY_STRING);
				}			
				break;
				
			case IdentityConstraint.KEY_REF_TYPE:
				
				String reference = constraint.getReference();
				List<String> referencedConstraint = null;
				if (reference == null)
				{
					return NLS.bind(SMLValidationMessages.identityMissingReference, constraint.getName());
				}
				else if ((referencedConstraint = constraintMap.get(reference)) == null)
				{
					return NLS.bind(SMLValidationMessages.identityMissingConstraint, constraint.getName());
				}
				
				/* For every field of the key ref constraint */
				for (int i = 0, fieldSize = fields.size(); i < fieldSize; i++)
				{
					String currentField = fields.get(i);
					boolean foundField = false;
					
					/* For every field of the referenced constraint */ 
					for (int j = 0, referenceList = referencedConstraint.size(); j < referenceList; j++)
					{
						if (currentField.equals(referencedConstraint.get(j)))
						{
							foundField = true;
							break;
						}
					}
					
					if (!foundField)
					{
						return NLS.bind(SMLValidationMessages.identityMissingKey, new String[]{currentField, constraint.getName()});
					}
				}
				break;
			default: 
				break;
		}

		return null;
	}

	private String extractField(Node context, String field)
	{
		try
		{
			XPathExpression fieldXPath = SMLValidatorUtil.xpath.compile(field);
			NodeList nodeList = (NodeList)fieldXPath.evaluate(context, XPathConstants.NODESET);
			Node fieldNode = nodeList.getLength() == 1 ? nodeList.item(0) : null;
			if (fieldNode == null)
				return null;
			
			if (field.charAt(0) == '@')
				return fieldNode.getNodeValue();
			NodeList childNodes = fieldNode.getChildNodes();
			StringBuffer fieldText = new StringBuffer();
			for (int i = 0, chidNodeCount = childNodes.getLength(); i < chidNodeCount; i++)
			{
				Node currentNode = childNodes.item(0);
				if (Node.TEXT_NODE != currentNode.getNodeType())
					return field.toString();
				fieldText.append(currentNode.getNodeValue() == null ? IValidationConstants.EMPTY_STRING : currentNode.getNodeValue());
			}
			return fieldText.toString();
		}
		catch (XPathExpressionException e)
		{
			return null;
		}
	}

	private Node findNode(Node node, int[] indices)
	{
		Node currentNode = node;
		NodeList nodeList = null;
		for (int i = 1; i < indices.length; i++)
		{
			nodeList = node.getChildNodes();
			if (indices[i] < 0 || indices[i] >= nodeList.getLength())
				return null;
			currentNode = nodeList.item(indices[i]);
		}
		
		return currentNode;
	}
	
}
