/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.validation.internal.smlvalidators;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.List;
import java.util.Map;

import org.eclipse.cosmos.rm.validation.internal.artifacts.ReferenceDescriptor;
import org.eclipse.cosmos.rm.validation.internal.common.IValidationMessage;
import org.eclipse.cosmos.rm.validation.internal.common.IValidationOutput;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.validation.internal.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.validation.internal.core.AbstractSMLValidator;
import org.eclipse.cosmos.rm.validation.internal.databuilders.DataBuilderRegistry;
import org.eclipse.cosmos.rm.validation.internal.databuilders.ReferenceExtractor;
import org.eclipse.cosmos.rm.validation.internal.reference.ReferenceContext;
import org.eclipse.cosmos.rm.validation.internal.reference.URIReference;
import org.eclipse.cosmos.rm.validation.internal.reference.XPointer;
import org.eclipse.cosmos.rm.validation.internal.reference.XPointerExpression;
import org.eclipse.osgi.util.NLS;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * A Reference validator is used to display warning messages for
 * any dangling references that are found. 
 * 
 * @author Ali Mehregani
 */
public class ReferenceValidator extends AbstractSMLValidator
{
	/**
	 * @see org.eclipse.cosmos.rm.validation.internal.launcher.IValidator#initialize(java.util.Map)
	 */
	public void initialize(Map<String, Object> validationAttribute)
	{
		super.initialize(validationAttribute);
		
		// Register the required data builders
		DataBuilderRegistry databuilderRegistry = DataBuilderRegistry.instance();
		databuilderRegistry.registerDataStructureBuilder(ReferenceExtractor.ID, new ReferenceExtractor());
	}
	
	public boolean validate()
	{
		setTaskName(SMLValidationMessages.validationReferences);
		
		// For each reference
		ReferenceExtractor referenceExtractor = (ReferenceExtractor)DataBuilderRegistry.instance().getDataStructureBuilder(ReferenceExtractor.ID);
		List<ReferenceContext> referenceContextList = referenceExtractor.getContextReferences();
		
		NodeList lastNodeList = null;
		ReferenceDescriptor lastReference = null;
		
		// For each reference context
		for (int i = 0, refContextCount = referenceContextList.size(); i < refContextCount; i++)
		{
			ReferenceContext referenceContext = referenceContextList.get(i);
			
			// Don't attempt to resolve the reference if this is a nilref
			if (referenceContext.isNilref())
			{
				continue;
			}
			
			List<ReferenceDescriptor> referenceDescriptors = referenceContext.getReferenceDescriptors();
			for (int j = 0, referenceDescCount = referenceDescriptors.size(); j < referenceDescCount; j++)
			{
				ReferenceDescriptor currentReference = null;
				try
				{
					currentReference = referenceDescriptors.get(j);			
					URIReference uriReference = new URIReference(currentReference.getReference());
					Node context = uriReference.retrieveDocumentDOM();
					NodeList nodeList = null;
					if (context == null)
					{
						showWarning(SMLValidationMessages.warningUnresolvedReference, currentReference);						
					}
					else
					{
						// Evaluate the XPointer expression of the reference (if one exists)
						String fragment = uriReference.getFragment();
						
						if (fragment != null && fragment.length() > 0)
						{
							XPointerExpression xpointerExpression = XPointer.compile(uriReference.getFragment());					
							nodeList = (NodeList)xpointerExpression.evaluate(referenceContext.getNamespaceContext(), context);
							if (nodeList == null || nodeList.getLength() <= 0)
							{
								showWarning(SMLValidationMessages.warningUnresolvedReference, currentReference);
							}
						}
					}
					
					
					
					
					if (j == 0)
					{
						lastNodeList = nodeList;
						lastReference = currentReference;
					}
					else if ((lastNodeList == null && nodeList != null) || !isSame(lastNodeList, nodeList))
					{
						showError(lastReference, currentReference);
						if (shouldAbortOnError()) 
						{
							return false;
						}
					}
				} 
				catch (Exception e)
				{
					showError (SMLValidationMessages.errorReferenceResolving, currentReference, e);
				} 			
			}
		}

		return true;
	}

	private boolean isSame(NodeList nodeList1, NodeList nodeList2)
	{
		if (nodeList1 == null)
		{
			return nodeList2 == null;
		}
		else if (nodeList2 == null)
		{
			return false;
		}
		
		
		int nodeCount = nodeList1.getLength();
		if (nodeCount != nodeList2.getLength())
		{
			return false;
		}
		
		for (int i = 0; i < nodeCount; i++)
		{
			if (!nodeList1.item(i).equals(nodeList2.item(i)))
			{
				return false;
			}			
		}
		return true;
	}

	private void showError(ReferenceDescriptor lastReference, ReferenceDescriptor currentReference)
	{
		showError (NLS.bind(SMLValidationMessages.errorInconsistentReference, 
				 new String[]{lastReference.getReference(), currentReference.getReference()}), lastReference);
	}
	
	private void showError(String message, ReferenceDescriptor uriReference, Exception e)
	{
		StringWriter sw = new StringWriter();
		e.printStackTrace(new PrintWriter(sw));			
		showMessage(ValidationMessageFactory.createErrorMessage(
				uriReference.getFilePath(), 
				uriReference.getLineNumber(), 
				NLS.bind(message, uriReference.getReference()) + " " + sw.getBuffer().toString()));
		
	}
	
	private void showError (String message, ReferenceDescriptor reference)
	{
		showMessage(ValidationMessageFactory.createErrorMessage(
				reference.getFilePath(), 
				reference.getLineNumber(), 
				message));
	}

	private void showWarning(String message, ReferenceDescriptor uriReference)
	{
		showMessage(ValidationMessageFactory.createWarningMessage(
				uriReference.getFilePath(), 
				uriReference.getLineNumber(), 
				NLS.bind(message, uriReference.getReference())));	
	}
	
	private void showMessage(IValidationMessage validationMessage)
	{
		IValidationOutput<String, Object> output = getValidationOutput();
		output.reportMessage(validationMessage);
	}
}
